function OmegaPi = NormalConditionalPrior(OverallTightness,HarmonicLagDecay,p,k)
% NormalConditionalPrior: Constructs the OmegaPi part of the prior covariance matrix
%                         of the parameters on lagged endogenous variables. The full
%                         prior covariance matrix is given by: kron(OmegaPi,Omega)
%                         where Omega is a value for the covariance matrix of the VAR
%                         residuals.
%
% USAGE:
%
%       OmegaPi = NormalConditionalPrior(OverallTightness,HarmonicLagDecay,p,k)
%
% REQUIRED INPUT:  OverallTightness (scalar), with the overall tightness hyperparameter.
%
%                  HarmonicLagDecay (scalar), with the harmonic lag decay hyperparameter.
%
%                  p (integer), with the number of endogenous variables.
%
%                  k (integer), the lag order of the VAR
%
% REQUIRED OUTPUT: OmegaPi (matrix), with the resulting prior variances.
%
% NOTE: The parameters on lags are assumed to be organized as follows:
%
%               PiMatrix = [Pi(1) ... Pi(k)]
%
% The prior covariance for PiMatrix is:
%
%           SigmaPi = kron(OmegaPi,Omega)
%
% where OmegaPi is block diagonal with diagonal blocks OmegaPi(l)
%
%           OmegaPi(l) = (OverallTightness/(l^HarmonicLagDecay)*eye(p)
%
% Here we provide the OmegaPi matrix.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: January 12, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

OmegaPi = zeros(p*k,p*k);
for i=1:k;
   OmegaPi(1+(p*(i-1)):p*i,1+(p*(i-1)):p*i) = (OverallTightness/(i^HarmonicLagDecay))*eye(p);
end;

%
% end of NormalConditionalPrior.m
%
