function [PredEventData,PredictionEvent,ShockCov,ParameterCov] = LoadBVARPredictionUncertainty(OutputDirectory,NameOfModel,PriorType,OmegaPriorType,h,NumDraws,NumPaths,AnnualizeData)
% LoadBVARPredictionUncertainty: Loads the results for the prediction uncertainty decomposition for a Bayesian VAR model
%                                from disk. It also loads any data it finds about prediction events.
%
% USAGE:
%
%       [PredEventData,PredictionEvent,ShockCov,ParameterCov] = LoadBVARPredictionUncertainty(OutputDirectory,NameOfModel,PriorType,...
%                                                                  OmegaPriorType,h,NumDraws,NumPaths,AnnualizeData)
%
% REQUIRED INPUT:  OutputDirectory (string vector) with the full path to the base output directory.
%
%                  NameOfModel (string vector) with the name of the model.
%
%                  PriorType (integer) determining the type of prior used for the parameters on lagged endogenous
%                            variables of the Bayesian VAR. The value "1" is the Minnesota-style prior, "2" is the
%                            normal conditional on the covariance matrix prior, and "3" the diffuse prior.
%
%                  OmegaPriorType (integer) determining the type of prior used for the covariance matrix parameters.
%                            The value "1" is a diffuse prior, and "2" and inverse Wishart prior.
%
%                  h (integer) with the length of the prediction horizon.
%
%                  NumDraws (integer) with the total number of parameter draws.
%
%                  NumPaths (integer) with tht total number of prediction paths per parameter draw.
%
%                  AnnualizeData (boolean) that take the value 1 if annualized results should be checked and 0 otherwise.
%
% REQUIRED OUTPUT: PredEventData (n x 3) matrix where the first column holds the number of times
%                            the PredPaths data satisfies the event, the second hold the number
%                            of times the PredPaths data is greater than the event, while the
%                            third column holds the number of times the event has been tested.
%
%                  PredictionEvent (p x 3) matrix with the prediction event information. The first
%                            column holds the upper bound, the second the lower bound for the event,
%                            while the third holds the event length in periods.
%
%                  ShockCov (p x p x h) matrix with the estimated shock uncertainty.
%
%                  ParameterCov (p x p x h) matrix with the estimated parameter uncertainty.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: May 10, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 11-5-2007:  Updated the documentation.
%
% * 6-6-2007:   The PredEventData matrix now has three columns.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
PredEventData = [];
PredictionEvent = [];
ShockCov = [];
ParameterCov = [];
if AnnualizeData==0;
   file = [OutputDirectory '\predictions\BVAR-Prediction-Original-' NameOfModel '-' num2str(PriorType,'%0.0f') '.' num2str(OmegaPriorType,'%0.0f') '-' num2str(h,'%0.0f') '-' num2str(NumDraws,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPaths,'%0.0f') '.mat'];
else;
   file = [OutputDirectory '\predictions\BVAR-Prediction-Annual-' NameOfModel '-' num2str(PriorType,'%0.0f') '.' num2str(OmegaPriorType,'%0.0f') '-' num2str(h,'%0.0f') '-' num2str(NumDraws,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPaths,'%0.0f') '.mat'];
end;
if FileExist(file)==1;
   PredData = load(file);
   if isfield(PredData,'ShockCov')==1;
      ShockCov = PredData.ShockCov;
      ParameterCov = PredData.ParameterCov;
   end;
   if isfield(PredData,'PredEventData')==1;
      PredEventData = PredData.PredEventData;
   end;
   if isfield(PredData,'PredictionEvent')==1;
      PredictionEvent = PredData.PredictionEvent;
   end;
end;

%
% end of LoadBVARPredictionUncertainty.m
%
