function [PsiPostSample,PiPostSample,OmegaPostSample,LogPsiDensity,LogPiDensity,NumBurnin] = LoadBVARPosteriorDraws(DSGEModel,CurrINI)
% LoadBVARPosteriorDraws: Loads the data from the posterior draws for the Bayesian VAR mode.
%
% USAGE:
%
%       [PsiPostSample,PiPostSample,OmegaPostSample,LogPsiDensity,LogPiDensity,NumBurnin] = LoadBVARPosteriorDraws(DSGEModel,CurrINI)
%
% REQUIRED INPUT:  DSGEModel (structure) whose fields contain all required information
%                            about the model, such as input files, variable and sample
%                            data. Actual observations are not stored in this file.
%
%                  CurrINI (structure) with initialization information.
%
% REQUIRED OUTPUT: PsiPostSample (matrix) of dimension (NumIterxpq) where p is the number of endogenous
%                            variables and q is the number of ecogenous variables in the Bayesian VAR.
%                            The Psi (steady state) parameters are located in each row using the vec
%                            of the Psi matrix and transposing it.
%
%                  PiPostSample (matrix) of dimension (NumIterxp^2k) where k is the number of lags in
%                            the Bayesian VAR model. The Pi parameters on lagged endogenous variables
%                            are located in each row using the vec operator on the Pi matrix and
%                            transposing it.
%
%                  OmegaPostSample (matrix) of dimension (NumIterxp(p+1)/2) with parameter draws for
%                            the unique elements of the residual covariance matrix. The Omega parameters
%                            are located in each row using the vech operator on the Omega matrix and
%                            transposing it.
%
%                  LogPsiDensity (vector) of dimension NumIter with values of log p(PsiMode|Omega(j),PiMode,Data).
%                            This data is used for estimating the marginal likelihood.
%
%                  LogPiDensity (vector) of dimension NumIter with values of log p(PiMode|Psi(i),Omega(i),Data).
%                            This data is used for estimating the marginal likelihood.
%
%                  NumBurnin (integer) with the number of burn-in draws.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: February 5, 2007.
%                        This version: December 18, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 6-2-2007:   Removed the outputs p, q, and k.
%
% * 28-2-2007:  The function now takes into account that the control "controls.posterior.burnin" displays the number
%               of burn-in draws rather than the percentage relative to the full posterior sample.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 18-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% locate the controls structure
%
maingui = findobj('Type','figure','Tag','YADA');
controls = get(maingui,'UserData');
%
% Collect the number of lags and number of variables
%
p = length(DSGEModel.BVARY);
k = DSGEModel.BVARLags;
q = length(DSGEModel.BVARX);
%
% Get the base values number of draws, etc.
%
% 1. Number of posterior draws
%
DrawsStr = get(controls.posterior.draws,'String');
NumPostDraws = str2num(StringTrim(strrep(DrawsStr(DSGEModel.PosteriorDrawsValue,:),',','')));
%
% 2. Number of posterior sample batches to save per chain
%
BatchStr = get(controls.posterior.batch,'String');
NumBatches = str2num(StringTrim(strrep(BatchStr(DSGEModel.SampleBatchValue,:),',','')));
if NumBatches>NumPostDraws;
   NumBatches = NumPostDraws;
end;
NumBatchDraws = NumPostDraws/NumBatches;
BatchIndex = (1:NumBatchDraws);
%
% 3. Number of burnin draws per chain
%
BurninStr = get(controls.posterior.burnin,'String');
NumBurnin = str2num(strrep(StringTrim(BurninStr(DSGEModel.BurnInValue,:)),',',''));
if NumBurnin>=NumPostDraws;
   txt = ['The number of burn-in draws (' StringTrim(BurninStr(DSGEModel.BurnInValue,:)) ') leaves no posterior draws (' StringTrim(DrawsStr(DSGEModel.PosteriorDrawsValue,:)) ') for inference. YADA will therefore use 0 burn-in draws.'];
   About(txt,'information','YADA - Burnin Draws Problem',180,500,CurrINI);
   NumBurnin = 0;
end;
%
% initialize the output
%
if q>0;
   PsiPostSample = zeros(NumPostDraws,p*q);
   LogPsiDensity = zeros(NumPostDraws,1);
else;
   PsiPostSample = [];
   LogPsiDensity = [];
end;
PiPostSample = zeros(NumPostDraws,p*p*k);
OmegaPostSample = zeros(NumPostDraws,p*(p+1)/2);
LogPiDensity = zeros(NumPostDraws,1);
%
% Time to load the data
%
for NBatch=1:NumBatches;
   PostSampleFile = [DSGEModel.OutputDirectory '\gibbs\BVAR-Gibbs-' DSGEModel.NameOfModel '-' num2str(DSGEModel.PriorType,'%0.0f') '.' num2str(DSGEModel.OmegaPriorType,'%0.0f') '-' num2str(NBatch,'%0.0f') '.' num2str(NumBatches,'%0.0f') '.' num2str(NumPostDraws,'%0.0f') '.mat'];
   GibbsData = load(PostSampleFile);
   if q>0;
      PsiPostSample(BatchIndex,:) = GibbsData.BatchPsiPostSample;
      LogPsiDensity(BatchIndex,1) = GibbsData.BatchLogPsiDensity;
   end;
   PiPostSample(BatchIndex,:) = GibbsData.BatchPiPostSample;
   OmegaPostSample(BatchIndex,:) = GibbsData.BatchOmegaPostSample;
   LogPiDensity(BatchIndex,1) = GibbsData.BatchLogPiDensity;
   BatchIndex = BatchIndex+NumBatchDraws;
end;

%
% end of LoadBVARPosteriorDraws.m
%
