function [ThetaBarPsi,SigmaBarPsi] = BVARPsiMean(Omega,Pi,thetaPsi,invSigmaPsi,x,X,d,dLag)
% BVARPsiMean: Computes the full condition posterior mean and, optionally, covariance of Psi in the BVAR.
%
% USAGE:
%
%       [ThetaBarPsi,SigmaBarPsi] = BVARPsiMean(Omega,Pi,thetaPsi,invSigmaPsi,x,X,d,dLag)
%
% REQUIRED INPUT:  Omega (pxp) covariance matrix of the VAR residuals.
%
%                  Pi (pxpk) matrix with the parameters on lagged enogenous variables.
%
%                  thetaPsi (pqx1) vector with the prior mean of vec(Psi).
%
%                  invSigmaPsi (pqxpq) matrix with the inverse of the prior covariance of vec(Psi).
%
%                  x (pxT) matrix with the contemporaneous data for the endogenous variables.
%
%                  X (pkxT) matrix with the lagged data for the endogenous variables.
%
%                  d (qxT) matrix with the contemporaneous data for the exogenous variables.
%
%                  dLag (qkxT) matrix with the lagged data for the exogenous variables.
%
% REQUIRED OUTPUT: ThetaBarPsi (pxq) matrix with the mean of the full conditional posterior of Psi.
%
% OPTIONAL OUTPUT: SigmaBarPsi (pqxpq) matrix with the covariance matrix of the full conditional posterior of Psi
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: January 15, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 30-1-2007:  Changed the function name from "BVARPsiMeanMinnesota" to "BVARPsiMean". The optional output
%               "SigmaBarPsi" is also computed.
%
% * 8-2-2007:   Renamed the argument invOmegsPsi as invSigmaPsi.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[p,T] = size(x);
q = size(d,1);
k = floor(size(X,1)/p);
%
% setting up the data
%
z = x-(Pi*X);
D = [d;dLag];
U = [eye(p*q);zeros(k*p*q,p*q)];
for i=1:k;
   U((p*q*i)+1:p*q*(i+1),:) = kron(eye(q),Pi(:,1+(p*(i-1)):p*i));
end;
invOmega = inv(Omega);
OmegaBarPsiInv = (U'*kron(D*D',invOmega)*U)+invSigmaPsi;
thetaBarPsi = OmegaBarPsiInv\((U'*vec(invOmega*z*D'))+(invSigmaPsi*thetaPsi));
ThetaBarPsi = reshape(thetaBarPsi,p,q);
if nargout==2;
   SigmaBarPsi = OmegaBarPsiInv\eye(p*q);
end;

%
% end of BVARPsiMean.m
%
