function [MeanPrediction,Quantiles,KernelX,KernelY] = BVARPredictionDistribution(OutputDirectory,NameOfModel,PriorType,OmegaPriorType,h,NumDraws,NumPaths,AnnualizeData,QuantileVector,VariableNames,CurrINI)
% BVARPredictionDistribution: Computes the mean value of unconditional prediction paths as well as quantiles
%                             for a Bayesian VAR by loading data from disk.
%
% USAGE:
%
%       [MeanPrediction,Quantiles,KernelX,KernelY] = BVARPredictionDistribution(OutputDirectory,NameOfModel,PriorType,...
%                                       OmegaPriorType,h,NumDraws,NumPaths,AnnualizeData,QuantileVector,VariableNames,...
%                                       CurrINI)
%
% REQUIRED INPUT:  OutputDirectory (string vector) with the full path to the base output directory.
%
%                  NameOfModel (string vector) with the name of the model.
%
%                  PriorType (integer) determining the type of prior used for the parameters on lagged endogenous
%                            variables of the Bayesian VAR. The value "1" is the Minnesota-style prior, "2" is the
%                            normal conditional on the covariance matrix prior, and "3" the diffuse prior.
%
%                  OmegaPriorType (integer) determining the type of prior used for the covariance matrix parameters.
%                            The value "1" is a diffuse prior, and "2" and inverse Wishart prior.
%
%                  h (integer) with the length of the prediction horizon.
%
%                  NumDraws (integer) with the total number of parameter draws.
%
%                  NumPaths (integer) with tht total number of prediction paths per parameter draw.
%
%                  AnnualizeData (boolean) that take the value 1 if annualized results should be checked and 0 otherwise.
%
%                  QuantileVector (vector) of dimension 1xNumQuants with percent values for the quantiles,
%                            e.g. [5 10 90 95].
%
%                  VariableNames (string matrix) with the names of the endogenous variables for the Bayesian VAR.
%
%                  CurrINI (structure) with initialization information.
%
% REQUIRED OUTPUT: MeanPrediction (matrix) of dimension nxh with the mean predictions.
%
%                  Quantiles (structure) of dimension NumQuants where each has a field called "Quantile" which
%                            is an nxh matrix with the quantiles of the prediction paths. The second field
%                            is "percent" that gives the percentile value.
%
%                  KernelX (3D matrix) of dimension n x 2^8 x MaxPH with the x-axis values.
%
%                  KernelY (3D matrix) of dimension n x 2^8 x MaxPH with the y-axis values.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: May 10, 2007.
%                        This version: March 3, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 13-11-2007: Updated the documentation.
%
% * 10-01-2008: Changed the last drawnow call to pause(0.02).
%
% * 23-05-2008: Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
% * 03-03-2009: Added the output variables "KernelX" and "KernelY".
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% initialize
%
p = size(VariableNames,1);
MeanPrediction = zeros(p,h);
NumQuants = length(QuantileVector);
KernelX = zeros(p,2^8,h);
KernelY = KernelX;
%
% Initialize quantiles output
%
for i=1:NumQuants;
   Quantiles(i).Quantile = zeros(p,h);
   Quantiles(i).percent = QuantileVector(i);
end;
%
% set up a wait dialog
%
txt = 'Please wait while YADA computes the mean predictions for the endogenous variables. The data are loaded sequentially from disk and it may therefore take some time.';
WaitHandle = WaitDLG(txt,'information','Predictive Distribution - Bayesian VAR',500,150,CurrINI,0);
WaitControls = get(WaitHandle,'UserData');
%
% compute the mean prediction
%
TotalPaths = NumPaths*NumDraws;
for ParamIndex=1:NumDraws;
   if AnnualizeData==0;
      file = [OutputDirectory '\predictions\BVAR-Prediction-Original-' NameOfModel '-' num2str(PriorType,'%0.0f') '.' num2str(OmegaPriorType,'%0.0f') '-' num2str(h,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPaths,'%0.0f') '.mat'];
   else;
      file = [OutputDirectory '\predictions\BVAR-Prediction-Annual-' NameOfModel '-' num2str(PriorType,'%0.0f') '.' num2str(OmegaPriorType,'%0.0f') '-' num2str(h,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPaths,'%0.0f') '.mat'];
   end;
   %
   % load the data
   %
   PredData = load(file);
   PredPaths = PredData.PredPaths;
   for i=1:NumPaths;
      MeanPrediction = MeanPrediction+((1/TotalPaths)*PredPaths(:,:,i));
   end;
   clear('PredData');
   drawnow;
end;
%
% compute all the quantile values
%
for j=1:h;
   for i=1:p;
      txt = ['Please wait while YADA computes quantile values and kernel density estimates for the predictions of the endogenous variables. Current variable: ' StringTrim(VariableNames(i,:)) ' and horizon ' num2str(j,'%0.0f') '.'];
      set(WaitControls.text,'String',txt);
      drawnow;
      %
      % initialzie a temporary variable
      %
      TempVar = zeros(TotalPaths,1);
      for ParamIndex=1:NumDraws;
         if AnnualizeData==0;
            file = [OutputDirectory '\predictions\BVAR-Prediction-Original-' NameOfModel '-' num2str(PriorType,'%0.0f') '.' num2str(OmegaPriorType,'%0.0f') '-' num2str(h,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPaths,'%0.0f') '.mat'];
         else;
            file = [OutputDirectory '\predictions\BVAR-Prediction-Annual-' NameOfModel '-' num2str(PriorType,'%0.0f') '.' num2str(OmegaPriorType,'%0.0f') '-' num2str(h,'%0.0f') '-' num2str(ParamIndex,'%0.0f') '.' num2str(NumDraws,'%0.0f') '.' num2str(NumPaths,'%0.0f') '.mat'];
         end;
         %
         % load the data
         %
         PredData = load(file);
         PredPaths = PredData.PredPaths;
         for k=1:NumPaths;
            CurrPath = k+(NumPaths*(ParamIndex-1));
            TempVar(CurrPath,1) = PredPaths(i,j,k);
         end;
         clear('PredData');
         drawnow;
      end;
      %
      % TempVar is now fully loaded and may
      %
      TempVar = sort(TempVar);
      for k=1:NumQuants;
         if QuantileVector(k)<50;
            QuantVal = ceil((QuantileVector(k)/100)*TotalPaths);
         else;
            QuantVal = floor((QuantileVector(k)/100)*TotalPaths);
         end;
         Quantiles(k).Quantile(i,j) = TempVar(QuantVal,1);
      end;
      if TempVar(1)<TempVar(TotalPaths);
         [x,f] = YADAMHDensEst(TempVar,2^8,0,'gaussian',[]);
         KernelX(i,:,j) = x';
         KernelY(i,:,j) = f';
      end;
   end;
end;
%
% close the wait dialog
%
set(WaitControls.text,'UserData','done');
delete(WaitHandle);
drawnow;
pause(0.02);

%
% end of BVARPredictionDistribution.m
%
