function [muBarPi,OmegaBarPi] = BVARPiMeanNormalCond(Psi,muPi,invOmegaPi,x,X,d,dLag)
% BVARPiMeanNormalCond: Computes the full condition posterior mean of Pi in the BVAR when a Normal conditional on the residual
%                       covariance matrix prior is used for the parameters on lagged endogenous variables. Optionally, it can
%                       also compute part of the covariance matrix of the full conditional posterior of vec(Pi).
%
% USAGE:
%
%       [muBarPi,OmegaBarPi] = BVARPiMeanNormalCond(Psi,muPi,invOmegaPi,x,X,d,dLag)
%
% REQUIRED INPUT:  Psi (pxq) matrix with steady state parameters.
%
%                  muPi (pxpk) matrix with the prior mean of Pi.
%
%                  invOmegaPi (pkxpk) matrix where the prior covariance of vec(Pi) is kron(inv(invOmegaPi),Omega).
%
%                  x (pxT) matrix with the contemporaneous data for the endogenous variables.
%
%                  X (pkxT) matrix with the lagged data for the endogenous variables.
%
%                  d (qxT) matrix with the contemporaneous data for the exogenous variables.
%
%                  dLag (qkxT) matrix with the lagged data for the exogenous variables.
%
% REQUIRED OUTPUT: muBarPi (pxpk) matrix with the mean of the full conditional posterior of Pi.
%
% OPTIONAL OUTPUT: OmegaBarPi (pkxpk) matrix where the full conditional posterior covariance of vec(Pi)
%                             is kron(OmegaBarPi,Omega).
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: January 17, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 30-1-2007:  Added the optional output "OmegaBarPi".
%
% * 8-2-2007:   Renamed the input ThetaPi to muPi, and the output ThetaBarPi to muBarPi.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[p,T] = size(x);
q = size(d,1);
k = floor(size(X,1)/p);
%
% Set up data matrices for computing the log-likelihood
%
if q>0;
   y = x-(Psi*d);
   Y = X+(kron(eye(k),Psi)*dLag);
else;
   y = x;
   Y = X;
end;
muBarPi = ((y*Y')+(muPi*invOmegaPi))/((Y*Y')+invOmegaPi);
if nargout==2;
   OmegaBarPi = ((Y*Y')+invOmegaPi)\eye(p*k);
end;

%
% end of BVARPiMeanNormalCond.m
%
