function [ThetaBarPi,SigmaBarPi] = BVARPiMeanMinnesota(Omega,Psi,thetaPi,invOmegaPi,x,X,d,dLag)
% BVARPiMeanMinnesota: Computes the full condition posterior mean of Pi in the BVAR when
%                      a Minnesota-style prior is used for these parameters. Optionally, it
%                      can also compute the covariance matrix of the full conditional posterior
%                      of vec(Pi).
%
% USAGE:
%
%       [ThetaBarPi,SigmaBarPi] = BVARPiMeanMinnesota(Omega,Psi,thetaPi,invOmegaPi,x,X,d,dLag)
%
% REQUIRED INPUT:  Omega (pxp) covariance matrix of the VAR residuals.
%
%                  Psi (pxq) matrix with steady state parameters.
%
%                  thetaPi (p^2kx1) vector with the prior mean of vec(Pi).
%
%                  invOmegaPi (p^2kxp^2k) matrix with the inverse of the prior covariance of vec(Pi).
%
%                  x (pxT) matrix with the contemporaneous data for the endogenous variables.
%
%                  X (pkxT) matrix with the lagged data for the endogenous variables.
%
%                  d (qxT) matrix with the contemporaneous data for the exogenous variables.
%
%                  dLag (qkxT) matrix with the lagged data for the exogenous variables.
%
% REQUIRED OUTPUT: ThetaBarPi (pxpk) matrix with the mean of the full conditional posterior of Pi.
%
% OPTIONAL OUTPUT: SigmaBarPi (p^2kxp^2k) matrix with the covariance matrix of the full conditional
%                             posterior of vec(Pi).
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: January 15, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 30-1-2007:  Added the optional output "SigmaBarPi".
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[p,T] = size(x);
q = size(d,1);
k = floor(size(X,1)/p);
%
% Set up data matrices for computing the log-likelihood
%
if q>0;
   y = x-(Psi*d);
   Y = X+(kron(eye(k),Psi)*dLag);
else;
   y = x;
   Y = X;
end;
invOmega = inv(Omega);
OmegaBarPiInv = kron(Y*Y',invOmega)+invOmegaPi;
thetaBarPi = OmegaBarPiInv\(vec(invOmega*y*Y')+(invOmegaPi*thetaPi));
ThetaBarPi = reshape(thetaBarPi,p,p*k);
if nargout==2;
   SigmaBarPi = OmegaBarPiInv\eye(p*p*k);
end;

%
% end of BVARPiMeanMinnesota.m
%
