function Omega = BVAROmegaNormal(Pi,Psi,A,qDF,muPi,invOmegaPi,x,X,d,dLag)
% BVAROmegaNormal: Compute a value for the posterior mode of the residual covariance matrix
%                  in the BVAR conditional on values for the Pi and Psi parameters.
%
% USAGE:
%
%       Omega = BVAROmegaNormal(Pi,Psi,A,qDF,x,X,d,dLag)
%
% REQUIRED INPUT:  Pi (pxpk) matrix with the parameters on lagged enogenous variables.
%
%                  Psi (pxq) matrix with steady state parameters.
%
%                  A (pxp) matrix, with the first parameter of the prior of Omega.
%
%                  qDF (integer), with the second (degrees of freedom) parameter of the prior of Omega.
%
%                  muPi (pxpk) matrix with the prior mean of Pi.
%
%                  invOmegaPi (pkxpk) matrix where the prior covariance of vec(Pi) is kron(inv(invOmegaPi),Omega).
%
%                  x (pxT) matrix with the contemporaneous data for the endogenous variables.
%
%                  X (pkxT) matrix with the lagged data for the endogenous variables.
%
%                  d (qxT) matrix with the contemporaneous data for the exogenous variables.
%
%                  dLag (qkxT) matrix with the lagged data for the exogenous variables.
%
% REQUIRED OUTPUT: OmegaBar (pxp) matrix with the posterior mode of Omega from the conditional posterior.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: January 18, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 30-1-2007:  Changed the check for invOmegaPi such that if it is 0 then the first case
%               is used.
%
% * 8-2-2007:   Renamed the input ThetaPi to muPi.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[p,T] = size(x);
q = size(d,1);
k = floor(size(X,1)/p);
%
% Set up data matrices for computing the log-likelihood
%
if q>0;
   y = x-(Psi*d);
   Y = X+(kron(eye(k),Psi)*dLag);
else;
   y = x;
   Y = X;
end;
Epsilon = y-(Pi*Y);
if (length(invOmegaPi)==1)&(isequal(invOmegaPi,0)==1);
   c = 0;
   B = 0;
else;
   c = p*k;
   B = (Pi-muPi)*invOmegaPi*(Pi-muPi)';
end;
Omega = (1/(T+qDF+p+c+1))*((Epsilon*Epsilon')+A+B);

%
% end of BVAROmegaNormal.m
%
