function logPost = BVARLogPosteriorMinnesota(Omega,Pi,Psi,A,qDF,thetaPsi,SigmaPsi,thetaPi,SigmaPi,x,X,d,dLag)
% BVARLogPosteriorMinnesota: Computes the value of the log posterior for a BVAR when a Minnesota-style
%                            prior is used for the parameters on lagged endogenous variables
%
% USAGE:
%
%       logPost = BVARLogPosteriorMinnesota(Omega,Pi,Psi,A,qDF,thetaPsi,SigmaPsi,thetaPi,SigmaPi,x,X,d,dLag)
%
% REQUIRED INPUT:  Omega (pxp) covariance matrix of the VAR residuals.
%
%                  Pi (pxpk) matrix with parameters on the k lags of x.
%
%                  Psi (pxq) matrix with steady state parameters.
%
%                  A (pxp) matrix, with the first parameter of the prior of Omega.
%
%                  qDF (integer), with the second (degrees of freedom) parameter of the prior of Omega.
%
%                  thetaPsi (pqx1) vector with the prior mean of vec(Psi).
%
%                  SigmaPsi (pqxpx) matrix with the prior covariance of vec(Psi).
%
%                  thetaPi (p^2kx1) vector with the prior mean of vec(Pi).
%
%                  SigmaPi (p^2kxp^2k) matrix with the prior covariance of vec(Pi).
%
%                  x (pxT) matrix with the contemporaneous data for the endogenous variables.
%
%                  X (pkxT) matrix with the lagged data for the endogenous variables.
%
%                  d (qxT) matrix with the contemporaneous data for the exogenous variables.
%
%                  dLag (qkxT) matrix with the lagged data for the exogenous variables.
%
% REQUIRED OUTPUT: logPost (scalar), with the value of the log posterior.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2008 European Central Bank.
%
%                       First version: January 15, 2007.
%                        This version: December 19, 2008.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 19-1-2007:  Added constant terms to the log densities.
%
% * 8-2-2007:   Changed the argument OmegaPsi to SigmaPsi, and OmegaPi to SigmaPi.
%
% * 13-11-2007: Updated the documentation.
%
% * 23-5-2008:  Updated the documentation.
%
% * 19-12-2008: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

[p,T] = size(x);
q = size(d,1);
k = floor(size(X,1)/p);
%
% Set up data matrices for computing the log-likelihood
%
if q>0;
   y = x-(Psi*d);
   Y = X+(kron(eye(k),Psi)*dLag);
else;
   y = x;
   Y = X;
end;
Epsilon = y-(Pi*Y);
logLikelihoodBVAR = -(((T*p)/2)*log(2*pi))-((T/2)*log(det(Omega)))-((1/2)*sum(diag(Omega\(Epsilon*Epsilon'))));
%
% compute the prior
%
logPriorPi = -(((p*p*k)/2)*log(2*pi))-((1/2)*log(det(SigmaPi)))-((1/2)*(vec(Pi)-thetaPi)'*inv(SigmaPi)*(vec(Pi)-thetaPi));
%
if q>0;
   logPriorPsi = -(((p*q)/2)*log(2*pi))-((1/2)*log(det(SigmaPsi)))-((1/2)*(vec(Psi)-thetaPsi)'*inv(SigmaPsi)*(vec(Psi)-thetaPsi));
else;
   logPriorPsi = 0;
end;
%
if det(A)>0;
   logPriorOmega = -(((qDF*p)/2)*log(2))-(((p*(p-1))/4)*log(pi))-LogGammaProduct(p,qDF)+((qDF/2)*log(det(A)))-(((qDF+p+1)/2)*log(det(Omega)))-((1/2)*sum(diag(Omega\A)));
else;
   logPriorOmega = -((p+1)/2)*log(det(Omega));
end;
logPost = logLikelihoodBVAR+logPriorPi+logPriorPsi+logPriorOmega;

%
% end of BVARLogPosteriorMinnesota.m
%
