function [h,q,iq,nnumeric] = AiMNumericShift(h,q,iq,qrows,qcols,neq,condn)
% AiMNumericShift: Compute the numeric shiftrights and store them in q.
%
% USAGE:
%
%       [h,q,iq,nnumeric] = AiMNumericShift(h,q,iq,qrows,qcols,neq,condn)
%
% REQUIRED INPUT:  h (matrix) with structural coefficients - dimension (neq,neq*(nlag+1+nlead)).
%
%                  q (matrix) with exact shiftrights.
%
%                  iq (integer) with the number of exact shiftrights.
%
%                  qrows (integer) with the number of rows (=neq*nlead) of q.
%
%                  qcols (integer) with the number of columns (=neq*(nlag+nlead)) of q.
%
%                  neq (integer) with the number of equations.
%
%                  condn (positive scalar) indicating numerical tolerance.
%
% REQUIRED OUTPUT: h (matrix) with structural coefficients - dimension (neq,neq*(nlag+1+nlead)).
%
%                  q (matrix) with all shiftrights.
%
%                  iq (integer) with the total number of shiftrights.
%
%                  nnumeric (integer) with the number of numeric shiftrights.
%
% NOTE: This function is based on aim_numeric_shift.m
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: August 25, 2006.
%                        This version: March 3, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 19-05-2008: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 05-12-2008: Updated the documentation.
%
% * 03-03-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

nnumeric = 0;
left     = 1:qcols;
right    = qcols+1:qcols+neq;
%
[Q,R,E]  = qr(h(:,right));
zerorows = find(abs(diag(R))<=condn);
%
while (any(zerorows)&(iq<=qrows));
   h = Q'*h;
   nz = length(zerorows);
   %   
   q(iq+1:iq+nz,:) = h(zerorows,left);
   h(zerorows,:) = AiMShiftRight( h(zerorows,:), neq );
   %
   iq       = iq + nz;
   nnumeric = nnumeric + nz;
   %
   [Q,R,E] = qr(h(:,right));
   zerorows = find(abs(diag(R))<=condn);
end;

%
% end of AiMNumericShift.m
%
