function [a,ia,js] = AiMBuildCompanion(h,qcols,neq)
% AiMBuildCompanion: Build the companion matrix, deleting inessential lags.
%
% USAGE:
%
%       [a,ia,js] = AiMBuildCompanion(h,qcols,neq)
%
% REQUIRED INPUT:  h (matrix) with structural coefficients - dimension (neq,neq*(nlag+1+nlead)).
%
%                  qcols (integer) with the number of columns (=neq*(nlag+nlead)) of q.
%
%                  neq (integer) with the number of equations.
%
% REQUIRED OUTPUT: a (matrix) with the companion matrix for computing the non-zero eigenvalues.
%
%                  ia (integer) with the number of essential lags of the model.
%
%                  js (vector) with the columns of the transition matrix with the essential
%                     lags.
%
% NOTE: This function is based on aim_build_a.m
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: August 25, 2006.
%                        This version: March 3, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 19-05-2008: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 05-12-2008: Updated the documentation.
%
% * 03-03-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%
%  Solve for x_{t+nlead} in terms of x_{t+nlag},...,x_{t+nlead-1}.
%
left  = 1:qcols;
right = qcols+1:qcols+neq;
%
h(:,left) = -h(:,right)\h(:,left);
%
%  Build the big transition matrix.
%
a = zeros(qcols,qcols);
if (qcols>neq);
   eyerows = 1:qcols-neq;
   eyecols = neq+1:qcols;
   a(eyerows,eyecols) = eye(qcols-neq);
end;
%
hrows      = qcols-neq+1:qcols;
a(hrows,:) = h(:,left);
%
%  Delete inessential lags and build index array js.  js indexes the
%  columns in the big transition matrix that correspond to the
%  essential lags in the model.  They are the columns of q that will
%  get the unstable left eigenvectors. 
%
js       = 1:qcols;
zerocols = sum(abs(a)) == 0;
%
while (any(zerocols));
   a(:,zerocols) = [];
   a(zerocols,:) = [];
   js(zerocols)  = [];
   zerocols = sum(abs(a)) == 0;
end;
ia = length(js);

%
% end of AiMBuildCompanion.m
%
