function err = CheckAiMSolution(neq,nlag,nlead,h,b)
% CheckAiMSolution: Checks the accuracy of the AiM solution to the model
%
% USAGE:
%
%       err = CheckAiMSolution(neq,nlag,nlead,h,b)
%
% REQUIRED INPUT:  neq (integer) with the number of equations of the model.
%
%                  nlag (integer) with the number of lags of the model.
%
%                  nlead (integer) with the number of leads of the model.
%
%                  h (matrix) with structural coefficients - dimension (neq,neq*(nlag+1+nlead)).
%
%                  b (matrix) with reduced form coefficients - dimension (neq,neq*nlag).
%
% REQUIRED OUTPUT: err (scalar) with the accuracy of the AiM solution.
%
% NOTE: This function is based on aim_checkaim.m
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: August 25, 2006.
%                        This version: March 3, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 19-05-2008: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 03-03-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
%   Substitute the reduced form, b, into the structural 
%   equations, h, and display the maximum absolute error.
%
% Append negative identity matrix to b
%
b = [b, -eye(neq)];
%
%  Define indexes into the lagged part (minus) and the current and
%  lead part (plus) of h 
minus =          1:           neq*nlag;
plus  = neq*nlag+1: neq*(nlag+1+nlead);
%
% Initialize q
%
q = zeros(neq*(nlead+1), neq*(nlag+1+nlead));
%
% Stuff b into the upper left-hand block of q
%
[rb,cb] = size(b);
q(1:rb,1:cb) = b(1:rb,1:cb);
%
%  Copy b into the (nlead) row blocks of q, shifting right by neq once
%  more for each block: this produces a coefficient matrix that 
%  solves for (x(t),..., x(t+nlead))' in terms of (x(t-nlag),..., x(t-1))'.
%
for i=1:nlead;
   rows = i*neq + (1:neq);
   q(rows,:) = AiMShiftRight(q(rows-neq,:),neq);
end;
%
%  Premultiply the left block of q by the negative inverse of the right
%  block of q
%
q(:,minus) =  -q(:,plus)\q(:,minus);
%
% Define the solution error
%
err = max(max(abs(h(:,minus)+h(:,plus)*q(:,minus))));

%
% end of CheckAiMSolution.m
%