function [F,B0] = AiMtoStateSpace(BMatrix,SMatrix,StateVariablePositions,StateShockPositions)
% AiMtoStateSpace: Transforms the AiM output matrices B and S into matrices that can be used by
%                  by the KalmanFilter function for the state equation.
%
% USAGE:
%
%       [F,B0] = AiMtoStateSpace(BMatrix,SMatrix)
%
% REQUIRED INPUTS:  BMatrix (nxnp) matrix provided by the AiMSolver function. It cointains the
%                           reduced form coefficient matrices from the model solution ordered as
%                           B(p), B(p-1),..., B(1). If p>1, we therefore need to reorder these
%                           matrices in F. This is handled by the AiMReverseCoeffMatrix function.
%
%                   SMatrix (nx(np+1)) matrix provided by the AiMSolver function. It contains
%                           the observed structure. The S(i) matrices are ordered in SMatrix
%                           according to S(p), S(p-1),...,S(0).
%                   NOTE: B(i) = -inv(S(0))*S(i), for i=1,...,p
%
%                   StateVariablePositions (N) vector with N<=n containing the positions of the
%                           actual state variables. AiM will typically stack the state variables
%                           and the state shocks given that the shocks have been formulated as
%                           equations!
%
%                   StateShockPositions (M) vector with M<=n containing the positions of the state
%                           shocks.
%                   NOTE: N+M should generally not be bigger than n.
%
% REQUIRED OUTPUTS: F (npxnp) matrix with the parameters on the lags of the state-equation in
%                           the state-space representation.
%
%                   B0 (npxn) is equal to inv(S(0)), i.e., the coefficient matrix on the shocks
%                           in the state equation. Letting Q be the covariance matrix of the shocks
%                           in the state equation, Q = B0*B0'
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: July 19, 2006.
%                        This version: March 3, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 03-08-2006: Added the input vectors StateVariablePositions and StateShockPositions
%
% * 08-08-2006: B0 now has a zero matrix below if p>1.
%
% * 30-10-2007: Updated the documentation.
%
% * 23-05-2008: Updated the documentation.
%
% * 05-12-2008: Updated the documentation.
%
% * 09-02-2009: Fixed a bug when setting up B0 for p>1 and added a missing semi-colon
%               when constructing "StateVarColumns". These errors were spotted by Stefan
%               Laseen.
%
% * 03-03-2009: Updated the documentation.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% We begin by checking the original size of BMatrix and SMatrix
%
[n,p] = size(BMatrix);
p = floor(p/n);
%
% Use StateVariablePositions to select the elements of BMatrix correponding to the actual state variables
%
if size(StateVariablePositions,1)>size(StateVariablePositions,2);
   StateVariablePositions = StateVariablePositions';
end;
if size(StateShockPositions,1)>size(StateShockPositions,2);
   StateShockPositions = StateShockPositions';
end;
StateVarColumns = StateVariablePositions;
for i=2:p;
   StateVarColumns = [StateVarColumns (n*(i-1) + StateVariablePositions )];
end;
%
s = size(SMatrix,2);
B0 = inv(SMatrix(:,s-n+1:s));
B0 = B0(StateVariablePositions,StateShockPositions);
%
BMatrix = BMatrix(StateVariablePositions,StateVarColumns);
[n,p] = size(BMatrix);
p = floor(p/n);
%
% First we select the last nxn columns of Smatrix, and transform that into B0
%
if p==1;
   F = BMatrix;
else;
   %
   % We have to shift. Construction if ShiftMatrix should only have to be peformed once
   %
   F = [[AiMReverseCoeffMatrix(BMatrix)];[eye(n*(p-1)) zeros(n*(p-1),n)]];
   B0 = [B0;zeros(n*(p-1),size(B0,2))];
end;

%
% end of AiMtoStateSpace.m
%
