function YADA
% YADA: Initializes the YADA GUI.
%
% USAGE:
%
%       YADA
%
% Accepts not inputs and returns no output.
%
% This file checks if the YADA files can be located by Matlab. Once the path is properly setup, it initializes
% the YADA GUI. When the GUI function exits, the GUI is deleted and any files stored in the tmp directory are
% deleted. Also, the Matlab path is reset to its original value.
%
% NOTE: To run YADA, this file should be executed from the Matlab prompt.
%
%
%                       Written by: Anders Warne
%                                   New Area Wide Model Project
%                                   DG-R/EMO
%                                   European Central Bank (ECB)
%                                   Email: anders.warne@ecb.europa.eu
%                                   Copyright  2006-2009 European Central Bank.
%
%                       First version: July 17, 2006.
%                        This version: March 9, 2009.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% LICENSE INFORMATION:
%
%      YADA is free software: you can redistribute it and/or modify
%      it under the terms of the GNU General Public License as published by
%      the Free Software Foundation, either version 3 of the License, or
%      (at your option) any later version.
%
%      This program is distributed in the hope that it will be useful,
%      but WITHOUT ANY WARRANTY; without even the implied warranty of
%      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%      GNU General Public License for more details.
%
%      You should have received a copy of the GNU General Public License
%      along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%      YADA is released under the GNU General Public License, Version 3,
%      29 June 2007 <http://www.gnu.org/licenses/>. The current release of
%      the program was last modified by the ECB on the "This version" date
%      above.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% CHANGELOG:
%
% * 19-07-2006: Updated function documentation.
%
% * 20-07-2006: The path is now stored in the variable ProgPath.
%
% * 21-07-2006: The YADA GUI call was added.
%
% * 24-07-2006: Updated function documentation.
%
% * 26-07-2006: Included tests for operating system and screen size.
%
% * 27-07-2006: Renamed FixPath to FixYADAPath! Made sure the path is only changed to include the YADA directories
%               while YADA is running. The original path is reset just before YADA exists. Also, all files copied to
%               the directory tmp are always deleted before YADA closes!
%
% * 01-08-2006: Added check for multiple instances of YADA.
%
% * 01-09-2006: Added a warning off for YADA.
%
% * 06-09-2006: Fixed the warning command for Matlab 7 and later.
%
% * 23-11-2006: YADAGUI no longer takes the ProgPath input. Moreover, FixYADAPath is not tested. The YADAPath
%               function is always run. The file "iswinxp.txt" is now deleted before running the YADAGUI function.
%
% * 02-02-2007: Updated the documentation.
%
% * 11-04-2007: Added a clear('all') call.
%
% * 02-11-2007: Updated the documentation.
%
% * 03-03-2008: Took the changed status of observed variable scenarios into account.
%
% * 04-03-2008: Took the changed status of shock scenarios into account.
%
% * 10-03-2008: Added the clear('all') command towards the end and the beginning.
%
% * 17-03-2008: The code now takes into account that the Waggoner-Zha type of conditional forecasting
%               has been stripped from the public version of YADA.
%
% * 18-03-2008: A slight change in directory structure where all add-ons for YADA are now stored
%               in sub-directories to "addons" rather than in "private".
%
% * 19-03-2008: Took into account that "CondPredictionKalmanSmoother" has been moved from the "filter"
%               directory.
%
% * 31-03-2008: Make sure that shellprint is removed from the bin directory.
%
% * 13-05-2008: Made sure that EditDataDLG is deleted from the gui directory.
%
% * 19-05-2008: Old YADA figures are now being deleted by YADA.
%
% * 21-05-2008: Updated the documentation.
%
% * 23-05-2008: Added the #function pragme.
%
% * 27-06-2008: Removed all code where files that have been moved are deleted in their old directories
%               when still found there.
%
% * 23-07-2008: The function now deletes "logStdbetaPDF.m" if found in "dist" directory. YADA now uses
%               a general beta prior.
%
% * 24-07-2008: Deletes "SimulateAnSchorfheideDSGEmodel.m" and "TestKalmanFilter.m" if found in the base
%               directory.
%
% * 18-08-2008: Deletes "DSGEPredictionOVSTheta.m" if found in the addons directory.
%
% * 25-08-2008: Removed a number of file checks and deletes.
%
% * 28-11-2008: Updated the documentation and added a dos('dir') check.
%
% * 04-12-2008: Changed the screen height requirement to a screen height nag!
%
% * 05-12-2008: Corrected spelling for the nag about the screen height. The screen height is now measured
%               through the 4th and 2nd elements of vector obtained through the get(0,'ScreenSize') command.
%
% * 08-12-2008: Added quotes around the path for checking if YADA.exe exists. Added code for checking if the
%               license has been accepted or not.
%
% * 05-01-2009: Updated the documentation.
%
% * 26-02-2009: The function "GraphEditDataDLG" has been moved to the "gui" directory. It's therefore deleted
%               from the addons directory is found to reside there.
%
% * 09-03-2009: Added quotation marks around paths for the dos del commands.
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%

%
% make sure we have as much memory as possible
%
clear('all');
%
if isempty(findobj('Type','figure','Tag','YADA'))==0;
   %
   % delete this handle
   %
   delete(findobj('Type','figure','Tag','YADA'));
   drawnow;
   pause(0.02);
end;
%
% check if the computer supports YADA
%
if strcmp(computer,'PCWIN')==0;
   disp('YADA only runs on Microsoft Windows computers.');
   return;
end;
scrsz = get(0,'ScreenSize');
if scrsz(4)+scrsz(2)-1<768;
   disp('YADA is best viewed on a screen with a minimum height of 768 pixels.');
end;
[Status,Txt] = dos('dir');
if Status~=0;
   disp('YADA wasn''t able to run the "dos" command successfully.');
   return;
end;
%
% add minimum requirement of functions for the #function pragma in case someone has the brilliant
% idea of compiling YADA with the Matlab compiler (not recommended).
%
%#function CloseGUI ShowStatus StopProgressDLG CancelProgressDLG DisplayComment TextGUICopy TextGUIPrint CloseProgressDLG
%
% update the path and make sure the path is stored in the ProgPath string variable.
%
OrigPath = path;
fid = fopen(['"' pwd '\YADA.exe"'],'rb');
if fid==-1;
   YADAPath;
else;
   fclose(fid);
end;
%
% Set warning level to off
%
if MatlabNumber>=7;
   WarnStr = warning;
else;
   [WarnStr,WarnFreq] = warning;
end;
warning('off');
%
% check if the user has accepted the GNU General Public License
%
AcceptLicense = 0;
if FileExist([pwd '\AcceptLicense.txt'])==1;
   AcceptTxt = ReadText([pwd '\AcceptLicense.txt']);
   if min(size(AcceptTxt))==1;
      AcceptTxt = AcceptTxt{1};
      AcceptTxt = strrep(AcceptTxt,' ','');
      AcceptLicense = str2double(AcceptTxt);
      if isnumeric(AcceptLicense)==0;
         AcceptLicense = 0;
      end;
   end;
end;
%
% check if the license has been accepted
%
if AcceptLicense==0;
   [VersionOfYADA,VersionDateStr] = YADAVersion;
   TxtStr = strvcat('APPROPRIATE LEGAL NOTICES FOR YADA','This program is free software: you can redistribute it and/or modify it in accordance with the terms of the GNU General Public License (http://www.gnu.org/licenses/) as published by the Free Software Foundation, either Version 3 of the license, or (at your option) any later version.',' ','In order to use the program you must first accept the following disclaimer and limitation of liability.',' ','DISCLAIMER OF WARRANTY','By accepting the disclaimer you assume the entire risk related to your use of this program. The European Central Bank (ECB) provides this program ''as is'' and the ECB disclaims any warranties, whether express or implied, including, but not limited to, the implied warranties of merchantability and fitness for a particular purpose. The entire risk as to the quality and performance of the program is with you.',' ','LIMITATION OF LIABILITY','In no event shall the ECB be liable to you or to any third party for any direct, indirect, incidental, consequential, general, special or exemplary damages or lost profit resulting from any use of or inability to use the program (including but not limited to loss of data or data being rendered inaccurate or losses sustained by you or third parties or a failure of the program to operate with any other programs). Should the program prove defective, you assume the cost of all necessary servicing, repair and/or corrections.',' ','COPYRIGHT NOTICE',['This program is released under the GNU General Public License, Version 3, 29 June 2007 (http://www.gnu.org/licenses/). The current release of the program is version ' num2str(VersionOfYADA,'%0.2f') ' and it was last modified by the ECB on ' VersionDateStr '.'],' ',['Copyright  2006-' datestr(now,10) ' European Central Bank.'],'Email: anders.warne@ecb.europa.eu');
   Answer = Query(TxtStr,'logo',500,'Appropriate Legal Notices for YADA',600,'no',InitializeINIFile,{'I Accept' 'I Don''t Accept' 'View License'},125,[pwd '\gpl.txt']);
   if strcmp(Answer,'I Accept')==1;
      %
      % write to file 
      %
      fid = fopen([pwd '\AcceptLicense.txt'],'wt');
      fprintf(fid,'1\n');
      fclose(fid);
      AcceptLicense = 1;
   end;
end;
if AcceptLicense==0;
   path(OrigPath);
   warning(WarnStr);
   clear('all');
   return;
end;
%
% delete the old iswinxp.txt file and create a new one. This was different OS's can
% use the same YADA directory, and copying the directory doesn't imply that the wrong
% iswinxp.txt file is examined.
%
file = [pwd '\iswinxp.txt'];
if FileExist(file)==1;
   stat = dos(['del /Q "' file '"']);
end;
%
if FileExist([pwd '\addons\ovs\GraphEditDataDLG.m'])==1;
   stat = dos(['del /Q "' pwd '\addons\ovs\GraphEditDataDLG.m"']);
end;
%
% we can now start up the YADA GUI
%
YADAHandle = YADAGUI('init');
%
% Close the GUI
%
if (isempty(YADAHandle)==0)&(ishandle(YADAHandle)==1);
   delete(YADAHandle);
else;
   %
   % search for YADA figure
   %
   YADAHandle = findobj('Type','figure','Tag','YADA');
   if isempty(YADAHandle)==0;
      controls = get(YADAHandle,'UserData');
      if (isempty(controls)==0)&(isstruct(controls)==1);
         if isfield(controls,'leave')==1;
            set(controls.leave,'UserData','done');
            delete(YADAHandle);
         end;
      end;
   end;
end;
%
% Delete any files that were copied to tmp
%
stat = dos('del /Q tmp\*.*');
%
% Make sure that the path is reset to the OrigPath string variable that was created before YADA
% was initialized.
%
path(OrigPath);
%
% reset the warning state.
%
warning(WarnStr);
%
% make sure that memory is cleared
%
clear('all');

%
% end of YADA.m
%
