---------------------------------------------------------------------------------------------------
-- Copyright (c) 2022 by Enclustra GmbH, Switzerland.
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy of
-- this hardware, software, firmware, and associated documentation files (the
-- "Product"), to deal in the Product without restriction, including without
-- limitation the rights to use, copy, modify, merge, publish, distribute,
-- sublicense, and/or sell copies of the Product, and to permit persons to whom the
-- Product is furnished to do so, subject to the following conditions:
--
-- The above copyright notice and this permission notice shall be included in all
-- copies or substantial portions of the Product.
--
-- THE PRODUCT IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
-- INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
-- PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
-- HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
-- OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
-- PRODUCT OR THE USE OR OTHER DEALINGS IN THE PRODUCT.
---------------------------------------------------------------------------------------------------

---------------------------------------------------------------------------------------------------
-- libraries
---------------------------------------------------------------------------------------------------
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

---------------------------------------------------------------------------------------------------
-- entity declaration
---------------------------------------------------------------------------------------------------
entity Mars_ZX2_ST3 is
  port (
    -- PS MIO Pins
    FIXED_IO_mio                   : inout  std_logic_vector(53 downto 0);
    FIXED_IO_ddr_vrn               : inout  std_logic;
    FIXED_IO_ddr_vrp               : inout  std_logic;
    FIXED_IO_ps_srstb              : inout  std_logic;
    FIXED_IO_ps_clk                : inout  std_logic;
    FIXED_IO_ps_porb               : inout  std_logic;

    -- PS DDR Interface
    DDR_cas_n                      : inout  std_logic;
    DDR_cke                        : inout  std_logic;
    DDR_ck_n                       : inout  std_logic;
    DDR_ck_p                       : inout  std_logic;
    DDR_cs_n                       : inout  std_logic;
    DDR_reset_n                    : inout  std_logic;
    DDR_odt                        : inout  std_logic;
    DDR_ras_n                      : inout  std_logic;
    DDR_we_n                       : inout  std_logic;
    DDR_ba                         : inout  std_logic_vector(2 downto 0);
    DDR_addr                       : inout  std_logic_vector(14 downto 0);
    DDR_dm                         : inout  std_logic_vector(3 downto 0);
    DDR_dq                         : inout  std_logic_vector(31 downto 0);
    DDR_dqs_n                      : inout  std_logic_vector(3 downto 0);
    DDR_dqs_p                      : inout  std_logic_vector(3 downto 0);

    -- Anios IO Connector 0
    IO0_D0_P                       : inout   std_logic;
    IO0_D1_N                       : inout   std_logic;
    IO0_D2_P                       : inout   std_logic;
    IO0_D3_N                       : inout   std_logic;
    IO0_D4_P                       : inout   std_logic;
    IO0_D5_N                       : inout   std_logic;
    IO0_D6_P                       : inout   std_logic;
    IO0_D7_N                       : inout   std_logic;
    IO0_D8_P                       : inout   std_logic;
    IO0_D9_N                       : inout   std_logic;
    IO0_D10_P                      : inout   std_logic;
    IO0_D11_N                      : inout   std_logic;
    IO0_D12_P                      : inout   std_logic;
    IO0_D13_N                      : inout   std_logic;
    IO0_D14_P                      : inout   std_logic;
    IO0_D15_N                      : inout   std_logic;
    IO0_D16_P                      : inout   std_logic;
    IO0_D17_N                      : inout   std_logic;
    IO0_D18_P                      : inout   std_logic;
    IO0_D19_N                      : inout   std_logic;
    IO0_D20_P                      : inout   std_logic;
    IO0_D21_N                      : inout   std_logic;
    IO0_D22_MIPI_D0LP_P            : inout   std_logic;
    IO0_D23_MIPI_D0LP_N            : inout   std_logic;
    IO0_CLK_N                      : inout   std_logic;
    IO0_CLK_P                      : inout   std_logic;

    -- Anios IO Connector 1
    IO1_D0_P                       : inout   std_logic;
    IO1_D1_N                       : inout   std_logic;
    IO1_D2_P                       : inout   std_logic;
    IO1_D3_N                       : inout   std_logic;
    IO1_D4_P                       : inout   std_logic;
    IO1_D5_N                       : inout   std_logic;
    IO1_D6_P                       : inout   std_logic;
    IO1_D7_N                       : inout   std_logic;
    IO1_D8_P                       : inout   std_logic;
    IO1_D9_N                       : inout   std_logic;
    IO1_D10_P                      : inout   std_logic;
    IO1_D11_N                      : inout   std_logic;
    IO1_D12_P                      : inout   std_logic;
    IO1_D13_N                      : inout   std_logic;
    IO1_D14_P                      : inout   std_logic;
    IO1_D15_N                      : inout   std_logic;
    IO1_D16_P                      : inout   std_logic;
    IO1_D17_N                      : inout   std_logic;
    IO1_D18_P                      : inout   std_logic;
    IO1_D19_N                      : inout   std_logic;
    IO1_D20_P                      : inout   std_logic;
    IO1_D21_N                      : inout   std_logic;
    IO1_D22_P                      : inout   std_logic;
    IO1_D23_N                      : inout   std_logic;
    IO1_CLK_N                      : inout   std_logic;
    IO1_CLK_P                      : inout   std_logic;

    -- 33 MHz user clock
    CLK33                          : in      std_logic; -- Only available on Z7020 modules

    -- Display Port
    DP_HPD                         : in      std_logic;
    DP_AUX_IN                      : in      std_logic;
    DP_AUX_OE                      : out     std_logic;
    DP_AUX_OUT                     : out     std_logic;
    DP_LANE0_N                     : inout   std_logic;
    DP_LANE0_P                     : inout   std_logic;
    DP_LANE1_N                     : inout   std_logic;
    DP_LANE1_P                     : inout   std_logic;

    -- ETH LED
    ETH_LED2_N                     : inout   std_logic; -- Only available on Z7020 modules

    -- HDMI
    HDMI_CEC_WDI                   : inout   std_logic;
    HDMI_HPD                       : in      std_logic;
    HDMI_D0_N                      : out     std_logic;
    HDMI_D0_P                      : out     std_logic;
    HDMI_D1_N                      : out     std_logic;
    HDMI_D1_P                      : out     std_logic;
    HDMI_D2_N                      : out     std_logic;
    HDMI_D2_P                      : out     std_logic;
    HDMI_CLK_N                     : out     std_logic;
    HDMI_CLK_P                     : out     std_logic;

    -- I2C PL
    I2C_MGMT_INT_N                 : in      std_logic; -- Only available on Z7020 modules
    I2C_MGMT_SCL                   : inout   std_logic; -- Only available on Z7020 modules
    I2C_MGMT_SDA                   : inout   std_logic; -- Only available on Z7020 modules

    -- I2C User
    I2C_USER_SCL                   : inout   std_logic;
    I2C_USER_SDA                   : inout   std_logic;

    -- IO Connector 2
    IO2_D0_P                       : inout   std_logic;
    IO2_D1_N                       : inout   std_logic;
    IO2_D2_P                       : inout   std_logic;
    IO2_D3_N                       : inout   std_logic;
    IO2_D4_P                       : inout   std_logic;
    IO2_D5_N                       : inout   std_logic;
    IO2_D6_P                       : inout   std_logic;
    IO2_D7_N                       : inout   std_logic;

    -- IO Connector 3
    IO3_D0_P                       : inout   std_logic;
    IO3_D1_N                       : inout   std_logic;
    IO3_D2_P                       : inout   std_logic;
    IO3_D3_N                       : inout   std_logic;

    -- LED
    LED0_N_PL                      : out     std_logic := 'Z';
    LED1_N_PL                      : out     std_logic := 'Z';
    LED2_N_PL                      : out     std_logic := 'Z';
    LED3_N_PL                      : out     std_logic := 'Z';

    -- MIPI
    MIPI_D0_N                      : inout   std_logic;
    MIPI_D0_P                      : inout   std_logic;
    MIPI_D1_N                      : inout   std_logic;
    MIPI_D1_P                      : inout   std_logic;
    MIPI_CLK_N                     : inout   std_logic;
    MIPI_CLK_P                     : inout   std_logic;

    -- USB3
    USBH_SSRX_N                    : inout   std_logic;
    USBH_SSRX_P                    : inout   std_logic;
    USBH_SSTX_N                    : inout   std_logic;
    USBH_SSTX_P                    : inout   std_logic;

    -- User Oscillator
    OSC_N                          : in      std_logic;
    OSC_P                          : in      std_logic
  );
end Mars_ZX2_ST3;

architecture rtl of Mars_ZX2_ST3 is
  -- differential output buffer primitive
  component OBUFDS is
    port (
      I  : in  std_logic;
      O  : out std_logic;
      OB : out std_logic
    );
  end component OBUFDS;

  -- system clock generator
  component PLL is
    port (
      reset  : in  std_logic;
      locked : out std_logic;
      clk33  : in  std_logic;
      clk100 : out std_logic;
      clk50  : out std_logic;
      clk50d : out std_logic
    );
  end component PLL;

  -- signal declarations#
  signal Clk50         : std_logic;
  signal Clk100        : std_logic;
  signal pll_locked    : std_logic;
  signal plkLedStatus  : std_logic;
  signal plkLedError   : std_logic;
  signal psCdi2UartTxd : std_logic;
  signal psCdi2UartRxd : std_logic;
  signal cdi2UartTxd   : std_logic;
  signal cdi2UartRxd   : std_logic;
  signal psPppUartTxd  : std_logic;
  signal psPppUartRxd  : std_logic;
  signal pppUartTxd    : std_logic;
  signal pppUartRxd    : std_logic;
  signal Rst_N         : std_logic;
  signal LED_N         : std_logic_vector(3 downto 0);
  signal LedCount      : unsigned(26 downto 0);
begin
  -- set (unused) DP outputs
  DP_AUX_OE  <= 'Z';
  DP_AUX_OUT <= 'Z';
  DP_LANE0_N <= 'Z';
  DP_LANE0_P <= 'Z';
  DP_LANE1_N <= 'Z';
  DP_LANE1_P <= 'Z';

  -- CDI2 and PPP UART TXD muxing
  psCdi2UartRxd <= cdi2UartTxd;
  IO1_D11_N     <= cdi2UartTxd;
  cdi2UartRxd   <= psCdi2UartTxd and IO1_D8_P;

  psPppUartRxd <= pppUartTxd;
  IO1_D9_N     <= pppUartTxd;
  pppUartRxd   <= psPppUartTxd and IO1_D6_P;

  -- system clock generator
  clkGen : component PLL
    port map (
      reset  => '0',
      locked => pll_locked,
      clk33  => Clk33,
      clk100 => Clk100,
      clk50  => Clk50,
      clk50d => IO1_D12_P
    );

  -- processor system instance
  ps : entity work.Mars_ZX2_wrapper(STRUCTURE)
    port map (
      Clk50                => open,
      Clk100               => open,
      Rst_N                => Rst_N,

      FIXED_IO_mio         => FIXED_IO_mio,
      FIXED_IO_ddr_vrn     => FIXED_IO_ddr_vrn,
      FIXED_IO_ddr_vrp     => FIXED_IO_ddr_vrp,
      FIXED_IO_ps_srstb    => FIXED_IO_ps_srstb,
      FIXED_IO_ps_clk      => FIXED_IO_ps_clk,
      FIXED_IO_ps_porb     => FIXED_IO_ps_porb,

      DDR_cas_n            => DDR_cas_n,
      DDR_cke              => DDR_cke,
      DDR_ck_n             => DDR_ck_n,
      DDR_ck_p             => DDR_ck_p,
      DDR_cs_n             => DDR_cs_n,
      DDR_reset_n          => DDR_reset_n,
      DDR_odt              => DDR_odt,
      DDR_ras_n            => DDR_ras_n,
      DDR_we_n             => DDR_we_n,
      DDR_ba               => DDR_ba,
      DDR_addr             => DDR_addr,
      DDR_dm               => DDR_dm,
      DDR_dq               => DDR_dq,
      DDR_dqs_n            => DDR_dqs_n,
      DDR_dqs_p            => DDR_dqs_p,

      IIC_USER_sda_io      => I2C_USER_SDA,
      IIC_USER_scl_io      => I2C_USER_SCL,

      LED_N                => LED_N,

      CDI2_UART_baudoutn   => open,
      CDI2_UART_ctsn       => '1',
      CDI2_UART_dcdn       => '1',
      CDI2_UART_ddis       => open,
      CDI2_UART_dsrn       => '1',
      CDI2_UART_dtrn       => open,
      CDI2_UART_out1n      => open,
      CDI2_UART_out2n      => open,
      CDI2_UART_ri         => '0',
      CDI2_UART_rtsn       => open,
      CDI2_UART_rxd        => psCdi2UartRxd,
      CDI2_UART_rxrdyn     => open,
      CDI2_UART_txd        => psCdi2UartTxd,
      CDI2_UART_txrdyn     => open,

      PPP_UART_baudoutn    => open,
      PPP_UART_ctsn        => '1',
      PPP_UART_dcdn        => '1',
      PPP_UART_ddis        => open,
      PPP_UART_dsrn        => '1',
      PPP_UART_dtrn        => open,
      PPP_UART_out1n       => open,
      PPP_UART_out2n       => open,
      PPP_UART_ri          => '0',
      PPP_UART_rtsn        => open,
      PPP_UART_rxd         => psPppUartRxd,
      PPP_UART_rxrdyn      => open,
      PPP_UART_txd         => psPppUartTxd,
      PPP_UART_txrdyn      => open
    );

  -- CDI2 Device Interface Core
  cdi2Device : entity work.CDI2_Device_wrapper(STRUCTURE)
    port map (
      clk100              => Clk100,
      clk50               => Clk50,
      dcm_locked          => pll_locked,
      reset_in            => '1',

      RMII_crs_dv(0)      => IO1_D16_P,
      RMII_rx_er(0)       => IO1_D19_N,
      RMII_rxd(0)         => IO1_D21_N,
      RMII_rxd(1)         => IO1_D18_P,
      RMII_tx_en(0)       => IO1_D22_P,
      RMII_txd(0)         => IO1_D23_N,
      RMII_txd(1)         => IO1_D20_P,
      RMII_clk_o          => open,

      SMII_PHY_reset_on   => open,
      SMI_mdc(0)          => IO1_D14_P,
      SMI_mdio_io(0)      => IO1_D15_N,

      TTS_bp              => IO1_D5_N,
      TTS_ready           => IO1_D3_N,
      TTS_reset           => IO1_D0_P,
      TTS_tc              => IO1_D2_P,

      cdi2_uart_baudoutn  => open,
      cdi2_uart_ctsn      => '1',
      cdi2_uart_dcdn      => '1',
      cdi2_uart_ddis      => open,
      cdi2_uart_dsrn      => '1',
      cdi2_uart_dtrn      => open,
      cdi2_uart_out1n     => open,
      cdi2_uart_out2n     => open,
      cdi2_uart_ri        => '0',
      cdi2_uart_rtsn      => open,
      cdi2_uart_rxd       => cdi2uartRxd,
      cdi2_uart_rxrdyn    => open,
      cdi2_uart_txd       => cdi2UartTxd,
      cdi2_uart_txrdyn    => open,

      ppp_uart_baudoutn   => open,
      ppp_uart_ctsn       => '1',
      ppp_uart_dcdn       => '1',
      ppp_uart_ddis       => open,
      ppp_uart_dsrn       => '1',
      ppp_uart_dtrn       => open,
      ppp_uart_out1n      => open,
      ppp_uart_out2n      => open,
      ppp_uart_ri         => '0',
      ppp_uart_rtsn       => open,
      ppp_uart_rxd        => pppUartRxd,
      ppp_uart_rxrdyn     => open,
      ppp_uart_txd        => pppUartTxd,
      ppp_uart_txrdyn     => open,

      status_led_tri_o(0) => plkLedStatus,
      status_led_tri_o(1) => plkLedError
    );

  -- ST3 PHY board implements D3 (green) and D4 (red) with negative (pull down) logic
  IO1_D7_N <= not plkLedStatus;
  IO1_D4_P <= not plkLedError;

  hdmi_clock_buf : component OBUFDS
    port map (
      I  => '0',
      O  => HDMI_CLK_P,
      OB => HDMI_CLK_N
    );

  hdmi_d0_buf : component OBUFDS
    port map (
      I  => '0',
      O  => HDMI_D0_P,
      OB => HDMI_D0_N
    );

  hdmi_d1_buf : component OBUFDS
    port map (
      I  => '0',
      O  => HDMI_D1_P,
      OB => HDMI_D1_N
    );

  hdmi_d2_buf : component OBUFDS
    port map (
      I  => '0',
      O  => HDMI_D2_P,
      OB => HDMI_D2_N
    );

  process (Clk33)
  begin
    if rising_edge (Clk33) then
      LedCount <= LedCount + 1;
    end if;
  end process;

  LED0_N_PL <= '0' when LedCount(LedCount'high) = '1' else 'Z';
  LED1_N_PL <= 'Z' when LedCount(LedCount'high) = '1' else '0';
end rtl;
