/**
********************************************************************************
\file   xilinx-microblaze/target-microblaze.c

\brief  Target specific functions for Microblaze without OS

This target depending module provides several functions that are necessary for
systems without OS and not using shared buffer library.

\ingroup module_target
*******************************************************************************/

/*------------------------------------------------------------------------------
Copyright (c) 2016, B&R Industrial Automation GmbH
Copyright (c) 2017, Kalycito Infotech Private Limited
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the copyright holders nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
------------------------------------------------------------------------------*/

//------------------------------------------------------------------------------
// includes
//------------------------------------------------------------------------------
#define _COMPILING_NEWLIB
#include <sys/time.h>

#include <common/oplkinc.h>
#include <common/target.h>

#include <sleep.h>
#include <xparameters.h>
#include <xtime_l.h>
#include <xgpio_l.h>

//============================================================================//
//            G L O B A L   D E F I N I T I O N S                             //
//============================================================================//

//------------------------------------------------------------------------------
// const defines
//------------------------------------------------------------------------------
#define TGTCONIO_MS_IN_US(x)    (x * 1000U)

//------------------------------------------------------------------------------
// module global vars
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// global function prototypes
//------------------------------------------------------------------------------

//============================================================================//
//            P R I V A T E   D E F I N I T I O N S                           //
//============================================================================//

//------------------------------------------------------------------------------
// const defines
//------------------------------------------------------------------------------
#define GPIO_STATUS_LED_BIT         3
#define GPIO_ERROR_LED_BIT          4

#ifdef XPAR_AXI_GPIO_0_BASEADDR
#define TARGET_POWERLINK_LED_BASE   XPAR_AXI_GPIO_0_BASEADDR
#endif

//------------------------------------------------------------------------------
// local types
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// local vars
//------------------------------------------------------------------------------
static UINT32   plkStatusErrorLeds_l;       ///< Local copy of the state of the POWERLINK status LEDs
static BOOL     fInterruptContextFlag_l;

//------------------------------------------------------------------------------
// local function prototypes
//------------------------------------------------------------------------------
static void enableInterruptMaster(void);
static void disableInterruptMaster(void);

static void setStatusLed(BOOL fOn_p);
static void setErrorLed(BOOL fOn_p);

//============================================================================//
//            P U B L I C   F U N C T I O N S                                 //
//============================================================================//

//------------------------------------------------------------------------------
/**
\brief    Get current system tick

This function returns the current system tick determined by the system timer.

\return Returns the system tick in milliseconds

\ingroup module_target
*/
//------------------------------------------------------------------------------
UINT32 target_getTickCount(void)
{
    XTime ticks;

    XTime_GetTime(&ticks);

    ticks /= (COUNTS_PER_SECOND / 1000);

    /* unsigned integral conversion takes lower bits */
    return (UINT32)ticks;
}

//------------------------------------------------------------------------------
/**
\brief  Get current timestamp

The function returns the current timestamp in nanoseconds.

\return The function returns the timestamp in nanoseconds
*/
//------------------------------------------------------------------------------
ULONGLONG target_getCurrentTimestamp(void)
{
    XTime ticks;

    XTime_GetTime(&ticks);

    /* on Zynq Ultrascale, the timestamp has 30ns resolution (33.333MHz clock) */
    return (ULONGLONG)ticks;
}

//------------------------------------------------------------------------------
/**
\brief    enables global interrupt

This function enables/disables global interrupts.

\param[in]      fEnable_p           TRUE = enable interrupts
                                    FALSE = disable interrupts

\ingroup module_target
*/
//------------------------------------------------------------------------------
void target_enableGlobalInterrupt(BOOL fEnable_p)
{
    static INT lockCount = 0;

    if (fEnable_p != FALSE) // restore interrupts
    {
        if (--lockCount == 0)
            enableInterruptMaster();
    }
    else
    {                       // disable interrupts
        if (lockCount == 0)
            disableInterruptMaster();

        lockCount++;
    }
}

//------------------------------------------------------------------------------
/**
\brief    Set interrupt context flag

This function enables/disables the interrupt context flag. The flag has to be
set when the CPU enters the interrupt context. The flag has to be cleared when
the interrupt context is left.

\param[in]      fEnable_p           TRUE = enable interrupt context flag
                                    FALSE = disable interrupt context flag

\ingroup module_target
*/
//------------------------------------------------------------------------------
void target_setInterruptContextFlag(BOOL fEnable_p)
{
    fInterruptContextFlag_l = fEnable_p;
}

//------------------------------------------------------------------------------
/**
\brief    Get interrupt context flag

This function returns the interrupt context flag.

\return The function returns the state of the interrupt context flag.

\ingroup module_target
*/
//------------------------------------------------------------------------------
BOOL target_getInterruptContextFlag(void)
{
    return fInterruptContextFlag_l;
}

//------------------------------------------------------------------------------
/**
\brief  Initialize target specific stuff

The function initialize target specific stuff which is needed to run the
openPOWERLINK stack.

\return The function returns a tOplkError error code.
*/
//------------------------------------------------------------------------------
tOplkError target_init(void)
{
    fInterruptContextFlag_l = FALSE;

    // enable the interrupt master
    enableInterruptMaster();

    plkStatusErrorLeds_l = 0;

    return kErrorOk;
}

//------------------------------------------------------------------------------
/**
\brief  Clean up target specific stuff

The function cleans up target specific stuff.

\return The function returns a tOplkError error code.
*/
//------------------------------------------------------------------------------
tOplkError target_cleanup(void)
{
    // disable the interrupt master
    disableInterruptMaster();

    fInterruptContextFlag_l = FALSE;

    return kErrorOk;
}

//------------------------------------------------------------------------------
/**
\brief Sleep for the specified number of milliseconds

The function makes the calling thread sleep until the number of specified
milliseconds has elapsed.

\param[in]      milliSeconds_p      Number of milliseconds to sleep

\ingroup module_target
*/
//------------------------------------------------------------------------------
void target_msleep(UINT32 milliSeconds_p)
{
    usleep_A53(TGTCONIO_MS_IN_US(milliSeconds_p));
}

//------------------------------------------------------------------------------
/**
\brief  Set IP address of specified Ethernet interface

The function sets the IP address, subnetMask and MTU of an Ethernet
interface.

\param[in]      ifName_p            Name of Ethernet interface.
\param[in]      ipAddress_p         IP address to set for interface.
\param[in]      subnetMask_p        Subnet mask to set for interface.
\param[in]      mtu_p               MTU to set for interface.

\return The function returns a tOplkError error code.

\ingroup module_target
*/
//------------------------------------------------------------------------------
tOplkError target_setIpAdrs(const char* ifName_p,
                            UINT32 ipAddress_p,
                            UINT32 subnetMask_p,
                            UINT16 mtu_p)
{
    UNUSED_PARAMETER(ifName_p);
    UNUSED_PARAMETER(ipAddress_p);
    UNUSED_PARAMETER(subnetMask_p);
    UNUSED_PARAMETER(mtu_p);

    //Note: The given parameters are ignored because the application must set
    //      these settings to the used IP stack by itself!

    return kErrorOk;
}

//------------------------------------------------------------------------------
/**
\brief  Set default gateway for Ethernet interface

The function sets the default gateway of an Ethernet interface.

\param[in]      defaultGateway_p    Default gateway to set.

\return The function returns a tOplkError error code.

\ingroup module_target
*/
//------------------------------------------------------------------------------
tOplkError target_setDefaultGateway(UINT32 defaultGateway_p)
{
    UNUSED_PARAMETER(defaultGateway_p);

    //Note: The given parameters are ignored because the application must set
    //      these settings to the used IP stack by itself!

    return kErrorOk;
}

//------------------------------------------------------------------------------
/**
\brief  Set POWERLINK status/error LED

The function sets the POWERLINK status/error LED.

\param[in]      ledType_p           Determines which LED shall be set/reset.
\param[in]      fLedOn_p            Set the addressed LED on (TRUE) or off (FALSE).

\return The function returns a tOplkError error code.

\ingroup module_target
*/
//------------------------------------------------------------------------------
tOplkError target_setLed(tLedType ledType_p, BOOL fLedOn_p)
{
    tOplkError  ret = kErrorOk;

    switch (ledType_p)
     {
         case kLedTypeStatus:
            setStatusLed(fLedOn_p);
            break;

         case kLedTypeError:
            setErrorLed(fLedOn_p);
            break;

         default:
            ret = kErrorIllegalInstance;
            break;
     }

    return ret;
}

#if (defined(CONFIG_INCLUDE_SOC_TIME_FORWARD) && defined(CONFIG_INCLUDE_NMT_MN))
//------------------------------------------------------------------------------
/**
\brief  Get system time

The function returns the current system timestamp.

\param[out]      pNetTime_p         Pointer to current system timestamp.
\param[out]      pValidSystemTime_p Pointer to flag which is set to indicate
                                    the system time is valid or not.

\return The function returns a tOplkError code.

\ingroup module_target
*/
//------------------------------------------------------------------------------
tOplkError target_getSystemTime(tNetTime* pNetTime_p, BOOL* pValidSystemTime_p)
{
    if ((pNetTime_p == NULL) || (pValidSystemTime_p == NULL))
        return kErrorNoResource;

    /* get current wall clock time - (our implementation of gettimeofday always returns success) */
    struct timeval tv = {0, 0};
    _gettimeofday(&tv, NULL);
    
    pNetTime_p->sec  = tv.tv_sec;
    pNetTime_p->nsec = tv.tv_usec * 1000;

    /* Set the flag to indicate the system time is valid */
    *pValidSystemTime_p = TRUE;
    return kErrorOk;
}
#endif

//============================================================================//
//            P R I V A T E   F U N C T I O N S                               //
//============================================================================//
/// \name Private Functions
/// \{

//------------------------------------------------------------------------------
/**
\brief Enable the global interrupt master

\ingroup module_target
*/
//------------------------------------------------------------------------------
static void enableInterruptMaster(void)
{
    __asm volatile ("msr DAIFClr,#0x02" ::: "cc");
}

//------------------------------------------------------------------------------
/**
\brief Disable the global interrupt master

\ingroup module_target
*/
//------------------------------------------------------------------------------
static void disableInterruptMaster(void)
{
    __asm volatile ("msr DAIFSet,#0x02" ::: "cc");
}

//------------------------------------------------------------------------------
/**
\brief  Sets the status LED

The function sets the POWERLINK status LED.

\param[in]      fOn_p               Determines the LED state.

\ingroup module_app_common
*/
//------------------------------------------------------------------------------
static void setStatusLed(BOOL fOn_p)
{
#if defined(TARGET_POWERLINK_LED_BASE)
    plkStatusErrorLeds_l = XGpio_ReadReg(TARGET_POWERLINK_LED_BASE, 0);
#endif

    if (fOn_p)
        plkStatusErrorLeds_l |= (1 << GPIO_STATUS_LED_BIT);
    else
        plkStatusErrorLeds_l &= ~(1 << GPIO_STATUS_LED_BIT);

#if defined(TARGET_POWERLINK_LED_BASE)
    XGpio_WriteReg(TARGET_POWERLINK_LED_BASE, 0, plkStatusErrorLeds_l);
#else
    UNUSED_PARAMETER(fOn_p);
#endif
}

//------------------------------------------------------------------------------
/**
\brief  Sets the error LED

The function sets the POWERLINK error LED.

\param[in]      fOn_p               Determines the LED state.

\ingroup module_app_common
*/
//------------------------------------------------------------------------------
static void setErrorLed(BOOL fOn_p)
{
#if defined(TARGET_POWERLINK_LED_BASE)
    plkStatusErrorLeds_l = XGpio_ReadReg(TARGET_POWERLINK_LED_BASE, 0);
#endif

    if (fOn_p)
        plkStatusErrorLeds_l |= (1 << GPIO_ERROR_LED_BIT);
    else
        plkStatusErrorLeds_l &= ~(1 << GPIO_ERROR_LED_BIT);

#if defined(TARGET_POWERLINK_LED_BASE)
    XGpio_WriteReg(TARGET_POWERLINK_LED_BASE, 0, plkStatusErrorLeds_l);
#else
    UNUSED_PARAMETER(fOn_p);
#endif
}

#define IS_ALIGNED(x, a) (((x) & ((__typeof__(x))(a) - 1)) == 0)

//------------------------------------------------------------------------------
/**
\brief  memset for device memory

The memset(3) implementation for aarch64 uses the "DC ZVA" cache instruction
to clear a memory block. Unfortunately, this instruction is not allowed for
device memory and causes an alignment exception.

\param[in] s memory area
\param[in] c fill value
\param[in] n number of bytes to fill with c
\return a pointer to the memory area s.

\ingroup module_app_common
*/
//------------------------------------------------------------------------------
void * __attribute__ ((optimize(0))) ultrascale_memset(void *s, int c, size_t n)
{
    if ((UINTPTR)s > XPAR_DDR_MEM_HIGHADDR) {
        /* from https://elixir.bootlin.com/linux/latest/source/arch/arm64/kernel/io.c#L82 */
        volatile u8  *dst = (u8 *)s;
        u64           qc  = (u8)c;

        qc |= qc << 8;
        qc |= qc << 16;
        qc |= qc << 32;

        while (n && !IS_ALIGNED((UINTPTR)dst, 8)) {
            *dst = c;
            dst++;
            n--;
        }

        while (n >= 8) {
            *((u64 *)dst) = qc;
            dst += 8;
            n   -= 8;
        }

        while (n) {
            *dst = c;
            dst++;
            n--;
        }

        return s;
    } else
        return memset(s, c, n);
}

//------------------------------------------------------------------------------
/**
\brief  memcpy for device memory

ARMv8 does not allow unaligned access to device memory,
http://infocenter.arm.com/help/index.jsp?topic=/com.arm.doc.faqs/ka15414.html

Unfortunately, this is exactly the the type of access used by the OpenPOWERLINK
stack, thus we work around it.

\param[in] dest destination memory area
\param[in] src  source memory area
\param[in] len  number of bytes to copy
\return a pointer to the destination memory area.

\ingroup module_app_common
*/
//------------------------------------------------------------------------------
__attribute__ ((optimize(0))) void *ultrascale_memcpy(void *dest, const void *src, size_t len)
{
    if (((UINTPTR)dest > XPAR_DDR_MEM_HIGHADDR) || (((UINTPTR)src > XPAR_DDR_MEM_HIGHADDR))) {
        volatile u8       *d = dest;
        volatile const u8 *s = src;

        if (IS_ALIGNED((UINTPTR)d, 8) && IS_ALIGNED((UINTPTR)s, 8))
            while (len > 8) {
                *((u64 *)d) = *((u64 *)s);
                d   += 8;
                s   += 8;
                len -= 8;
            }

        if (IS_ALIGNED((UINTPTR)d, 4) && IS_ALIGNED((UINTPTR)s, 4))
            while (len > 4) {
                *((u32 *)d) = *((u32 *)s);
                d   += 4;
                s   += 4;
                len -= 4;
            }

        while (len--) {
            *((u8 *)d) = *((u8 *)s);
            d += 1;
            s += 1;
        }

        return dest;
    } else
        return memcpy(dest, src, len);
}

/// \}
