/**
 * @file freertos-microblaze/target-microblaze.c
 * @brief Target specific functions for FreeRTOS/Microblaze
 *
 * @copy 2023 AIT Austrian Institute of Technology
 * @see xilinx-microblaze/target-microblaze.c
 * @ingroup module_target
*/

#include <limits.h>
#include <stdint.h>
#include <sleep.h>
extern void msleep(unsigned long mseconds);     /* missing declaration in sleep.h */
#include <sys/time.h>

#include <common/oplkinc.h>
#include <common/target.h>

#include <xparameters.h>
#include <xgpio_l.h>
#include <xiltimer.h>
#include <xintc.h>

#include "systemtimer.h"

#define GPIO_STATUS_LED_BIT         0
#define GPIO_ERROR_LED_BIT          1

static UINT32 plkStatusErrorLeds_l;             /**< local shadow copy of PowerLink status LEDs */
static BOOL   fInterruptContextFlag_l;          /**< global interrupt context flag */

static void enableInterruptMaster(void);
static void disableInterruptMaster(void);

static void setStatusLed(BOOL fOn_p);
static void setErrorLed(BOOL fOn_p);

/**
 * @brief Get current system tick
 *
 * @return Returns the system tick in [ms]
 * @ingroup module_target
 */
UINT32
target_getTickCount(void)
{
    return timer_getMsCount();
}

/**
 * @brief Get current timestamp
 *
 * The function returns the current timestamp in nanoseconds.
 *
 * @return current timestamp in [ns]
 * @ingroup module_target
 */
ULONGLONG
target_getCurrentTimestamp(void)
{
    return timer_getTimestamp();
}

/**
 * @brief Enable/Disable global interrupt
 *
 * @param[in] fEnable_p TRUE = enable interrupts
 *                      FALSE = disable interrupts
 *
 * @ingroup module_target
 */
void
target_enableGlobalInterrupt(BOOL fEnable_p)
{
    static int lockCount = 0;

    if (fEnable_p != FALSE) {
        /* restore interrupts */
        assert(lockCount > 0);

        if (--lockCount == 0)
            enableInterruptMaster();
    } else {
        /* disable interrupts */
        assert(lockCount < INT_MAX);

        if (lockCount == 0)
            disableInterruptMaster();

        lockCount++;
    }
}

/**
 * @brief Set interrupt context flag
 *
 * This function enables/disables the interrupt context flag. The flag has to be
 * set when the CPU enters the interrupt context. The flag has to be cleared when
 * the interrupt context is left.
 *
 * @param[in] fEnable_p TRUE = enable interrupt context flag
 *                      FALSE = disable interrupt context flag
 *
 * @ingroup module_target
 */
void
target_setInterruptContextFlag(BOOL fEnable_p)
{
    fInterruptContextFlag_l = fEnable_p;
}

/**
 * @brief Get interrupt context flag
 *
 * This function returns the interrupt context flag.
 *
 * @return The function returns the state of the interrupt context flag.
 *
 * @ingroup module_target
 */
BOOL
target_getInterruptContextFlag(void)
{
    return fInterruptContextFlag_l;
}

/**
 * @brief Initialize OpenPowerLink HAL layer for FreeRTOS/Microblaze
 *
 * @return kErrorOk if everything went well, one of the eOplError codes else
 * @ingroup module_target
 */
tOplkError
target_init(void)
{
    /* reset global interrupt flag */
    fInterruptContextFlag_l = FALSE;

    /* initialize system timer */
    timer_init();

    /* reset shadow state of status LEDs */
    plkStatusErrorLeds_l = 0;

    return kErrorOk;
}

/**
 * @brief Clean OpenPowerLink HAL layer for FreeRTOS/Microblaze
 *
 * @return kErrorOk if everything went well, one of the eOplError codes else
 * @ingroup module_target
 */
tOplkError
target_cleanup(void)
{
    fInterruptContextFlag_l = FALSE;

    return kErrorOk;
}

/**
 * @brief Sleep for the specified number of milliseconds
 * @param[in] milliSeconds_p Number of milliseconds to sleep
 *
 * @ingroup module_target
 */
void
target_msleep(UINT32 milliSeconds_p)
{
    msleep(milliSeconds_p);
}

/**
 * @brief Set IP address parameters of specified Ethernet interface
 *
 * @param[in] ifName_p Name of Ethernet interface (ignored).
 * @param[in] ipAddress_p IP address to set for interface (ignored).
 * @param[in] subnetMask_p Subnet mask to set for interface (ignored).
 * @param[in] mtu_p MTU to set for interface (ignored).
 *
 * @return kErrorOk if everything went well, one of the eOplError codes else
 * @ingroup module_target
 */
tOplkError
target_setIpAdrs(const char *ifName_p,
                 UINT32      ipAddress_p,
                 UINT32      subnetMask_p,
                 UINT16      mtu_p)
{
    UNUSED_PARAMETER(ifName_p);
    UNUSED_PARAMETER(ipAddress_p);
    UNUSED_PARAMETER(subnetMask_p);
    UNUSED_PARAMETER(mtu_p);
    return kErrorOk;
}

/**
 * @brief  Set default gateway for Ethernet interface
 *
 * @param[in] defaultGateway_p  Default gateway to set (ignored)
 *
 * @return kErrorOk if everything went well, one of the eOplError codes else
 * @ingroup module_target
 */
tOplkError
target_setDefaultGateway(UINT32 defaultGateway_p)
{
    UNUSED_PARAMETER(defaultGateway_p);
    return kErrorOk;
}

/**
 * @brief Set/Reset POWERLINK status or error LEDs
 * @param[in] ledType_p Selects which LED shall be set/reset.
 * @param[in] fLedOn_p  the desired status, on (TRUE) or off (FALSE).
 *
 * @return kErrorOk if everything went well, one of the eOplError codes else
 * @ingroup module_target
 */
tOplkError
target_setLed(tLedType ledType_p,
              BOOL     fLedOn_p)
{
    tOplkError ret = kErrorOk;

    switch (ledType_p) {
    case kLedTypeStatus:
        setStatusLed(fLedOn_p);
        break;

    case kLedTypeError:
        setErrorLed(fLedOn_p);
        break;

    default:
        ret = kErrorIllegalInstance;
        break;
    }

    return ret;
}

#if (defined(CONFIG_INCLUDE_SOC_TIME_FORWARD) && defined(CONFIG_INCLUDE_NMT_MN))
/**
 * @brief Get system time
 *
 * @param[out] pNetTime_p Pointer to current system timestamp.
 * @param[out] pValidSystemTime_p Pointer to flag which is set to indicate
 *                                the system time is valid or not.
 *
 * @return kErrorOk if everything went well, one of the eOplError codes else
 * @ingroup module_target
 */
tOplkError
target_getSystemTime(tNetTime *pNetTime_p,
                     BOOL     *pValidSystemTime_p)
{
    if ((pNetTime_p == NULL) || (pValidSystemTime_p == NULL))
        return kErrorNoResource;

    /* TODO: get current wall clock time - (our implementation of gettimeofday always returns success) */

    struct timeval tv = {0, 0};
#if 0
    _gettimeofday(&tv, NULL);
#endif

    pNetTime_p->sec  = tv.tv_sec;
    pNetTime_p->nsec = tv.tv_usec * 1000;

    /* Set the flag to indicate the system time is valid */
    *pValidSystemTime_p = TRUE;
    return kErrorOk;
}
#endif

/**
 * @brief Enable the global interrupt master
 *
 * @ingroup module_target
 */
static void
enableInterruptMaster(void)
{
    /*
     * enable peripheral interrupts by setting the MER (Master Enable Register)
     * in the AXI interrupt controller
     */
    XIntc_MasterEnable(XPAR_INTERRUPTCONTROLLER_BASEADDR);
}

/**
 * @brief Disable the global interrupt master
 *
 * @ingroup module_target
 */
static void
disableInterruptMaster(void)
{
    /*
     * disable peripheral interrupts by clearing the MER (Master Enable Register)
     * in the AXI interrupt controller
     */
    XIntc_MasterDisable(XPAR_INTERRUPTCONTROLLER_BASEADDR);
}

/**
 * @brief Set/Reset the OpenPowerLink status LED
 *
 * @param[in] fOn_p desired OpenPowerLink status LED state
 * @ingroup module_app_common
 */
static void
setStatusLed(BOOL fOn_p)
{
#if defined(XPAR_STATUSLED_BASEADDR)
    if (fOn_p)
        plkStatusErrorLeds_l |= (1 << GPIO_STATUS_LED_BIT);
    else
        plkStatusErrorLeds_l &= ~(1 << GPIO_STATUS_LED_BIT);

    XGpio_WriteReg(XPAR_STATUSLED_BASEADDR, XGPIO_DATA_OFFSET, plkStatusErrorLeds_l);
#else
    UNUSED_PARAMETER(fOn_p);
#endif
}

/**
 * @brief Set/Reset the OpenPowerLink error LED
 *
 * @param[in] fOn_p desired OpenPowerLink error LED state
 * @ingroup module_app_common
 */
static void
setErrorLed(BOOL fOn_p)
{
#if defined(XPAR_STATUSLED_BASEADDR)
    if (fOn_p)
        plkStatusErrorLeds_l |= (1 << GPIO_ERROR_LED_BIT);
    else
        plkStatusErrorLeds_l &= ~(1 << GPIO_ERROR_LED_BIT);

    XGpio_WriteReg(XPAR_STATUSLED_BASEADDR, XGPIO_DATA_OFFSET, plkStatusErrorLeds_l);
#else
    UNUSED_PARAMETER(fOn_p);
#endif
}
