/**
 * @file freertos-microblaze/systemtimer.c
 * @brief Implement system timer by using a periodic millisecond counter
 *
 * @copy 2023 AIT Austrian Institute of Technology
 * @see xilinx-microblaze/systemtimer.h
 * @ingroup module_target
 */

#include <stdint.h>
#include <sleep.h>

#include <xparameters.h>
#include <xiltimer.h>

static uint64_t offNettimeNs = 0;   /**< offset [ns] = SystemClock [ns] - Soc NETTIME [ns] */

/**
 * @brief initialize system timer
 * @ingroup module target
 */
void
timer_init(void)
{
    /*
     * <braindamage>
     * XTime_GetTime() ultimately uses the counter of the AXI sleep timer.
     * However, this timer is properly initialized only after the 1st call
     * to axi_timer.c::XAxiTimer_ModifyInterval() which is called by a
     * non-header-exported function xiltimer.c::XilTimer_Sleep which is
     * the central function implementing the sleep(), msleep(), and usleep()
     * library functions.
     * </braindamage>
     */
    sleep(0);
}

/**
 * @brief Get current system timer tick count
 * @return current timer count
 * @see XPAR_SLEEPTIMER_CLOCK_FREQ_HZ
 * @ingroup module target
 */
uint64_t
timer_getCount(void)
{
    /* get running system clock [ticks] */
    uint64_t sysClockTicks = 0;
    XTime_GetTime(&sysClockTicks);
    return sysClockTicks;
}

/**
 * @brief Get current timer in milliseconds ticks
 * @return current timer count in [ms] ticks
 * @ingroup module target
 */
unsigned int
timer_getMsCount(void)
{
    return timer_getCount() / (XPAR_SLEEPTIMER_CLOCK_FREQ_HZ / 1000);
}

/**
 * @brief update NETTIME offset
 *
 * This function is called from PowerLink SYNC handler every 1ms in interrupt
 * context. It updates the clock offset of the system sleep timer clock to the
 * SoC NETTIME clock:
 *
 *   offNettimeNs [ns] = SystemClock [ns] - Soc NETTIME [ns]
 *
 * @param[in] tsSoc SoC system time stamp
 * @param[in] sec SoC NETTIME seconds
 * @param[in] nsec SoC NETTIME nano seconds
 *
 * @see oplk_getSocTime()
 * @see powerlink.c:processSyncEvent()
 * @ingroup module target
 */
void
timer_updateNettime(uint64_t tsSoc, uint32_t sec, uint32_t nsec)
{
    /* convert system Soc time stamp [ticks] to [ns] */
    uint64_t sysClockNs = tsSoc * (1000000000 / XPAR_SLEEPTIMER_CLOCK_FREQ_HZ);

    /* calculate offset */
    offNettimeNs  = sysClockNs - (uint64_t)sec * 1000000000;
    offNettimeNs -= nsec;
}

/**
 * @brief Get current NETTIME (EPSG DS 301 V1.3.0, §6.1.6.7) time stamp
 *
 * The function returns the current time stamp in nanoseconds since epoch
 * (1970-01-01T00:00:00 UTC).
 *
 * @return current time stamp in [ns]
 * @ingroup module_target
 */
uint64_t
timer_getTimestamp(void)
{
    /* convert [ticks] to [ns] */
    uint64_t sysClockNs = timer_getCount() * (1000000000 / XPAR_SLEEPTIMER_CLOCK_FREQ_HZ);

    /* nettime [ns] = sysClock [ns] - offset [ns] */
    return sysClockNs - offNettimeNs;
}
