/**
 * @file freertos-microblaze/openmac-microblaze.c
 * @brief openMAC HAL layer
 *
 * @copy 2023 AIT Austrian Institute of Technology
 * @see xilinx-microblaze/openmac-microblaze.c
 * @ingroup module_openmac
*/

#include <common/oplkinc.h>
#include <target/openmac.h>

#include <xparameters.h>
#include <xinterrupt_wrap.h>
#include <xio.h>

/** DLLK PowerLink frame RX/TX interrupt handler */
static tOpenmacIrqCb pRxTxCallback = NULL;

/**
 * wrap the DLLK frame RX/TX interrupt handler into a
 * function that polls the CDI2 and PPP UARTs on every
 *
 * @param[in] pArg original argument passed to openmac_isrReg()
 */
static void
rxtxInterruptWrapper(void *pArg)
{
    extern void cdi2IfPollUartIsr(void);
    extern void pppIfPollUartIsr(void);

    /* poll UART RX FIFOs */
    cdi2IfPollUartIsr();
    pppIfPollUartIsr();

    if (pRxTxCallback != NULL)
        pRxTxCallback(pArg);
}


/**
 * @brief Register interrupt callbacks for openMAC
 *
 * @param[in] irqSource_p Specified interrupt source (MAC or Sync)
 * @param[in] pfnIsrCb_p  Interrupt service routine callback
 * @param[in] pArg_p Argument given to the callback
 *
 * @return kErrorOk if everything went well, one of the eOplError codes else
 * @ingroup module_openmac
 */
tOplkError
openmac_isrReg(tOpenmacIrqSource  irqSource_p,
               tOpenmacIrqCb      pfnIsrCb_p,
               void              *pArg_p)
{
    switch (irqSource_p) {
    case kOpenmacIrqSync:
        if (pfnIsrCb_p != NULL) {
            /* Register interrupt callback */
            XConnectToInterruptCntrl(XPAR_INTERRUPTCONTROLLER_OPENMAC_TIMER_IRQ_INTR,
                                     (XInterruptHandler)pfnIsrCb_p,
                                     (void *)pArg_p,
                                     XPAR_TICKTIMER_INTR_PARENT);
            XEnableIntrId(XPAR_INTERRUPTCONTROLLER_OPENMAC_TIMER_IRQ_INTR,
                          XPAR_TICKTIMER_INTR_PARENT);
        } else {
            /* Unregister interrupt callback */
            XDisableIntrId(XPAR_INTERRUPTCONTROLLER_OPENMAC_TIMER_IRQ_INTR,
                           XPAR_TICKTIMER_INTR_PARENT);
            XDisconnectInterruptCntrl(XPAR_INTERRUPTCONTROLLER_OPENMAC_TIMER_IRQ_INTR,
                                      XPAR_TICKTIMER_INTR_PARENT);
        }

        break;

    case kOpenmacIrqTxRx:
        if (pfnIsrCb_p != NULL) {
            /* Register interrupt callback */
            pRxTxCallback = pfnIsrCb_p;
            XConnectToInterruptCntrl(XPAR_INTERRUPTCONTROLLER_OPENMAC_MAC_IRQ_INTR,
                                     (XInterruptHandler)rxtxInterruptWrapper,
                                     (void *)pArg_p,
                                     XPAR_TICKTIMER_INTR_PARENT);
            XEnableIntrId(XPAR_INTERRUPTCONTROLLER_OPENMAC_MAC_IRQ_INTR,
                          XPAR_TICKTIMER_INTR_PARENT);
        } else {
            /* Unregister interrupt callback */
            XDisableIntrId(XPAR_INTERRUPTCONTROLLER_OPENMAC_MAC_IRQ_INTR,
                           XPAR_TICKTIMER_INTR_PARENT);
            XDisconnectInterruptCntrl(XPAR_INTERRUPTCONTROLLER_OPENMAC_MAC_IRQ_INTR,
                                      XPAR_TICKTIMER_INTR_PARENT);
        }

        break;

    default:
        return kErrorNoResource;
    }

    return kErrorOk;
}

/**
 * @brief Allocate uncached memory
 *
 * @param[in] size_p Size of uncached memory to be allocated
 * @return pointer to uncached memory block or NULL
 *
 * @ingroup module_openmac
 */
void *
openmac_uncachedMalloc(size_t size_p)
{
    return malloc(size_p);
}

/**
 * @brief  Free uncached memory
 *
 * @\param[in] pMem_p Uncached memory to be freed
 * @ingroup module_openmac
 */
void
openmac_uncachedFree(void* pMem_p)
{
    free(pMem_p);
}
