/**
********************************************************************************
\file   oplk/targetdefs/ultrascale.h

\brief  Target specific definitions for Xilinx Zynq Ultrascale+ ARM core systems

This file contains target specific definitions for ARM Cortex-A53 systems.

\copyright &copy;2018 AIT Austrian Institute of Technology
*******************************************************************************/
#ifndef _INC_oplk_targetdefs_ultrascale_H_
#define _INC_oplk_targetdefs_ultrascale_H_

//------------------------------------------------------------------------------
// includes
//------------------------------------------------------------------------------
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stdint.h>

#include <oplk/basictypes.h>

#include <xil_types.h>
#include <xil_io.h>
#include <xil_cache.h>
#include <metal/atomic.h>

//------------------------------------------------------------------------------
// const defines
//------------------------------------------------------------------------------
#define OPLKDLLEXPORT

#define INLINE                                  inline

#define OPLK_FILE_HANDLE                        int

#define UNUSED_PARAMETER(par)                   (void)par

#ifndef NDEBUG
#define PRINTF(...)                             printf(__VA_ARGS__)
#else /* NDEBUG */
#define PRINTF(...)
#endif

// Target IO functions
// - Write
#define OPLK_IO_WR8(addr, val)      Xil_Out8(addr, val)
#define OPLK_IO_WR16(addr, val)     Xil_Out16(addr, val)
#define OPLK_IO_WR32(addr, val)     Xil_Out32(addr, val)

// - Read
#define OPLK_IO_RD8(addr)           Xil_In8(addr)
#define OPLK_IO_RD16(addr)          Xil_In16(addr)
#define OPLK_IO_RD32(addr)          Xil_In32(addr)

// Target memory barrier function
#define OPLK_MEMBAR()               __asm volatile ("dmb sy")

void *ultrascale_memset(void *s, int c, size_t n);
void *ultrascale_memcpy (void *dest, const void *src, size_t len);

#define OPLK_MEMSET(dst, val, siz)  ultrascale_memset((dst), (val), (siz))
#define OPLK_MEMCPY(dst, src, siz)  ultrascale_memcpy((dst), (src), (siz))

// Target data cache functions
#define OPLK_DCACHE_FLUSH(addr, len)        Xil_DCacheFlushRange((INTPTR)(addr), (INTPTR)(len))
#define OPLK_DCACHE_INVALIDATE(addr, len)   Xil_DCacheInvalidateRange((INTPTR)(addr), (INTPTR)(len))

// Target atomic instructions use Xilinx's libmetal, which, in turn, are build upon the __sync* intrinsics of GCC, which, in turn, are build upon the __atomic* instrinsics of GCC
#define OPLK_MUTEX_T                UINT8
#define OPLK_ATOMIC_T               UINT8
#define OPLK_LOCK_T                 UINT8
#define OPLK_ATOMIC_INIT(base)      atomic_init(&base->lock, 0)
#define OPLK_ATOMIC_EXCHANGE(address, newval, oldval) do { oldval = atomic_exchange(address, newval); } while (0)

#endif /* _INC_oplk_targetdefs_ultrascale_H_ */
