/**
********************************************************************************
\file   c5niosld.c

\brief  Linux kernel driver Nios loader C5arm/FPGA

This module collects the functions necessary to reset, start, and load
the Nios2 processor acting as PCP (Protocol Control Processor) in the
openPOWERLINK stack

\ingroup module_driver_linux_kernel_c5arm
*******************************************************************************/

/*------------------------------------------------------------------------------
Copyright (c) 2017, AIT
------------------------------------------------------------------------------*/

//------------------------------------------------------------------------------
// includes
//------------------------------------------------------------------------------

#include <linux/delay.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/io.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/errno.h>
#include <uapi/asm-generic/fcntl.h>
#include <uapi/linux/elf.h>

#include <common/driver.h>
#include "c5niosld.h"

//============================================================================//
//            G L O B A L   D E F I N I T I O N S                             //
//============================================================================//

//------------------------------------------------------------------------------
// const defines
//------------------------------------------------------------------------------
#if (LINUX_VERSION_CODE < KERNEL_VERSION(4, 4, 0))
#error "Linux Kernel versions older 4.4.0 are not supported by this driver!"
#endif

//------------------------------------------------------------------------------
// module global vars
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// global function prototypes
//------------------------------------------------------------------------------

//============================================================================//
//            P R I V A T E   D E F I N I T I O N S                           //
//============================================================================//

//------------------------------------------------------------------------------
// const defines
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// local types
//------------------------------------------------------------------------------

/**

 \brief Nios2 reserved memory region

 The structure holds the information of a memory region reserved for the Nios2 PCP processor

*/
struct MemRegion {
    void   *phyBase;    ///< physical base address (from DT)
    void   *pBase;      ///< I/O mapped base address
    size_t  size;       ///< length in bytes
};

//------------------------------------------------------------------------------
// local function prototypes
//------------------------------------------------------------------------------

static void                     *getRstmgrBase(void);
static struct MemRegion         *getPcpHoles(void);
static void                      releasePcpHoles(void);
inline static struct MemRegion  *getNiosText(void)  { return getPcpHoles() + 0; };
inline static struct MemRegion  *getNiosEntry(void) { return getPcpHoles() + 1; };

//------------------------------------------------------------------------------
// local vars
//------------------------------------------------------------------------------

//============================================================================//
//            P U B L I C   F U N C T I O N S                                 //
//============================================================================//

//------------------------------------------------------------------------------
/**
\brief  openPOWERLINK PCP reset

This function resets the PCP. The PCP stays in reset until c5niosld_start releases
the reset.

\return The function returns a tOplkError error code.

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
tOplkError c5niosld_reset(void)
{
    void *pRstmgrBase = getRstmgrBase();

    if (pRstmgrBase != NULL)
    {
        uint32_t miscmodrst = ioread32(pRstmgrBase + 0x20);

        DEBUG_LVL_DRVINTF_TRACE("%s(): reset PCP\n", __func__);

        miscmodrst |= 0x0080;
        iowrite32(miscmodrst, pRstmgrBase + 0x20);
        iounmap(pRstmgrBase);
        return kErrorOk;
    } else {
        return kErrorNoResource;
    }
}

//------------------------------------------------------------------------------
/**
\brief  openPOWERLINK PCP start

This function starts the PCP.

\return The function returns a tOplkError error code.

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
tOplkError c5niosld_start(void)
{
    void *pRstmgrBase = getRstmgrBase();

    if (pRstmgrBase != NULL)
    {
        uint32_t miscmodrst = ioread32(pRstmgrBase + 0x20);

        DEBUG_LVL_DRVINTF_TRACE("%s(): start PCP\n", __func__);

        miscmodrst |= 0x0080;
        iowrite32(miscmodrst, pRstmgrBase + 0x20);
        ndelay(100);
        miscmodrst &= ~0x0080;
        iowrite32(miscmodrst, pRstmgrBase + 0x20);

        {
            volatile void *gpio1 = ioremap(0xFF709000, 0x0100);
            int            i;

            pr_notice("%s: resetting DP83822...\n", __func__);
            iowrite16(0x0040, gpio1 + 0x00000004);
            iowrite16(0x0000, gpio1);
            udelay(200);
            iowrite16(0x0040, gpio1);

            for (i = 0; i < 200; i++)
                udelay(1000);

            iounmap(gpio1);
        }

        /* wait for Nios2 to come out of reset */
        mdelay(1000);

        iounmap(pRstmgrBase);
        return kErrorOk;
    } else {
        return kErrorNoResource;
    }
}

//------------------------------------------------------------------------------
/**
\brief  openPOWERLINK PCP load

This function loads the PCP SW from a specfied file

\param[in]      elfFileName            name of PCP software file (ELF)

\return The function returns a tOplkError error code.

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
tOplkError c5niosld_load(const char *elfFileName)
{
    INT               result    = 0;
    struct MemRegion *mrText    = getNiosText();
    struct MemRegion *mrEntry   = getNiosEntry();
    struct file      *fp        = NULL;
    void             *vEntryEnd = NULL;
    int               i;
    Elf32_Ehdr        elfHdr;

    /* sanity */
    if ((elfFileName == NULL) ||
        (mrText == NULL)  ||
        (mrEntry == NULL)) {
        releasePcpHoles();
        return kErrorInvalidInstanceParam;
    }

    fp = filp_open(elfFileName, O_RDONLY, 0);

    if (IS_ERR(fp)) {
        DEBUG_LVL_ERROR_TRACE("%s(): cannot open %s\n", __func__, elfFileName);
        releasePcpHoles();
        return kErrorNoResource;
    }

    if (kernel_read(fp, 0, (char *)&elfHdr, sizeof(Elf32_Ehdr)) != sizeof(Elf32_Ehdr)) {
        DEBUG_LVL_ERROR_TRACE("%s(): cannot read full ELF32 header of %s\n", __func__, elfFileName);
        result = kErrorNoResource;
        goto ErrorExit;
    }

    /* ELF header sanity */
    if (strncmp((char *)elfHdr.e_ident, ELFMAG, 4) != 0) {
        DEBUG_LVL_ERROR_TRACE("%s(): %s invalid ELF32 magic\n", __func__, elfFileName);
        result = kErrorNoResource;
        goto ErrorExit;
    }

    if (elfHdr.e_ident[EI_CLASS] != ELFCLASS32) {
        DEBUG_LVL_ERROR_TRACE("%s(): %s not a 32-bit executable\n", __func__, elfFileName);
        result = kErrorNoResource;
        goto ErrorExit;
    }

    if (elfHdr.e_ident[EI_DATA] != ELFDATA2LSB) {
        DEBUG_LVL_ERROR_TRACE("%s(): %s not little-endian\n", __func__, elfFileName);
        result = kErrorNoResource;
        goto ErrorExit;
    }

    if (elfHdr.e_ident[EI_VERSION] != EV_CURRENT) {
        DEBUG_LVL_ERROR_TRACE("%s(): %s unknown ELF header version\n", __func__, elfFileName);
        result = kErrorNoResource;
        goto ErrorExit;
    }

    if (elfHdr.e_ident[EI_OSABI] != ELFOSABI_NONE) {
        DEBUG_LVL_ERROR_TRACE("%s(): %s unknown ELF OS ABI\n", __func__, elfFileName);
        result = kErrorNoResource;
        goto ErrorExit;
    }

    if (elfHdr.e_type != ET_EXEC) {
        DEBUG_LVL_ERROR_TRACE("%s(): %s not an executable\n", __func__, elfFileName);
        result = kErrorNoResource;
        goto ErrorExit;
    }

    if (elfHdr.e_machine != 113) {
        DEBUG_LVL_ERROR_TRACE("%s(): %s not an Nios2 executable\n", __func__, elfFileName);
        result = kErrorNoResource;
        goto ErrorExit;
    }

    if (elfHdr.e_phentsize != sizeof(Elf32_Phdr)) {
        DEBUG_LVL_ERROR_TRACE("%s(): %s implausible program header size\n", __func__, elfFileName);
        result = kErrorNoResource;
        goto ErrorExit;
    }

    DEBUG_LVL_ALWAYS_TRACE("loading PCP/Nios2 with ELF file %s\n", __func__, elfFileName);

    for (i = 0; i < elfHdr.e_phnum; i++) {
        Elf32_Phdr prgHdr;

        if (kernel_read(fp, elfHdr.e_phoff + i * sizeof(Elf32_Phdr), (char *)&prgHdr, sizeof(Elf32_Phdr)) != sizeof(Elf32_Phdr)) {
            DEBUG_LVL_ERROR_TRACE("%s(): cannot read ELF32 program header of %s\n", __func__, elfFileName);
            result = kErrorNoResource;
            goto ErrorExit;
        }

        DEBUG_LVL_DRVINTF_TRACE("%s(): ph[%d] type %d, offset %u, va %08x, pa %08x, filesize %u, memsize %u\n",
                                __func__, i, prgHdr.p_type, prgHdr.p_offset, prgHdr.p_vaddr, prgHdr.p_paddr, prgHdr.p_filesz, prgHdr.p_memsz);

        if (prgHdr.p_type != PT_LOAD)
            continue;

        /* sanity */
        if (prgHdr.p_filesz > prgHdr.p_memsz) {
            DEBUG_LVL_ERROR_TRACE("%s(): %s initialization of segment %d > memory\n", __func__, elfFileName, i);
            result = kErrorNoResource;
            goto ErrorExit;
        }

        /* Hack for on-chip memory .tc_mem */
#define TCMEMBASE 0x08000000U
        /* Heuristic: everything below 4M is .text */
#define TEXTLIMIT 0x00400000U
        if (prgHdr.p_vaddr < TEXTLIMIT) {
            /* sanity */
            if (prgHdr.p_vaddr + prgHdr.p_memsz >= mrText->size) {
                DEBUG_LVL_ERROR_TRACE("%s(): %s ELF32 .text segment %d out of bounds\n", __func__, elfFileName, i);
                result = kErrorNoResource;
                goto ErrorExit;
            }

            /* clear memory, then load init values. */
            memset(mrText->pBase + prgHdr.p_vaddr, 0, prgHdr.p_memsz);

            if (prgHdr.p_filesz != 0) {
                if (kernel_read(fp, prgHdr.p_offset, mrText->pBase + prgHdr.p_vaddr, prgHdr.p_filesz) != prgHdr.p_filesz) {
                    DEBUG_LVL_ERROR_TRACE("%s(): cannot read ELF32 .text segment %d of %s\n", __func__, i, elfFileName);
                    result = kErrorNoResource;
                    goto ErrorExit;
                }
            }

            DEBUG_LVL_DRVINTF_TRACE("%s(): .text segment %d loaded %p/%p\n", __func__, i, mrText->pBase + prgHdr.p_vaddr, (void *)prgHdr.p_vaddr);
        } else if (prgHdr.p_vaddr < TCMEMBASE) {
            /* sanity */
            if ((prgHdr.p_vaddr + prgHdr.p_memsz - TEXTLIMIT) >= mrEntry->size) {
                DEBUG_LVL_ERROR_TRACE("%s(): %s ELF32 .entry segment %d out of bounds\n", __func__, elfFileName, i);
                result = kErrorNoResource;
                goto ErrorExit;
            }

            /* save end of .entry for .tc_mem loader hack below */
            vEntryEnd = mrEntry->pBase +  + prgHdr.p_vaddr - TEXTLIMIT + prgHdr.p_memsz;

            /* clear memory, then load init values. */
            memset(mrEntry->pBase + prgHdr.p_vaddr - TEXTLIMIT, 0, prgHdr.p_memsz);

            if (prgHdr.p_filesz != 0) {
                if (kernel_read(fp, prgHdr.p_offset, mrEntry->pBase + prgHdr.p_vaddr - TEXTLIMIT, prgHdr.p_filesz) != prgHdr.p_filesz) {
                    DEBUG_LVL_ERROR_TRACE("%s(): cannot read ELF32 .entry segment %d of %s\n", __func__, i, elfFileName);
                    result = kErrorNoResource;
                    goto ErrorExit;
                }
            }

            DEBUG_LVL_DRVINTF_TRACE("%s(): .entry segment %d loaded %p/%p\n", __func__, i, mrEntry->pBase + prgHdr.p_vaddr - TEXTLIMIT, (void *)prgHdr.p_vaddr);
        } else {
            if (vEntryEnd == NULL) {
                DEBUG_LVL_ERROR_TRACE("%s(): %s ELF32 .tc_mem segment %d defined, but no .entry segment present. Expect trouble\n", __func__, elfFileName, i);

                /* skip .tc_mem section */
                continue;
            }

            /* write magic */
#define MAGIC 0xdeadbeefU
            *((uint32_t *)vEntryEnd) = MAGIC;

            /* write .tc_mem */
            if (prgHdr.p_filesz != 0) {
                if (kernel_read(fp, prgHdr.p_offset, vEntryEnd + sizeof(uint32_t), prgHdr.p_filesz) != prgHdr.p_filesz) {
                    DEBUG_LVL_ERROR_TRACE("%s(): cannot read ELF32 .tc_mem segment %d of %s\n", __func__, i, elfFileName);
                    result = kErrorNoResource;
                    goto ErrorExit;
                }
            }
        }
    }

    filp_close(fp, fp);
    releasePcpHoles();
    return kErrorOk;

ErrorExit:
    filp_close(fp, fp);
    releasePcpHoles();
    return result;
}

//============================================================================//
//            P R I V A T E   F U N C T I O N S                               //
//============================================================================//
/// \name Private Functions
/// \{

//------------------------------------------------------------------------------
/**
\brief  get I/O base address of HPS reset manager

\return I/O base address or NULL if error

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
static void *getRstmgrBase(void)
{
    void *pRstmgrBase = NULL;

    if (pRstmgrBase == NULL) {
        /* 1st call of this function, search the device tree */
        struct device_node *rstmgr = of_find_compatible_node(NULL, NULL, "altr,rst-mgr");

        /* Cyclone 5 reset manager node present in device tree */
        if (rstmgr != NULL)
        {
            uint32_t reg[2];

            /* get the base address of the reset manager's register file from the node and map it */
            if (of_property_read_u32_array(rstmgr, "reg", reg, 2) == 0)
            {
                pRstmgrBase = (void *)ioremap(reg[0], reg[1]);

                if (pRstmgrBase == NULL)
                {
                    DEBUG_LVL_ERROR_TRACE("%s(): cannot map I/O region of altr,rst-mgr: <0x%08x, 0x%08x>\n", __func__, reg[0], reg[1]);
                }
            } else {
                DEBUG_LVL_ERROR_TRACE("%s(): cannot read reg propery of altr,rst-mgr\n", __func__);
            }

            /* release DT node */
            of_node_put(rstmgr);
        } else {
            DEBUG_LVL_ERROR_TRACE("%s(): cannot find device tree node compatible with \"altr,rst-mgr\"\n", __func__);
        }
    }

    return pRstmgrBase;
}

static struct MemRegion holes[2] = {
    { NULL, NULL, 0 },
    { NULL, NULL, 0 }
};

//------------------------------------------------------------------------------
/**
\brief  map the PCP memory holes for loading

\return I/O base addresses or NULL if error

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
static struct MemRegion *getPcpHoles(void)
{
    if (holes[0].phyBase == NULL) {
        /* 1st call of this function, search the device tree for the .text segment */
        struct device_node *pcpText = of_find_node_by_path("/reserved-memory/hole@20000000");

        if (pcpText != NULL)
        {
            uint32_t reg[2];

            /* get physical base address and length of the .text segment and map it */
            if (of_property_read_u32_array(pcpText, "reg", reg, 2) == 0)
            {
                /* sanity */
                if ((reg[0] == 0) || (reg[1] == 0))
                {
                    DEBUG_LVL_ERROR_TRACE("%s(): implausible I/O region of hole@20000000: <0x%08x, 0x%08x>\n", __func__, reg[0], reg[1]);
                } else {
                    holes[0].phyBase = (void *)reg[0];
                    holes[0].size    = reg[1];
                    holes[0].pBase   = (void *)ioremap(reg[0], reg[1]);
                }

                if (holes[0].pBase == NULL)
                {
                    DEBUG_LVL_ERROR_TRACE("%s(): cannot map I/O region of hole@20000000: <0x%08x, 0x%08x>\n", __func__, reg[0], reg[1]);
                    holes[0].phyBase = 0;
                    holes[0].size    = 0;
                }
            } else {
                DEBUG_LVL_ERROR_TRACE("%s(): cannot read reg propery of hole@20000000\n", __func__);
            }

            /* release DT node */
            of_node_put(pcpText);
        } else {
            DEBUG_LVL_ERROR_TRACE("%s(): cannot find device tree node hole@20000000\n", __func__);
        }
    }

    if (holes[1].phyBase == NULL) {
        /* 1st call of this function, search the device tree for the .entry segment */
        struct device_node *pcpEntry = of_find_node_by_path("/reserved-memory/hole@20400000");

        if (pcpEntry != NULL)
        {
            uint32_t reg[2];

            /* get physical base address and length of the .bss segment and map it */
            if (of_property_read_u32_array(pcpEntry, "reg", reg, 2) == 0)
            {
                /* sanity */
                if ((reg[0] == 0) || (reg[1] == 0))
                {
                    DEBUG_LVL_ERROR_TRACE("%s(): implausible I/O region of hole@20400000: <0x%08x, 0x%08x>\n", __func__, reg[0], reg[1]);
                } else {
                    holes[1].phyBase = (void *)reg[0];
                    holes[1].size    = reg[1];
                    holes[1].pBase   = (void *)ioremap(reg[0], reg[1]);
                }

                if (holes[1].pBase == NULL)
                {
                    DEBUG_LVL_ERROR_TRACE("%s(): cannot map I/O region of hole@20400000: <0x%08x, 0x%08x>\n", __func__, reg[0], reg[1]);
                    holes[1].phyBase = 0;
                    holes[1].size    = 0;
                }
            } else {
                DEBUG_LVL_ERROR_TRACE("%s(): cannot read reg propery of hole@20400000\n", __func__);
            }

            /* release DT node */
            of_node_put(pcpEntry);
        } else {
            DEBUG_LVL_ERROR_TRACE("%s(): cannot find device tree node hole@20400000\n", __func__);
        }
    }

    if ((holes[0].phyBase == NULL) || (holes[1].phyBase == NULL))
        return NULL;

    // Clear these memory regions
    memset(holes[0].pBase, 0, holes[0].size);
    memset(holes[1].pBase, 0, holes[1].size);

    return holes;
}

//------------------------------------------------------------------------------
/**
\brief  unmap the PCP memory holes after loading

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
static void releasePcpHoles(void)
{
    int i;

    for (i = 0; i < sizeof(holes)/sizeof(holes[0]); i++)
    {
        if (holes[i].phyBase != NULL)
        {
            iounmap(holes[i].pBase);
        }
    }

    memset(holes, 0, sizeof(holes));
}

/// \}
