/**
********************************************************************************
\file   c5armdrv.c

\brief  Linux kernel driver for C5arm/FPGA

This module handles initialization and hardware specific operations of the
C5arm/FPGA - Linux on ARM + Nios2 design.

It facilitates the communication between ARM and Nios2. It also enables
and handles the openMAC timer pulse interrupt from the openPOWERLINK driver.

\ingroup module_driver_linux_kernel_c5arm
*******************************************************************************/

/*------------------------------------------------------------------------------
Copyright (c) 2016, Kalycito Infotech Private Limited
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the copyright holders nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
------------------------------------------------------------------------------*/

//------------------------------------------------------------------------------
// includes
//------------------------------------------------------------------------------

#include "c5armdrv.h"
#include "c5niosld.h"
#include "drvintf.h"

#include <common/driver.h>

#include <linux/mm.h>
#include <asm/pgtable.h>
#include <asm/tlbflush.h>
#include <asm/io.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/version.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/atomic.h>
#include <asm/irq.h>
#include <linux/sched.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/gfp.h>
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 4, 0))
#include <linux/semaphore.h>
#endif

//============================================================================//
//            G L O B A L   D E F I N I T I O N S                             //
//============================================================================//

//------------------------------------------------------------------------------
// const defines
//------------------------------------------------------------------------------
#if (LINUX_VERSION_CODE < KERNEL_VERSION(4, 4, 0))
#error "Linux Kernel versions older 4.4.0 are not supported by this driver!"
#endif

//------------------------------------------------------------------------------
// module global vars
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// global function prototypes
//------------------------------------------------------------------------------

//============================================================================//
//            P R I V A T E   D E F I N I T I O N S                           //
//============================================================================//

//------------------------------------------------------------------------------
// const defines
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// local types
//------------------------------------------------------------------------------
/**

 \brief Platform device IO memory resource

 The structure holds the information of the IO memory resource
 and the size of the memory regions.

 */
typedef struct
{
    struct resource*             pResource;      ///< Pointer to the IO Memory resource allocated by Linux.
    void*                        pBase;          ///< Virtual address of IO memory regions.
    resource_size_t              size;           ///< Size of the memory allocated for the device.
} tIoMemoryResource;

/**
 \brief  C5arm kernel driver instance

 Provides all the necessary information used by the C5arm kernel driver module
 to interact with the device and interface with the stack above it.
 */
typedef struct
{
    struct platform_device*      pPlatformDev;                          ///< Pointer to platform device structure for driver.
    tIoMemoryResource            aIoMemRes[kIoMemRegionLast];           ///< Memory instances of the platform device.
    int                          irqNumber;                             ///< Interrupt number for the platform device.
    tIrqCallback                 pfnCbIrq;                              ///< Sync irq callback function of the upper user layer.
    BOOL                         fIrqEnabled;                           ///< Flag to check if sync irq for user has been enabled.
} tPcpDrvInstance;

//------------------------------------------------------------------------------
// local function prototypes
//------------------------------------------------------------------------------
static int          initOnePlatformDev(struct platform_device* pDev_p);
static int          removeOnePlatformDev(struct platform_device* pDev_p);
static irqreturn_t  pcpIrqHandler(int irqNum_p,
                                  void* ppDevInstData_p);

//------------------------------------------------------------------------------
// local vars
//------------------------------------------------------------------------------
// Platform device identification
#if defined(CONFIG_OF)
static struct of_device_id  drv_of_match[] =
{
    {   .compatible = "plk_driver,DDR", },               // __devinitdata creates warning!
    { /* end of table */}                                // keep devinit in separate data section,
};                                                       // linker is not able to link

MODULE_DEVICE_TABLE(of, drv_of_match);
#else
#define drv_of_match    NULL
#endif /* CONFIG_OF */

static struct platform_driver   pcpDriver_l =
{
    .probe      = initOnePlatformDev,
    .remove     = removeOnePlatformDev,
    .suspend    = NULL,                                     // Not handling power management functions
    .resume     = NULL,                                     // Not handling power management functions
    .driver     = { .name = "c5arm-platform-device",
                    .owner = THIS_MODULE,
                    .of_match_table = drv_of_match,         // This function is to check the device
                  },                                        // from the device tree
};

static tPcpDrvInstance instance_l;

//============================================================================//
//            P U B L I C   F U N C T I O N S                                 //
//============================================================================//

//------------------------------------------------------------------------------
/**
\brief  openPOWERLINK c5arm kernel driver initialization

This function initializes the openPOWERLINK c5arm driver.

\return The function returns a tOplkError error code.

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
tOplkError c5armdrv_init(void)
{
    tOplkError  ret = kErrorOk;
    INT         result;

    // Clear instance structure
    OPLK_MEMSET(&instance_l, 0, sizeof(instance_l));

    instance_l.irqNumber = -1;

    DEBUG_LVL_ALWAYS_TRACE("%s(): Registering the driver to the kernel...",
                           __func__);

    /*
     * TODO: This function can be replaced with platform_driver_probe
     *       to reduce memory footprints
     */
    result = platform_driver_register(&pcpDriver_l);
    if (result != 0)
        return kErrorNoResource;

    DEBUG_LVL_ALWAYS_TRACE("Done\n");

    return ret;
}

//------------------------------------------------------------------------------
/**
\brief  openPOWERLINK c5arm kernel driver shutdown

This function shuts down the openPOWERLINK c5arm driver.

\return The function returns a tOplkError error code.

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
tOplkError c5armdrv_exit(void)
{
    DEBUG_LVL_DRVINTF_TRACE("%s(): Calling platform_driver_unregister()\n",
                            __func__);
    platform_driver_unregister(&pcpDriver_l);

    return kErrorOk;
}

//------------------------------------------------------------------------------
/**
\brief  Get common memory base virtual address

This routine fetches the common memory base address after
its been remapped into Linux kernel virtual address space..

 \param[in]      memId_p          ID of the requested memory region.

 \return Returns the base address of common memory.

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
void* c5armdrv_getMemRegionAddr(tIoMemRegions memId_p)
{
    if (memId_p >= kIoMemRegionLast)
        return NULL;

    return instance_l.aIoMemRes[memId_p].pBase;
}

//------------------------------------------------------------------------------
/**
\brief  Get shared memory physical address

This routine fetches the physical address of shared Memory.

 \param[in]      memId_p          ID of the requested memory region.

 \return Returns the physical address of Shared Memory.

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
void* c5armdrv_getMemPhyAddr(tIoMemRegions memId_p)
{
    if (memId_p >= kIoMemRegionLast)
        return NULL;

    return (void*)instance_l.aIoMemRes[memId_p].pResource->start;
}

//------------------------------------------------------------------------------
/**
\brief  Register user sync interrupt callback

This function stores the user sync event callback function to be called from
the sync ISR.

\param[in]      cbSync_p            Pointer to the user sync callback function.

\return The function returns a tOplkError error code.

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
tOplkError c5armdrv_regIrqHandler(tIrqCallback cbIrq_p)
{
    instance_l.pfnCbIrq = cbIrq_p;

    return kErrorOk;
}

//------------------------------------------------------------------------------
/**
\brief  Enable/Disable user sync interrupt

Enables or disable the forwarding of sync interrupt to user.

\param[in]      fEnable_p           Boolean value indicating whether or not to
                                    forward sync IRQ to user.

\return The function returns a tOplkError error code.

\ingroup module_driver_linux_kernel_c5arm
*/
//------------------------------------------------------------------------------
tOplkError c5armdrv_enableIrq(BOOL fEnable_p)
{
    UINT32 *pAddr;

    if ((pAddr = (UINT32 *)c5armdrv_getMemRegionAddr(kIoMemRegionFpgaReg)) == NULL)
        return kErrorOk;

    if (fEnable_p == TRUE) {
        iowrite32(C5ARMDRV_IRQ_MASK, &pAddr[kIoRegIrqEnable]);
    } else {
        iowrite32(0, &pAddr[kIoRegIrqEnable]);
    }

    iowrite32(C5ARMDRV_IRQ_MASK, &pAddr[kIoRegIrqClear]);

    instance_l.fIrqEnabled = fEnable_p;

    return kErrorOk;
}

//============================================================================//
//            P R I V A T E   F U N C T I O N S                               //
//============================================================================//
/// \name Private Functions
/// \{

//------------------------------------------------------------------------------
/**
\brief  openPOWERLINK C5arm kernel driver interrupt handler

This function is the interrupt service routine for the openPOWERLINK c5arm driver.

\param[in]      irqNum_p            IRQ number
\param[in]      ppDevInstData_p     Pointer to private data provided by request_irq

\return The function returns an IRQ handled code.
*/
//------------------------------------------------------------------------------
static irqreturn_t pcpIrqHandler(int irqNum_p,
                                 void* ppDevInstData_p)
{
    UINT32 *pAddr;
    UINT32 pending;
    irqreturn_t ret = IRQ_HANDLED;

    UNUSED_PARAMETER(irqNum_p);
    UNUSED_PARAMETER(ppDevInstData_p);

    if ((pAddr = (UINT32 *)c5armdrv_getMemRegionAddr(kIoMemRegionFpgaReg)) == NULL)
        return ret;

    pending = ioread32(&pAddr[kIoRegIrqPending]) & C5ARMDRV_IRQ_MASK;
    if (pending) {
        iowrite32(pending, &pAddr[kIoRegIrqClear]);

        if ((instance_l.pfnCbIrq != NULL) &&
            (instance_l.fIrqEnabled == TRUE))
        {
            // User wants the interrupt, forward it without any argument
            instance_l.pfnCbIrq(pending);
        }
    }

    return ret;
}

//------------------------------------------------------------------------------
/**
\brief  Initialize one C5arm device

This function initializes one C5arm device.

\param[in,out]  pDev_p              Pointer to corresponding C5arm device structure

\return The function returns an integer error code.
\retval 0                           Successful
\retval Otherwise                   Error
*/
//------------------------------------------------------------------------------
static int initOnePlatformDev(struct platform_device* pDev_p)
{
    INT                 result = 0;
    tIoMemRegions       memId = 0;

    if (pcpLoad != NULL) {
        /* put the PCP into reset */
        if (c5niosld_reset() != kErrorOk) {
            result = -EIO;
            goto ExitClean;
        }

        /* load the PCP with the specified ELF file */
        if (c5niosld_load(pcpLoad) != kErrorOk) {
            c5niosld_start();
            result = -EBADF;
            goto ExitClean;
        }

        /* start the PCP */
        if (c5niosld_start() != kErrorOk) {
            result = -ENOEXEC;
            goto ExitClean;
        }
    }

    if (pDev_p == NULL)
    {
        DEBUG_LVL_DRVINTF_TRACE("%s(): Device discarded\n", __func__);
        result = -ENODEV;
        goto ExitClean;
    }

    if (instance_l.pPlatformDev != NULL)
    {
        DEBUG_LVL_DRVINTF_TRACE("%s(): Device (%s) already registered\n",
                                __func__,
                                pDev_p->name);
        result = -ENODEV;
        goto ExitClean;
    }

    // Save the handle for the platform device
    instance_l.pPlatformDev = pDev_p;

    for (memId = 0; memId < kIoMemRegionLast; memId++)
    {
        tIoMemoryResource* pMemResource = &instance_l.aIoMemRes[memId];

        DEBUG_LVL_DRVINTF_TRACE("%s(): IOMEM resource initialization...%s\n",
                                __func__,
                                ((memId == 0) ? "Common Memory" : "Shared Memory"));

        pMemResource->pResource =
                  platform_get_resource(pDev_p, IORESOURCE_MEM, memId);

        if (pMemResource->pResource == NULL)
        {
            result = -ENODEV;
            goto ExitClean;
        }

        pMemResource->size = (pMemResource->pResource->end -
                              pMemResource->pResource->start + 1);

        /* Mark the common memory region exclusively for C5arm device */
        if (memId == 0)
        {
            /* common memory is a RAM in the I/O region (f2h bridge) */
            if (!request_mem_region(pMemResource->pResource->start,
                                    pMemResource->size,
                                    PLK_DRV_NAME))
            {
                DEBUG_LVL_DRVINTF_TRACE("Request memory region failed for %s\n",
                                        ((memId == 0) ? "Common Memory" : "Shared Memory"));
                result = -ENOMEM;
                goto ExitClean;
            }

            /* Remap Io memory regions into Linux virtual address space */
            pMemResource->pBase = ioremap(pMemResource->pResource->start,
                                          pMemResource->size);

            if (pMemResource->pBase == NULL)
            {
                DEBUG_LVL_DRVINTF_TRACE("Ioremap failed for %s\n",
                                        ((memId == 0) ? "Common Memory" : "Shared Memory"));
                result = -EIO;
                goto ExitClean;
            }
        } else {
            /* The shared RAM region is part of the DRAM area, but reserved as memory hole */
            pMemResource->pBase = ioremap(pMemResource->pResource->start, pMemResource->size);

            if (pMemResource->pBase == NULL) {
                DEBUG_LVL_ERROR_TRACE("%s(): cannot set shared memeory to uncached\n", __func__);

                result = -EIO;
                goto ExitClean;
            }

            DEBUG_LVL_DRVINTF_TRACE("%s: shared mem %p/%p %zu uncached\n", __func__ ,pMemResource->pResource->start, pMemResource->pBase, pMemResource->size);
        }

        DEBUG_LVL_DRVINTF_TRACE("MEM_RESOURCE: start/virt 0x(%p)/0x(%p), size %zu\n",
                                pMemResource->pResource->start,
                                pMemResource->pBase,
                                pMemResource->size);
    }

    DEBUG_LVL_DRVINTF_TRACE("%s(): IRQ resource initialization...", __func__);

    /* Get Interrupt number for device */
    instance_l.irqNumber = platform_get_irq(pDev_p, 0);

    if (instance_l.irqNumber < 0)
    {
        DEBUG_LVL_DRVINTF_TRACE("Failed\n");
        result = -ENODEV;
        goto ExitClean;
    }

    /* Request IRQ */
    DEBUG_LVL_DRVINTF_TRACE("Requesting IRQ resource...");

    if (request_irq(instance_l.irqNumber,
                    pcpIrqHandler,
                    IRQF_SHARED,
                    PLK_DRV_NAME,
                    pDev_p))
    {
        DEBUG_LVL_DRVINTF_TRACE("Request IRQ failed \n");
        result = -EIO;
        goto ExitClean;
    }

    DEBUG_LVL_DRVINTF_TRACE("%s(): finished with %d\n", __func__, result);
    return result;

ExitClean:
    DEBUG_LVL_DRVINTF_TRACE("%s(): finished with %d\n", __func__, result);
    removeOnePlatformDev(pDev_p);
    return result;
}

//------------------------------------------------------------------------------
/**
\brief  Remove one c5arm device

This function removes one c5arm device.

\param[in,out]  pPciDev_p           Pointer to corresponding C5arm device structure
*/
//------------------------------------------------------------------------------
static int removeOnePlatformDev(struct platform_device* pDev_p)
{
    tIoMemRegions memId = 0;

    // Make sure the veth device is removed
    drvintf_exit();

    c5armdrv_enableIrq(FALSE);

    /* Remove interrupt handler */
    if (instance_l.irqNumber >= 0)
    {
        free_irq(instance_l.irqNumber, pDev_p);
        instance_l.irqNumber = -1;
    }

    for (memId = 0; memId < kIoMemRegionLast; memId++)
    {
        tIoMemoryResource* pMemResource = &instance_l.aIoMemRes[memId];

        if (memId == 0)
        {
            /* unmap and release I/O memory (common memory) */
            if (pMemResource->pBase != NULL)
            {
                iounmap(pMemResource->pBase);
                pMemResource->pBase = NULL;
            }

            if (pMemResource->pResource != NULL)
            {
                release_mem_region(pMemResource->pResource->start,
                                   pMemResource->size);
                pMemResource->pResource = NULL;
            }
        } else {
            /* shared memory is a fixed resource */
            iounmap(pMemResource->pBase);
            pMemResource->pBase     = NULL;
            pMemResource->pResource = NULL;
        }
    }

    instance_l.pPlatformDev = NULL;
    return 0;
}
/// \}
