/**
********************************************************************************
 \file   firmwareinfo.c

\brief  Source file of the firmware info module

This module implements an access to firmware information generated by the
openCONFIGURATOR.

\ingroup module_app_firmwaremanager
*******************************************************************************/

/*------------------------------------------------------------------------------
Copyright (c) 2017, B&R Industrial Automation GmbH
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
    * Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
    * Neither the name of the copyright holders nor the
      names of its contributors may be used to endorse or promote products
      derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
------------------------------------------------------------------------------*/

//------------------------------------------------------------------------------
// includes
//------------------------------------------------------------------------------
#include <firmwaremanager/firmwareinfo.h>

//============================================================================//
//            G L O B A L   D E F I N I T I O N S                             //
//============================================================================//

//------------------------------------------------------------------------------
// const defines
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// module global vars
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// global function prototypes
//------------------------------------------------------------------------------

//============================================================================//
//            P R I V A T E   D E F I N I T I O N S                           //
//============================================================================//

//------------------------------------------------------------------------------
// const defines
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// local types
//------------------------------------------------------------------------------

/**
\brief Firmware info instance
*/
typedef struct tFirmwareInfoInstance
{
    tFirmwareInfoConfig config;     ///< Firmware info configuration
    tFirmwareInfoList   pInfoList;  ///< Firmware info list
} tFirmwareInfoInstance;

//------------------------------------------------------------------------------
// local vars
//------------------------------------------------------------------------------

//------------------------------------------------------------------------------
// local function prototypes
//------------------------------------------------------------------------------

static tFirmwareRet initInfoList(tFirmwareInfoInstance* pHandle_p);
static tFirmwareInfo* findModuleInInfoList(tFirmwareInfoInstance* pHandle_p,
                                           const tFirmwareModuleInfo * pModuleInfo_p);

//============================================================================//
//            P U B L I C   F U N C T I O N S                                 //
//============================================================================//

//------------------------------------------------------------------------------
/**
\brief  Create a firmware info instance

This function creates an instance of the firmware info module.

\param pConfig_p [in]   Pointer to the configuration structure for instance
                        creation.
\param ppHandle_p [out] Pointer to a firmware info handle which is filled if
                        creation was successful.

\return This functions returns a value of \ref tFirmwareRet.

\ingroup module_app_firmwaremanager
*/
//------------------------------------------------------------------------------
tFirmwareRet firmwareinfo_create(const tFirmwareInfoConfig* pConfig_p,
                                 tFirmwareInfoHandle* ppHandle_p)
{
    tFirmwareRet            ret = kFwReturnOk;
    tFirmwareInfoInstance*  instance = NULL;

    if ((pConfig_p == NULL) || (ppHandle_p == NULL))
    {
        ret = kFwReturnInvalidParameter;
        goto EXIT;
    }

    instance = (tFirmwareInfoInstance*)malloc(sizeof(tFirmwareInfoInstance));
    if (instance == NULL)
    {
        ret = kFwReturnNoResource;
        goto EXIT;
    }

    memset(instance, 0, sizeof(tFirmwareInfoInstance));
    memcpy(&instance->config, pConfig_p, sizeof(tFirmwareInfoConfig));

    ret = initInfoList(instance);

EXIT:
    if (ret == kFwReturnOk)
    {
        *ppHandle_p = instance;
    }
    else
    {
        free(instance);
    }

    return ret;
}

//------------------------------------------------------------------------------
/**
\brief  Destroy a firmware info instance

This function destroys an instance of the firmware info module.

\param pHandle_p [in]   Handle of the firmware info instance which shall be
                        destroyed.

\return This functions returns a value of \ref tFirmwareRet.

\ingroup module_app_firmwaremanager
*/
//------------------------------------------------------------------------------
tFirmwareRet firmwareinfo_destroy(tFirmwareInfoHandle pHandle_p)
{
    tFirmwareRet ret = kFwReturnOk;

    if (pHandle_p != NULL)
    {
        firmwareinfodecode_freeInfo(pHandle_p->pInfoList);
        free(pHandle_p);
    }
    else
    {
        ret = kFwReturnInvalidInstance;
    }

    return ret;
}

//------------------------------------------------------------------------------
/**
\brief  Get the firmware info a specific module

This function gets the available firmware information for a specified module.

\param pHandle_p [in]       Handle of the firmware info instance
\param pModuleInfo_p [in]   Pointer to the module info specifying the requested
                            module.
\param ppFwInfo_p [out]     Pointer to a firmware info pointer which is filled
                            if an associated firmware info is available for
                            the specified module.

\return This functions returns a value of \ref tFirmwareRet.

\ingroup module_app_firmwaremanager
*/
//------------------------------------------------------------------------------
tFirmwareRet firmwareinfo_getInfoForNode(tFirmwareInfoHandle pHandle_p,
                                         const tFirmwareModuleInfo * pModuleInfo_p,
                                         tFirmwareInfo** ppFwInfo_p)
{
    tFirmwareRet    ret = kFwReturnOk;
    tFirmwareInfo*  pInfo = NULL;

    if (pHandle_p == NULL)
    {
        ret = kFwReturnInvalidInstance;
        goto EXIT;
    }

    if ((pModuleInfo_p == NULL) || (ppFwInfo_p == NULL))
    {
        ret = kFwReturnInvalidParameter;
        goto EXIT;
    }

    pInfo = findModuleInInfoList(pHandle_p, pModuleInfo_p);

    if (pInfo != NULL)
    {
        *ppFwInfo_p = pInfo;
    }
    else
    {
        ret = kFwReturnModuleNotFound;
    }

EXIT:
    return ret;
}

//============================================================================//
//            P R I V A T E   F U N C T I O N S                               //
//============================================================================//
/// \name Private Functions
/// \{

//------------------------------------------------------------------------------
/**
\brief  Initialize firmware info list

\param pHandle_p [in]       Handle of the firmware info instance

\return This functions returns a value of \ref tFirmwareRet.
*/
//------------------------------------------------------------------------------
static tFirmwareRet initInfoList(tFirmwareInfoInstance* pHandle_p)
{
    tFirmwareRet ret = kFwReturnOk;

    ret = firmwareinfodecode_decodeInfo(pHandle_p->config.pFwStore,
                                        &pHandle_p->pInfoList);

    return ret;
}

//------------------------------------------------------------------------------
/**
\brief  Find module in firmware info list

\param pHandle_p [in]       Handle of the firmware info instance
\param pModuleInfo_p [in]   Pointer to module info

\return This functions returns a pointer to the firmware info list.
\retval NULL    No firmware info is found
\retval !NULL   Pointer to the found firmware info
*/
//------------------------------------------------------------------------------
static tFirmwareInfo* findModuleInInfoList(tFirmwareInfoInstance* pHandle_p,
                                           const tFirmwareModuleInfo * pModuleInfo_p)
{
    tFirmwareInfo*      pInfo = NULL;
    tFirmwareInfoEntry* pIter = pHandle_p->pInfoList;

    while (pIter != NULL)
    {
        if (memcmp(pModuleInfo_p, &pIter->fwInfo.moduleInfo,
                   sizeof(tFirmwareModuleInfo)) == 0)
        {
            pInfo = &pIter->fwInfo;
            break;
        }

        pIter = pIter->pNext;
    }

    return pInfo;
}

/// \}
