/**
 * @file wd.c
 * @brief Watchdog and reset handling
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#include <stdbool.h>
#include <stdint.h>

#include <xparameters.h>
#include <xstatus.h>
#include <xil_io.h>
#include <xwdttb.h>

#include "logging.h"
#include "wd.h"

/** WD module instance state */
struct WdModule {
    bool isWdKick;                  /**< if true, actually kick watchdog */
} mWD;

/* watchdog reset type: force it into .data section to avoid clobbering it when clearing .bss */
volatile unsigned long wdResetType __attribute__ ((section(".data"))) = WD_INIT;

void wdKick(void)
{
    if (!mWD.isWdKick)
        return;

    /* get TWCSR0 */
    uint32_t twcsr0 = Xil_In32(XPAR_WATCHDOG_BASEADDR + 0x00);

    /* reset Watchdog by setting TWCRS0[WRS] and TWCRS0[WDS] bits */
    twcsr0 |= XWT_CSR0_WRS_MASK;
    twcsr0 |= XWT_CSR0_WDS_MASK;
    Xil_Out32(XPAR_WATCHDOG_BASEADDR + 0x00, twcsr0);
}

void wdReset(void)
{
    wdResetType  = WD_SCHEDULED;
    mWD.isWdKick = false;
}

void wdStop(void)
{
    /* get TWCSR0 */
    uint32_t twcsr0 = Xil_In32(XPAR_WATCHDOG_BASEADDR + 0x00);

    /* clear TWCRS0[EWDT1] to disable the watchdog */
    twcsr0 &= ~XWT_CSR0_EWDT1_MASK;

    /* disable Watchdog by clearing both TWCSR0[EWDT1] and TWCSR1[EWDT2] */
    Xil_Out32(XPAR_WATCHDOG_BASEADDR + 0x00, twcsr0);
    Xil_Out32(XPAR_WATCHDOG_BASEADDR + 0x04, ~XWT_CSRX_EWDT2_MASK);
}

int
wdSetup(void)
{
    LOG_TRACE("# WD: setup Watchdog\r\n");

    if (wdResetType == WD_SCHEDULED)
        LOG_PRINT("# WD: scheduled reset\r\n");
    else if (wdResetType != WD_INIT)
        LOG_PRINT("* WD: watchdog reset\r\n");

    wdResetType  = WD_WATCHDOG;
    mWD.isWdKick = true;

    /*
     * <braindamage>
     * The xwdttb library is completely confused regarding the time base watchdog
     * mode. Regrettably, we therefore must use raw register access for this
     * device
     * </braindamage>
     */

    /*
     * program watchdog register width in MWR[4:0]
     *
     * This sets the timeout interval to 2^18 * 100MHz = 2.62ms
     */
    Xil_Out32(XPAR_WATCHDOG_BASEADDR + 0x0C, 18);

    /* get TWCSR0 */
    uint32_t twcsr0 = Xil_In32(XPAR_WATCHDOG_BASEADDR + 0x00);

    /* clear TWCRS0[WRS] and TWCRS0[WDS] bits by writing a '1' */
    twcsr0 |= XWT_CSR0_WRS_MASK;
    twcsr0 |= XWT_CSR0_WDS_MASK;

    /* set TWCRS0[EWDT1] to enable the watchdog */
    twcsr0 |= XWT_CSR0_EWDT1_MASK;

    /* enable Watchdog by setting TWCSR0[EWDT1] and TWCSR1[EWDT2] */
    Xil_Out32(XPAR_WATCHDOG_BASEADDR + 0x00, twcsr0);
    Xil_Out32(XPAR_WATCHDOG_BASEADDR + 0x04, XWT_CSRX_EWDT2_MASK);

    return XST_SUCCESS;
}
