/**
 * @file tts.c
 *
 * @brief implementation details of the CDI2 TTS Device logic
 * @author AIT
 * @copyright &copy;2023 Austrian Institute of Technology (AIT)
 */

#include <stddef.h>
#include <stdint.h>
#include <stdbool.h>

#include <xparameters.h>
#include <xinterrupt_wrap.h>
#include <xstatus.h>
#include <xiltimer.h>
#include <xil_io.h>
#include <tts_device.h>

#include "cdi2.h"
#include "logging.h"
#include "powerlink.h"
#include "statistics.h"
#include "tts.h"

/*
 * TC transmitted via BSMINFO are sampled by the BSM at SoC TX edge.
 *
 * In addition to the SoC RX time (5.76us), our implementation adds
 * overhead/latency: RX interrupt handling, plus the SoC frame
 * processing time in ddlkframe.c until the system clock is sampled
 * for the SoC time stamp. This overhead/latency essentially is a
 * measured value that must be subtracted from the running TC offset
 */

#ifdef NDEBUG
#define MEAS_TC_OFFSET_OVERHEAD 11175       /**< measured RX and IRQ overhead [ticks] */
#define SCHED_BP_OVERHEAD       610         /**< measured timer triggered BP overhead/latency [ticks] */
#else
#define MEAS_TC_OFFSET_OVERHEAD 11700       /**< measured RX and IRQ overhead [ticks] */
#define SCHED_BP_OVERHEAD       850         /**< measured timer triggered BP overhead/latency [ticks] */
#endif

/** TTS module instance state */
struct TtsModule {
    XTmrCtr  *pBpTimer;                     /**< AXI timer instance for timed BP interrupts */
    uint32_t  lastTc;                       /**< last TC count value from BSMINFO */
    uint32_t  tcPeriodMeas;                 /**< measured BSMINFO:TC period [ticks] */
    uint32_t  tcOffsetMeas;                 /**< measured BSMINFO:TC offset [ticks] from system sleep clock */
} mTTS;

uint32_t
ttsSetBpInterruptEnable(bool isEnable)
{
    volatile struct TTSRegisterFile *pTTS = (struct TTSRegisterFile *)XPAR_TTS_DEVICE_BASEADDR;

    /* enable/disable TTS BP interrupt bit: irq_enable[1] */
    uint32_t irq_enable = pTTS->irq_enable;

    if (isEnable)
        irq_enable |= 0x02;
    else
        irq_enable &= ~0x02;

    pTTS->irq_enable = irq_enable;

    return pTTS->irq_enable;
}

uint32_t
ttsClearBpInterrupt(void)
{
    volatile struct TTSRegisterFile *pTTS = (struct TTSRegisterFile *)XPAR_TTS_DEVICE_BASEADDR;

    /* set BP interrupt bit in clear register: irq_clear[1] */
    pTTS->irq_clear = 0x02;

    return pTTS->irq_pending;
}

uint32_t
ttsGetPendingInterrupts(void)
{
    volatile struct TTSRegisterFile *pTTS = (struct TTSRegisterFile *)XPAR_TTS_DEVICE_BASEADDR;

    return pTTS->irq_pending;
}

uint32_t
ttsTriggerBpInterrupt(void)
{
    volatile struct TTSRegisterFile *pTTS = (struct TTSRegisterFile *)XPAR_TTS_DEVICE_BASEADDR;

    /* set the BP interrupt request bit: irq_set[1] */
    pTTS->irq_set = 0x02;

    return pTTS->irq_pending;
}

uint32_t
ttsGetTcPeriod(void)
{
    volatile struct TTSRegisterFile *pTTS      = (struct TTSRegisterFile *)XPAR_TTS_DEVICE_BASEADDR;
    uint32_t                         ttsPeriod = pTTS->ttstc_period;

    if (ttsPeriod == (uint32_t)-1)
        /* no signal/measurement */
        ttsPeriod = 0;

    /* TTS tick [ns] = ttsPeriod [ticks] 10^9 [ns/s] / 50MHz) */
    return (ttsPeriod * (1000000000 / XPAR_TTS_DEVICE_CORE_CLK_FREQ));
}

enum TcCheck
ttsCheckTc(void)
{
    volatile struct TTSRegisterFile *pTTS      = (struct TTSRegisterFile *)XPAR_TTS_DEVICE_BASEADDR;
    const uint32_t                   ttsPeriod = pTTS->ttstc_period;

    if (ttsPeriod == (uint32_t)-1)
        /* no TC */
        return TTSTC_NOCLOCK;
    else if ((ttsPeriod < kTcMinPeriod) || (ttsPeriod > kTcMaxPeriod))
        /* TC out of range */
        return TTSTC_RANGE;

    /* TC OK */
    return TTSTC_OK;
}

void
ttsResetTcMeas(void)
{
    mTTS.lastTc       = 0;
    mTTS.tcPeriodMeas = 0;
    mTTS.tcOffsetMeas = 0;
}

void
ttsUpdateTcMeas(uint64_t tsSoc, uint32_t tc)
{
#define SYNC_PERIOD_TICKS (XPAR_SLEEPTIMER_CLOCK_FREQ_HZ / 1000)    /* nominal CDI2 SYNC interval: 1ms in [ticks] */

    /* calculate delta to last saved TC count */
    uint32_t tcDelta  = tc - mTTS.lastTc;

    /* save current interval values */
    mTTS.lastTc = tc;

    /* sanity: ignore all TC counts > 1000 (>100m/s belt speed) */
    if ((tcDelta > 1000) || (tcDelta <= 0))
        return;

    /* assuming that our system clock (100MHz, T=10ns) is much faster than the TC (125kkHz, T=8us) */
    uint32_t tcPeriodTicks = SYNC_PERIOD_TICKS / tcDelta;

    /* calculate MEAS values as moving average: u[n] = \lambda s[n] + (1 - \lambda) u[n-1] */
    mTTS.tcPeriodMeas = (12 * tcPeriodTicks + 4 * mTTS.tcPeriodMeas) >> 4;

    /* Remember, tsSoc really is a 32 bit value zero padded to 64 bits */
    uint32_t tcOffsetTicks = (uint32_t)tsSoc - tc * mTTS.tcPeriodMeas;

    if (mTTS.tcOffsetMeas != 0)
        mTTS.tcOffsetMeas = 12 * (tcOffsetTicks >> 4) + 4 * (mTTS.tcOffsetMeas >> 4);
    else
        mTTS.tcOffsetMeas = tcOffsetTicks;
}

uint64_t
ttsTc2SysTick(uint32_t tc)
{
    return tc * mTTS.tcPeriodMeas + (mTTS.tcOffsetMeas - MEAS_TC_OFFSET_OVERHEAD);
}

uint32_t
ttsSysTick2Tc(uint64_t sysClk)
{
    if (mTTS.tcPeriodMeas != 0)
        return (sysClk - (mTTS.tcOffsetMeas - MEAS_TC_OFFSET_OVERHEAD)) / mTTS.tcPeriodMeas;
    else
        return  sysClk - (mTTS.tcOffsetMeas - MEAS_TC_OFFSET_OVERHEAD);
}

unsigned int
ttsSampleReset(void)
{
    volatile struct TTSRegisterFile *pTTS      = (struct TTSRegisterFile *)XPAR_TTS_DEVICE_BASEADDR;
    const uint32_t                   ttsStatus = pTTS->ttsifc_status;

    /* return synchronized & sampled value if TTS reset line */
    return (ttsStatus & 0x04) != 0;
}

void
ttsSetDeviceReady(bool isDeviceReady)
{
    volatile struct TTSRegisterFile *pTTS = (struct TTSRegisterFile *)XPAR_TTS_DEVICE_BASEADDR;

    pTTS->ttsready_device = isDeviceReady ? 1 : 0;
}

/**
 * TTS Device BP interrupt handler
 *
 * @param[in] intr interrupt number
 */
static void
ttsBpInterruptHandler(void *intr)
{
    /* clear pending interrupt in TTS device */
    (void)ttsClearBpInterrupt();

    /* sanity */
    if ((uintptr_t)intr != XPAR_INTERRUPTCONTROLLER_TTS_DEVICE_IRQ_O_INTR)
        return;

    const QueueHandle_t hCdi2Queue = cdi2GetMessageQueue();

    if (hCdi2Queue != NULL) {
        /* get running system clock [ticks] */
        uint64_t sysClockTicks = 0;
        XTime_GetTime(&sysClockTicks);

        static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_BPPRESENT };
        cdi2Msg.bpPresent.ts = sysClockTicks;

        if (xQueueSendFromISR(hCdi2Queue, &cdi2Msg, NULL) != pdTRUE)
            statsIncrBp();
    }
}

/**
 * TTS timer scheduled BP interrupt handler
 *
 * @param[in] intr interrupt number
 */
static void
ttsTimerInterruptHandler(void *intr)
{
    /* Clear the timer interrupt */
    if (mTTS.pBpTimer != NULL) {
        uint32_t  TCSR1 = XTmrCtr_GetControlStatusReg(mTTS.pBpTimer->BaseAddress, 1);
        XTmrCtr_SetControlStatusReg(mTTS.pBpTimer->BaseAddress, 1, TCSR1);
    }

    /* sanity */
    if ((uintptr_t)intr != XPAR_INTERRUPTCONTROLLER_SLEEPTIMER_INTERRUPT_INTR)
        return;

    const QueueHandle_t hCdi2Queue = cdi2GetMessageQueue();

    if (hCdi2Queue != NULL) {
        /* get running system clock [ticks] */
        uint64_t sysClockTicks = 0;
        XTime_GetTime(&sysClockTicks);

        static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_BPPRESENT };
        cdi2Msg.bpPresent.ts = sysClockTicks;

        if (xQueueSendFromISR(hCdi2Queue, &cdi2Msg, NULL) != pdTRUE)
            statsIncrBp();
    }
}

uint64_t
ttsScheduleBp(uint32_t tcBp)
{
    /* get running system clock [ticks] */
    uint64_t sysClockTicks = 0;
    XTime_GetTime(&sysClockTicks);

    uint64_t tsTrigger = ttsTc2SysTick(tcBp);

    /* number of [ticks] from now to interrupt */
    uint32_t deltaTicks = tsTrigger - sysClockTicks;

    if (deltaTicks > SCHED_BP_OVERHEAD)
        deltaTicks -= SCHED_BP_OVERHEAD;

    /* set TLR1 to the desired delay [ticks] and set TCSR1[LOAD1] to load time counter */
    XTmrCtr_WriteReg(mTTS.pBpTimer->BaseAddress, 1, XTC_TLR_OFFSET, deltaTicks);
    XTmrCtr_WriteReg(mTTS.pBpTimer->BaseAddress, 1, XTC_TCSR_OFFSET, XTC_CSR_LOAD_MASK);

    /* configure and start BP time counter: interrupt mode, count-down */
    XTmrCtr_WriteReg(mTTS.pBpTimer->BaseAddress, 1, XTC_TCSR_OFFSET, XTC_CSR_DOWN_COUNT_MASK |
                                                                     XTC_CSR_ENABLE_INT_MASK |
                                                                     XTC_CSR_ENABLE_TMR_MASK);

    return tsTrigger;
}

int
ttsSetup(void)
{
    LOG_TRACE("# TTS: setup TTS BP timed interrupt counter\r\n");

    /* reset clock/TC tracking */
    mTTS.lastTc       = 0;
    mTTS.tcOffsetMeas = 0;
    mTTS.tcPeriodMeas = 0;

    /* use time counter 1 of the AXI sleep timer for BP timed interrupts */
    mTTS.pBpTimer = &TimerInst.AxiTimer_SleepInst;

    /* clear time counter 1 of the AXI sleep timer */
    XTmrCtr_WriteReg(mTTS.pBpTimer->BaseAddress, 1, XTC_TLR_OFFSET, 0);                                             /* clear load register TLR1 */
    XTmrCtr_WriteReg(mTTS.pBpTimer->BaseAddress, 1, XTC_TCSR_OFFSET, XTC_CSR_INT_OCCURED_MASK | XTC_CSR_LOAD_MASK); /* reset timer counter and interrupt */
    XTmrCtr_WriteReg(mTTS.pBpTimer->BaseAddress, 1, XTC_TCSR_OFFSET, 0);                                            /* reset timer status */

    LOG_TRACE("# TTS: register TTS BP interrupt handlers\r\n");

    /* Register TTS:BP  interrupt callback */
    XConnectToInterruptCntrl(XPAR_INTERRUPTCONTROLLER_TTS_DEVICE_IRQ_O_INTR,
                             ttsBpInterruptHandler,
                             (void *)XPAR_INTERRUPTCONTROLLER_TTS_DEVICE_IRQ_O_INTR,
                             XPAR_TICKTIMER_INTR_PARENT);
    XEnableIntrId(XPAR_INTERRUPTCONTROLLER_TTS_DEVICE_IRQ_O_INTR,
                  XPAR_TICKTIMER_INTR_PARENT);

    /* Register timed BP interrupt callback */
    XConnectToInterruptCntrl(XPAR_INTERRUPTCONTROLLER_SLEEPTIMER_INTERRUPT_INTR,
                             ttsTimerInterruptHandler,
                             (void *)XPAR_INTERRUPTCONTROLLER_SLEEPTIMER_INTERRUPT_INTR,
                             XPAR_TICKTIMER_INTR_PARENT);
    XEnableIntrId(XPAR_INTERRUPTCONTROLLER_SLEEPTIMER_INTERRUPT_INTR,
                  XPAR_TICKTIMER_INTR_PARENT);

    return XST_SUCCESS;
}
