/**
 * @file statstics.c
 * @brief CDI2 Device statistics and counters
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#include <string.h>

#include <xuartns550.h>

#include <omethlib.h>

#include "logging.h"
#include "statistics.h"

extern XUartNs550Stats *pppIfGetStats(void);
extern XUartNs550Stats *cdi2IfGetStats(void);

/** definition of CDI2 error/statistics counters */
struct Statistics cdi2Statistics;

/* this ensures that externally visible functions are emitted for the inlined functions to calm down a desperate ld(1) */
extern inline void statsIncrCdi2IfRa(void);
extern inline void statsIncrCdi2IfTx(void);
extern inline void statsIncrPppIfRa(void);
extern inline void statsIncrPppIfTx(void);
extern inline void statsIncrPlkTick(void);
extern inline void statsIncrBnRecognition(void);
extern inline void statsIncrBnResult(void);
extern inline void statsIncrSeqNo(void);
extern inline void statsIncrNmtStatus(void);
extern inline void statsIncrBsmStatus(void);
extern inline void statsIncrBp(void);
extern inline void statsIncrBnTrigger(void);
extern inline void statsIncrBnInfo(void);
extern inline void statsIncrCdi2Queue(void);

void
statsReset(void)
{
    memset(&cdi2Statistics, 0, sizeof(cdi2Statistics));
}

void
statsLog(void)
{
    ometh_stat_typ *pOmethStat = omethStatistics(omethGetHandle(0));

    if (pOmethStat != NULL) {
        logFormat(INFO, 0, "OMETH: ok %lu, lost %lu, size %lu, crc %lu, ov %lu",
                  pOmethStat->rxOk, pOmethStat->rxLost, pOmethStat->rxOversize, pOmethStat->rxCrcError, pOmethStat->rxHookOverflow);
    }

    logFormat(INFO, 0, "CDI2IF: dropped RA %lu, TX %lu", cdi2Statistics.cdi2IfRaDropped, cdi2Statistics.cdi2IfTxDropped);

    XUartNs550Stats *pCdi2IfStat = cdi2IfGetStats();

    if (pCdi2IfStat != NULL) {
        logFormat(INFO, 0, "CDI2IF: ov %hu, pe %hu, fr %hu",
                  pCdi2IfStat->ReceiveOverrunErrors, pCdi2IfStat->ReceiveParityErrors, pCdi2IfStat->ReceiveFramingErrors);
    }

    logFormat(INFO, 0, "PPPIF: dropped RA %lu, TX %lu", cdi2Statistics.pppIfRaDropped, cdi2Statistics.pppIfTxDropped);

    XUartNs550Stats *pPppIfStat = pppIfGetStats();

    if (pPppIfStat != NULL) {
        logFormat(INFO, 0, "PPPIF: ov %hu, pe %hu, fr %hu",
                  pPppIfStat->ReceiveOverrunErrors, pPppIfStat->ReceiveParityErrors, pPppIfStat->ReceiveFramingErrors);
    }

    logFormat(INFO, 0, "PLK: dropped tk %lu, bnRec %lu, BnRes %lu, SeqNo %lu", cdi2Statistics.plkTickDropped, cdi2Statistics.plkBnRecognitionDropped, cdi2Statistics.plkBnResultDropped, cdi2Statistics.plkSeqNoError);

    logFormat(INFO, 0, "CDI2: dropped bp %lu, bnTrig %lu, bnInf %lu, qu %lu", cdi2Statistics.cdi2BpDropped, cdi2Statistics.cdi2BnTriggerDropped, cdi2Statistics.cdi2BnInfoDropped, cdi2Statistics.cdi2QueueDropped);
}
