/**
 * @file sbuf.c
 * @brief Serial Interface Buffer management
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#include <assert.h>
#include <stdbool.h>
#include <stdint.h>
#include <string.h>

#include <FreeRTOS.h>
#include <queue.h>

#include <xstatus.h>

#include "logging.h"
#include "sbuf.h"

struct SbufModule {
    QueueHandle_t hQueueSbuf;                   /**< handle of sbuf ring request queue */
    struct sbuf   sbufSpace[SBUF_RING_SIZE];    /**< statically define sbuf memory */
} mSBUF;

struct sbuf *
sbufGet(bool isBlocking)
{
    struct sbuf *p = NULL;

    if (xQueueReceive(mSBUF.hQueueSbuf, &p, isBlocking ? portMAX_DELAY : 0) != pdTRUE)
        return NULL;

    /* defensive programming: cut chain */
    assert((p == NULL) || (p->pNext == NULL));

    if (p != NULL)
        p->pNext = NULL;

    return p;
}

void
sbufPut(struct sbuf *pHead)
{
    extern void STOP(void);

    /* walk chain and release buffers */
    while (pHead != NULL) {
        struct sbuf *p = pHead;
        pHead          = pHead->pNext;

        /* sanity */
        assert((p >= mSBUF.sbufSpace) && (p <= (mSBUF.sbufSpace + SBUF_RING_SIZE - 1)));

        if ((p < mSBUF.sbufSpace) || (p >= (mSBUF.sbufSpace + SBUF_RING_SIZE - 1)))
            break;

        /* clear chain link and put back to ring */
        p->pNext = NULL;

        if (xQueueSendToBack(mSBUF.hQueueSbuf, &p, 0) != pdTRUE) {
            LOG_PRINT("* sbuf inconsistent\r\n");
            STOP();
        }
    }
}

struct sbuf *
sbufGetIsr()
{
    struct sbuf *p = NULL;

    if (xQueueReceiveFromISR(mSBUF.hQueueSbuf, &p, NULL) != pdTRUE)
        return NULL;

    /* defensive programming: cut chain */
    assert((p == NULL) || (p->pNext == NULL));

    if (p != NULL)
        p->pNext = NULL;

    return p;
}

void
sbufPutIsr(struct sbuf *pHead)
{
    extern void STOP(void);

    /* walk chain and release buffers */
    while (pHead != NULL) {
        struct sbuf *p = pHead;
        pHead          = pHead->pNext;

        /* sanity */
        assert((p >= mSBUF.sbufSpace) && (p <= (mSBUF.sbufSpace + SBUF_RING_SIZE - 1)));

        if ((p < mSBUF.sbufSpace) || (p >= (mSBUF.sbufSpace + SBUF_RING_SIZE - 1)))
            break;

        /* clear chain link and put back to ring */
        p->pNext = NULL;

        if (xQueueSendToBackFromISR(mSBUF.hQueueSbuf, &p, NULL) != pdTRUE) {
            LOG_PRINT("* sbuf inconsistent\r\n");
            STOP();
        }
    }
}

size_t
sbufAvailable(void)
{
    return uxQueueMessagesWaiting(mSBUF.hQueueSbuf);
}

int
sbufSetupRing()
{
    /* clear ring memory */
    memset(mSBUF.sbufSpace, 0, sizeof(mSBUF.sbufSpace));

    /* allocate sbuf ring as FreeRTOS message queue */
    mSBUF.hQueueSbuf = xQueueCreate(SBUF_RING_SIZE,
                                    sizeof(struct sbuf *));

    if (mSBUF.hQueueSbuf == NULL)
        return XST_FAILURE;

    /* fill ring with empty sbuf descriptors/pointers */
    for (size_t i = 0; i < sizeof(mSBUF.sbufSpace) / sizeof(mSBUF.sbufSpace[0]); i++) {
        struct sbuf *pSbuf = mSBUF.sbufSpace + i;

        if (xQueueSendToBack(mSBUF.hQueueSbuf, &pSbuf, 0) != pdTRUE)
            return XST_FAILURE;
    }

    return XST_SUCCESS;
}
