/**
 * @file sbrk.c
 * @brief link-override of Microblaze heap management
 *
 * Resetting the Microblaze softcore w/o reloading the FPGA fabric
 * is such an bizarre and uncommon use case that the Xilinx provided
 * libc does not handle it well: the heap management (sbrk and
 * newlib/mallocr) keep state in the .data section that must be
 * reset to its initial value after a non-fabric reset.
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

/* Copyright (c) 2004, 2009 Xilinx, Inc.  All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:

   1.  Redistributions source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

   2.  Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

   3.  Neither the name of Xilinx nor the names of its contributors may be
   used to endorse or promote products derived from this software without
   specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER AND CONTRIBUTORS "AS
   IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
   TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
   PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
   HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
   TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#include <errno.h>
#include <stddef.h>
#include <string.h>

#include <xparameters.h>

#include "logging.h"
#include "wd.h"

/* import text, data, heap, and stack definitions from the linker script */
extern char _start[];
extern char __data_start[];
extern char __data_end[];
extern char _heap_start[];
extern char _heap[];
extern char _heap_end[];
extern char _stack_end[];
extern char _stack[];

/** sbrk managed heap pointer: force it into .data section to avoid clobbering it when clearing .bss */
static char *heap_ptr __attribute__ ((section(".data"))) = NULL;

/**
 * reset heap management and restore .data after reset
 */
void
sbrkInit(void)
{
    const size_t dataSize  = (char *)&__data_end - (char *)&__data_start;
    char * const pDataCopy = (void *)(XPAR_LOCALMEMORY_DLMB_BRAM_IF_CNTLR_HIGHADDR + 1 - dataSize);

    if (pDataCopy < (char *)&_stack) {
        LOG_PRINT("* SBRK: cannot save .data\r\n");
        return;
    }

    if (wdResetType == WD_INIT) {
        /* This is an POR/fabric reset. Save static initialization .data to TOM */
        memcpy(pDataCopy, &__data_start, dataSize);
    } else {
        /* non-fabric reset: restore .data section from copy at TOM */
        const unsigned long wdResetTypeOrig = wdResetType;  /* save actual reset type */
        memcpy(&__data_start, pDataCopy, dataSize);
        wdResetType = wdResetTypeOrig;                      /* restore actual reset type */

        /* reset heap pointer */
        heap_ptr = (char *)&_heap;

        /* clear heap */
        // memset(heap_ptr, 0, (char *)&_heap_end - (char *)&_heap_start);
        __asm__ volatile (
            "            addi r6,  r0, _heap_start \n"
            "            addi r7,  r0, _heap_end   \n"
            "            rsub r18, r6, r7          \n"
            "            blei r18, .Lendheap       \n"
            ".Lloopheap: swi  r0,  r6, 0           \n"
            "            addi r6,  r6, 4           \n"
            "            rsub r18, r6, r7          \n"
            "            bgti r18, .Lloopheap      \n"
            ".Lendheap:                            \n"
        );
    }

    return;
}

/**
 * print current heap statistics
 */
void
sbrkStats(void)
{
    LOG_TRACE("# SBRK: .heap [@%p, @%p], sbrk @%p, size %lu, free %lu\r\n",
               &_heap_start, &_heap_end, heap_ptr, heap_ptr - (char *)&_heap_start, (char *)&_heap_end - heap_ptr);
}

char *
sbrk(int nbytes)
{
    char *base;

    if (!heap_ptr)
        heap_ptr = (char *)&_heap;

    base      = heap_ptr;
    heap_ptr += nbytes;

    if (heap_ptr <= ((char *)&_heap_end + 1))
        return base;
    else {
        errno = ENOMEM;
        return (char *)-1;
    }
}
