/**
 * @file pppIf.c
 *
 * @brief implementation details of the PPP UART interface
 * @author AIT
 * @copyright &copy;2023 Austrian Institute of Technology (AIT)
 */

#include <assert.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>

#include <FreeRTOS.h>
#include <queue.h>
#include <semphr.h>

#include <xparameters.h>
#include <xinterrupt_wrap.h>
#include <xstatus.h>
#include <xuartns550.h>

#include "fcs.h"
#include "gw.h"
#include "logging.h"
#include "powerlink.h"
#include "pppIf.h"
#include "sbuf.h"
#include "statistics.h"
#include "wd.h"

/** PPP frame characters/octets  */
enum PppFrameCharacters {
    PPP_FLAG        = 0x7e,     /**< PPP flag, RFC 1662 */
    PPP_ESC         = 0x7d,     /**< PPP asynchronous control escape, RFC 1662 */
    PPP_TRANS       = 0x20,     /**< PPP asynchronous transparency modifier */
    PPP_ALLSTATIONS = 0xff,     /**< PPP address field all-stations address */
    PPP_UI          = 0x03,     /**< PPP control field unnumbered information */
    SLIP_END        = 0xc0,     /**< SLIP frame end */
    SLIP_ESC        = 0xdb,     /**< SLIP frame escape */
    SLIP_ESC_END    = 0xdc,     /**< SLIP transposed frame end */
    SLIP_ESC_ESC    = 0xdd      /**< SLIP transposed frame escape */
};

/** PPP protocol field values */
enum PppProtocols {
    PPP_IP         = 0x0021,    /**< Internet Protocol */
    PPP_AT         = 0x0029,    /**< AppleTalk Protocol */
    PPP_IPX        = 0x002b,    /**< IPX protocol */
    PPP_VJC_COMP   = 0x002d,    /**< VJ compressed TCP */
    PPP_VJC_UNCOMP = 0x002f,    /**< VJ uncompressed TCP */
    PPP_IPV6       = 0x0057,    /**< Internet Protocol Version 6 */
    PPP_COMP       = 0x00fd,    /**< compressed packet */
    PPP_IPCP       = 0x8021,    /**< IP Control Protocol */
    PPP_ATCP       = 0x8029,    /**< AppleTalk Control Protocol */
    PPP_IPXCP      = 0x802b,    /**< IPX Control Protocol */
    PPP_IPV6CP     = 0x8057,    /**< IPv6 Control Protocol */
    PPP_CCP        = 0x80fd,    /**< Compression Control Protocol */
    PPP_ECP        = 0x8053,    /**< Encryption Control Protocol */
    PPP_LCP        = 0xc021,    /**< Link Control Protocol */
    PPP_PAP        = 0xc023,    /**< Password Authentication Protocol */
    PPP_LQR        = 0xc025,    /**< Link Quality Report protocol */
    PPP_CHAP       = 0xc223,    /**< Cryptographic Handshake Auth. Protocol */
    PPP_CBCP       = 0xc029,    /**< Callback Control Protocol */
    PPP_EAP        = 0xc227     /**< Extensible Authentication Protocol */
};

/** RFC 1661 §5 LCP (and IPCP) codes; also used for RFC 1332 IPCP */
enum LcpCodes {
    LCP_CONFREQ =  1,           /**< Configuration Request */
    LCP_CONFACK =  2,           /**< Configuration Ack */
    LCP_CONFNAK =  3,           /**< Configuration Nak */
    LCP_CONFREJ =  4,           /**< Configuration Reject */
    LCP_TERMREQ =  5,           /**< Termination Request */
    LCP_TERMACK =  6,           /**< Termination Ack */
    LCP_CODEREJ =  7,           /**< Code Reject */
    LCP_PROTREJ =  8,           /**< Protocol Reject */
    LCP_ECHOREQ =  9,           /**< Echo Request */
    LCP_ECHOREP = 10,           /**< Echo Reply */
    LCP_DISCREQ = 11,           /**< Discard Request */
    LCP_IDENTIF = 12,           /**< Identification */
    LCP_TIMEREM = 13            /**< Time Remaining */
};

/** PPP link states */
enum PppLinkStates {
    PPP_FSM_INITIAL  = 0,       /**< Down, hasn't been opened */
    PPP_FSM_STARTING = 1,       /**< Down, been opened */
    PPP_FSM_CLOSED   = 2,       /**< Up, hasn't been opened */
    PPP_FSM_STOPPED  = 3,       /**< Open, waiting for down event */
    PPP_FSM_CLOSING  = 4,       /**< Terminating the connection, not open */
    PPP_FSM_STOPPING = 5,       /**< Terminating, but open */
    PPP_FSM_REQSENT  = 6,       /**< We've sent a Config Request */
    PPP_FSM_ACKRCVD  = 7,       /**< We've received a Config Ack */
    PPP_FSM_ACKSENT  = 8,       /**< We've sent a Config Ack */
    PPP_FSM_OPENED   = 9        /**< Connection available */
};

/** PPP UART sbuf chain message */
struct MsgSbufChain {
    struct sbuf *p;             /**< address of head of sbuf chain */
    size_t       len;           /**< data length over complete chain */
};

/** PPP frame parser states */
enum ParserStatus {
    PDIDLE      = 0,            /**< Idle state - waiting. */
    PDADDRESS   = 1,            /**< Process address field. */
    PDCONTROL   = 2,            /**< Process control field. */
    PDPROTOCOL1 = 3,            /**< Process protocol field 1. */
    PDPROTOCOL2 = 4,            /**< Process protocol field 2. */
    PDDATA      = 5             /**< Process data byte. */
};

/** PPP frame parser state */
struct ParserState {
    enum ParserStatus  status;  /**< frame parser status */
    struct sbuf       *pHead;   /**< head of packet sbuf chain */
    struct sbuf       *pCurr;   /**< current sbuf */
    size_t             totLen;  /**< current total length over sbuf chain */
    size_t             ix;      /**< index into current sbuf */
    uint16_t           fcs;     /**< running FCS16 */
};

/** depth of PPP UART Tx task queue */
#define PPPIF_TX_TASK_QUEUE_LENGTH      SBUF_RING_SIZE

/** PPP UART Tx task stack size in StackType_t (uint32_t) words */
#define PPPIF_TX_TASK_STACK_SIZE        (2048 / sizeof(StackType_t))

/** PPP UART Tx task priority */
#define PPPIF_TX_TASK_PRIORITY          (tskIDLE_PRIORITY + 1)

/** depth of PPP UART Reassemble task queue */
#define PPPIF_RA_TASK_QUEUE_LENGTH      SBUF_RING_SIZE

/** PPP UART Reassemble task stack size in StackType_t (uint32_t) words */
#define PPPIF_RA_TASK_STACK_SIZE        (4096 / sizeof(StackType_t))

/** CDI2 UART Reassemble task priority */
#define PPPIF_RA_TASK_PRIORITY          (configMAX_PRIORITIES - 5)

/** PPP MTU/MRU is Protocol(2) + 1500 + FCS(2)*/
#define PPPIF_MTU 1504

/** PPPIF module instance state */
struct PppIfModule {
    SemaphoreHandle_t  hSemTx;              /**< PPP UART Tx serialization semaphore: Taken by the Tx task, given by the UART ISR */
    QueueHandle_t      hQueueTx;            /**< PPP UART Tx queue */
    QueueHandle_t      hQueueRa;            /**< PPP UART Reassemble queue */
    TaskHandle_t       hTaskTx;             /**< PPP UART Tx task */
    TaskHandle_t       hTaskRa;             /**< PPP UART Reassemble task */
    enum PppLinkStates lcpLinkState;        /**< PPP LCP link state */
    enum PppLinkStates ncpLinkState;        /**< PPP NCP link state */
    struct ParserState parserState;         /**< PPP frame parser state */
    XUartNs550         drvUart;             /**< PPP UART driver instance data */

    /** Rx data buffer */
    uint8_t rxBuffer[sizeof(((struct sbuf *)NULL)->data)];

    /** Tx data buffer: FLAG(1)|ADDRESS(1)|CONTROL(2)|DATA(2 * 60)|FCS(4)|FLAG(1) */
    uint8_t txBuffer[1 + 1 + 2 + 2 * sizeof(((struct sbuf *)NULL)->data) + 4 + 1];
} mPPPIF;

/**
 * indicate whether a given character must be RFC 1662 escaped
 *
 * @param[in] c character
 * @return true, if character must be escaped
 */
static inline bool
isEscape(uint8_t c)
{
    /** RFC 1662 sending asynchronous-control-character map */
    static const uint32_t defaultAsyncMap[8] = {
        0xffffffff,                             /* [0x00, 0x1F] */
        0x00000000,                             /* [0x20, 0x3F] */
        0x00000000,                             /* [0x40, 0x5F] */
        0x60000000,                             /* [0x60, 0x7F] */
        0x00000000,                             /* [0x80, 0x9F] */
        0x00000000,                             /* [0xA0, 0xBF] */
        0x00000000,                             /* [0xC0, 0xDF] */
        0x00000000                              /* [0xE0, 0xFF] */
    };

    return defaultAsyncMap[c >> 5] & (0x01 << (c & 0x1f));
}

/**
 * allow external access to UART statistics
 *
 * @return address of AXI UART error/statistic counters
 */
XUartNs550Stats *
pppIfGetStats(void)
{
    return &mPPPIF.drvUart.Stats;
}

/**
 * UART interrupt application event handler
 *
 * @param[in] arg callback reference from XUartNs550_SetHandler()
 * @param[in] ev callback event type
 * @param[in] evData callback event data
 */
static void
pppIfHandler(void *arg, uint32_t ev, unsigned int evData)
{
    /* unused */
    (void)arg;

    /* kick watchdog */
    wdKick();

    switch (ev) {
    case XUN_EVENT_RECV_DATA:
    case XUN_EVENT_RECV_TIMEOUT: {
        /* sanity */
        assert(evData <= sizeof(mPPPIF.rxBuffer));
        assert(mPPPIF.drvUart.ReceiveBuffer.NextBytePtr >= mPPPIF.rxBuffer);
        assert(mPPPIF.drvUart.ReceiveBuffer.NextBytePtr <= mPPPIF.rxBuffer + sizeof(mPPPIF.rxBuffer));

        if (evData <= 0)
            break;
        else if (evData > sizeof(mPPPIF.rxBuffer))
            evData = sizeof(mPPPIF.rxBuffer);

        /* keep collecting RX characters if sbuf not full and no PPP frame flag present */
        if ((evData < sizeof(mPPPIF.rxBuffer)) && (memchr(mPPPIF.rxBuffer, PPP_FLAG, evData) == NULL))
            break;

        /* forward to reassemble task */
        struct MsgSbufChain raMsg = {
                .p   = sbufGetIsr(),
                .len = evData
        };

        if (raMsg.p != NULL) {
            assert(raMsg.len > 0);
            assert(raMsg.len <= sizeof(mPPPIF.rxBuffer));

            memcpy(raMsg.p->data, mPPPIF.rxBuffer, raMsg.len);

            if ((mPPPIF.hQueueRa == NULL) || (xQueueSendFromISR(mPPPIF.hQueueRa, &raMsg, NULL) != pdTRUE)) {
                sbufPutIsr(raMsg.p);
                statsIncrPppIfRa();
            }
        } else
            statsIncrPppIfRa();

        /* reset RX data buffer */
        mPPPIF.drvUart.ReceiveBuffer.NextBytePtr    = mPPPIF.rxBuffer;
        mPPPIF.drvUart.ReceiveBuffer.RequestedBytes = sizeof(mPPPIF.rxBuffer);
        mPPPIF.drvUart.ReceiveBuffer.RemainingBytes = sizeof(mPPPIF.rxBuffer);

        break;
    }

    case XUN_EVENT_SENT_DATA:
        /* data buffer transmitted */
        assert(mPPPIF.drvUart.SendBuffer.NextBytePtr >= mPPPIF.txBuffer);
        assert(mPPPIF.drvUart.SendBuffer.NextBytePtr <= mPPPIF.txBuffer + sizeof(mPPPIF.txBuffer));

        /* release PPP UART Tx semaphore, intentionally ignore return value. */
        (void)xSemaphoreGiveFromISR(mPPPIF.hSemTx, NULL);
        break;

    case XUN_EVENT_RECV_ERROR: {
        /* sanity */
        assert(evData <= sizeof(mPPPIF.rxBuffer));
        assert(mPPPIF.drvUart.ReceiveBuffer.NextBytePtr >= mPPPIF.rxBuffer);
        assert(mPPPIF.drvUart.ReceiveBuffer.NextBytePtr <= mPPPIF.rxBuffer + sizeof(mPPPIF.rxBuffer));

        if (evData <= 0)
            break;
        else if (evData > sizeof(mPPPIF.rxBuffer))
            evData = sizeof(mPPPIF.rxBuffer);

        /* forward to reassemble task */
        struct MsgSbufChain raMsg = {
                .p   = sbufGetIsr(),
                .len = evData
        };

        if (raMsg.p != NULL) {
            assert(raMsg.len > 0);
            assert(raMsg.len <= sizeof(mPPPIF.rxBuffer));

            memcpy(raMsg.p->data, mPPPIF.rxBuffer, raMsg.len);

            if ((mPPPIF.hQueueRa == NULL) || (xQueueSendFromISR(mPPPIF.hQueueRa, &raMsg, NULL) != pdTRUE)) {
                sbufPutIsr(raMsg.p);
                statsIncrPppIfRa();
            }
        } else
            statsIncrPppIfRa();

        /* reset RX data buffer */
        mPPPIF.drvUart.ReceiveBuffer.NextBytePtr    = mPPPIF.rxBuffer;
        mPPPIF.drvUart.ReceiveBuffer.RequestedBytes = sizeof(mPPPIF.rxBuffer);
        mPPPIF.drvUart.ReceiveBuffer.RemainingBytes = sizeof(mPPPIF.rxBuffer);

        break;
    }

    case XUN_EVENT_MODEM:
        /* modem status lines changed */
        break;

    default:
        /* ignore unknown event codes */
        break;
    }
}

void
pppIfPollUartIsr()
{
    extern unsigned int XUartNs550_ReceiveBuffer(XUartNs550 *InstancePtr);

    /* save original position in mPPPIF.rxBuffer */
    const uint8_t *p0 = mPPPIF.drvUart.ReceiveBuffer.NextBytePtr;

    /* poll PPP UART */
    const unsigned int rxCnt = XUartNs550_ReceiveBuffer(&mPPPIF.drvUart);

    if (rxCnt > 0) {
        /* forward to reassembly task, if RX data buffer full or FLAG present */
        if ((mPPPIF.drvUart.ReceiveBuffer.RemainingBytes == 0) || (memchr(p0, PPP_FLAG, rxCnt) != NULL)) {
            struct MsgSbufChain raMsg = {
                    .p   = sbufGetIsr(),
                    .len = sizeof(mPPPIF.rxBuffer) - mPPPIF.drvUart.ReceiveBuffer.RemainingBytes
            };

            if (raMsg.p != NULL) {
                assert(raMsg.len > 0);
                assert(raMsg.len <= sizeof(mPPPIF.rxBuffer));

                memcpy(raMsg.p->data, mPPPIF.rxBuffer, raMsg.len);

                if ((mPPPIF.hQueueRa == NULL) || (xQueueSendFromISR(mPPPIF.hQueueRa, &raMsg, NULL) != pdTRUE)) {
                    sbufPutIsr(raMsg.p);
                    statsIncrPppIfRa();
                }
            } else
                statsIncrPppIfRa();

            /* reset RX data buffer */
            mPPPIF.drvUart.ReceiveBuffer.NextBytePtr    = mPPPIF.rxBuffer;
            mPPPIF.drvUart.ReceiveBuffer.RequestedBytes = sizeof(mPPPIF.rxBuffer);
            mPPPIF.drvUart.ReceiveBuffer.RemainingBytes = sizeof(mPPPIF.rxBuffer);
        }
    }
}

/**
 * send an RFC 1661 Configure-Request message to the peer
 *
 * @param[in] id Identifier to be used in this request
 */
static void
lcpConfigureRequest(uint8_t id)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 6);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1616 §5.1 Configure Request message */
    p->data[ix++] = (PPP_LCP >> 8) & 0xff;      /* Proto(2) */
    p->data[ix++] = PPP_LCP & 0xff;
    p->data[ix++] = LCP_CONFREQ;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2) */
    p->data[ix++] = 4;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * send an RFC 1661 Configure-Ack message to the peer
 *
 * @param[in] id Identifier to be used in this request
 */
static void
lcpConfigureAck(uint8_t id)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 6);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1616 §5.2 Configure Ack message */
    p->data[ix++] = (PPP_LCP >> 8) & 0xff;      /* Proto(2) */
    p->data[ix++] = PPP_LCP & 0xff;
    p->data[ix++] = LCP_CONFACK;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2) */
    p->data[ix++] = 4;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * send an RFC 1661 Configure-Reject message to the peer
 *
 * <DesignToCost>
 * We just reflect the first sbuf bytes of the rejected
 * configuration request back to the peer; This assumes
 * that all sane Configuration-Requests are well below 60
 * bytes
 * </DesignToCost>
 *
 * @param[in] id Identifier to be used in this request
 * @param[in] pHead sbuf chain holding PPP information field of rejected Configure-Request
 * @param[in] totLen size of PPP Information field
 */
static void
lcpConfigureReject(uint8_t id, struct sbuf *pHead, size_t totLen)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 6);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1616 §5.4 Configure Reject message */
    p->data[ix++] = (PPP_LCP >> 8) & 0xff;      /* Proto(2) */
    p->data[ix++] = PPP_LCP & 0xff;
    p->data[ix++] = LCP_CONFREJ;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2), actual length will be backpatched below */
    p->data[ix++] = 0;

    /* fill rest of sbuf with rejected options */
    for (size_t i = 6; (i < totLen) && (ix < sizeof(p->data));)
        p->data[ix++] = pHead->data[i++];

    /* fix length */
    const uint16_t len = ix - 2;
    p->data[4] = (len >> 8) & 0xff;
    p->data[5] = len & 0xff;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * send an RFC 1661 Code-Reject message to the peer
 *
 * <DesignToCost>
 * We just reflect the first sbuf bytes of the rejected
 * LCP packet back to the peer
 * </DesignToCost>
 *
 * @param[in] id Identifier to be used in this request
 * @param[in] pHead sbuf chain holding PPP information field of rejected Configure-Request
 * @param[in] totLen size of PPP Information field
 */
static void
lcpCodeReject(uint8_t id, struct sbuf *pHead, size_t totLen)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 6);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1616 §5.6 Code Reject message */
    p->data[ix++] = (PPP_LCP >> 8) & 0xff;      /* Proto(2) */
    p->data[ix++] = PPP_LCP & 0xff;
    p->data[ix++] = LCP_CODEREJ;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2), actual length will be backpatched below */
    p->data[ix++] = 0;

    /* fill rest of sbuf with rejected Information */
    for (size_t i = 0; (i < totLen) && (ix < sizeof(p->data));)
        p->data[ix++] = pHead->data[i++];

    /* fix length */
    const uint16_t len = ix - 2;
    p->data[4] = (len >> 8) & 0xff;
    p->data[5] = len & 0xff;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * send an RFC 1661 Protocol-Reject message to the peer
 *
 * <DesignToCost>
 * We just reflect the first sbuf bytes of the rejected
 * frame back to the peer
 * </DesignToCost>
 *
 * @param[in] proto rejected protocol code
 * @param[in] pHead sbuf chain holding PPP information field of rejected protocol
 * @param[in] totLen size of PPP information field
 */
static void
lcpProtocolReject(uint16_t proto, struct sbuf *pHead, size_t totLen)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 8);

    struct sbuf    *p  = sbufGet(true);
    size_t          ix = 0;
    static uint8_t  id = 0;

    /* build RFC 1616 §5.7 Protocol Reject message */
    p->data[ix++] = (PPP_LCP >> 8) & 0xff;      /* Proto(2) */
    p->data[ix++] = PPP_LCP & 0xff;
    p->data[ix++] = LCP_PROTREJ;                /* Code(1) */
    p->data[ix++] = id++;                       /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2), actual length will be backpatched below */
    p->data[ix++] = 0;
    p->data[ix++] = (proto >> 8) & 0xff;        /* Rejected-protocol(2) */
    p->data[ix++] = proto & 0xff;

    /* fill rest of sbuf with rejected Information */
    for (size_t i = 0; (i < totLen) && (ix < sizeof(p->data));)
        p->data[ix++] = pHead->data[i++];

    /* fix length */
    const uint16_t len = ix - 2;
    p->data[4] = (len >> 8) & 0xff;
    p->data[5] = len & 0xff;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * send an RFC 1661 Terminate-Ack message to the peer
 *
 * @param[in] id Identifier to be used in this request
 */
static void
lcpTerminateAck(uint8_t id)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 6);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1616 §5.5 Terminate Ack message */
    p->data[ix++] = (PPP_LCP >> 8) & 0xff;      /* Proto(2) */
    p->data[ix++] = PPP_LCP & 0xff;
    p->data[ix++] = LCP_TERMACK;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2) */
    p->data[ix++] = 4;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * send an RFC 1661 Echo-Reply message to the peer
 *
 * @param[in] id Identifier to be used in this request
 */
static void
lcpEchoReply(uint8_t id)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 10);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1616 §5.8 Echo Reply message */
    p->data[ix++] = (PPP_LCP >> 8) & 0xff;      /* Proto(2) */
    p->data[ix++] = PPP_LCP & 0xff;
    p->data[ix++] = LCP_ECHOREP;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2) */
    p->data[ix++] = 8;
    p->data[ix++] = 0;                          /* Magic-Number(4) */
    p->data[ix++] = 0;
    p->data[ix++] = 0;
    p->data[ix++] = 0;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * process a PPP RFC 1661 LCP frame
 *
 * @param[in] pHead sbuf chain holding PPP information field data
 * @param[in] totLen size of PPP information field
 */
static void
pppIfLCP(struct sbuf *pHead, size_t totLen)
{
    static uint8_t id = 0;

    assert(pHead != NULL);
    assert(totLen > 0);

    switch (pHead->data[2]) {
    case LCP_CONFREQ:
        if (totLen != 6)
            /* reject everything containing options */
            lcpConfigureReject(pHead->data[3], pHead, totLen);
        else {
            /* accept default configuration */
            lcpConfigureAck(pHead->data[3]);

            if (mPPPIF.lcpLinkState == PPP_FSM_ACKRCVD) {
                mPPPIF.lcpLinkState = PPP_FSM_OPENED;
            } else {
                lcpConfigureRequest(id);
                mPPPIF.lcpLinkState = PPP_FSM_REQSENT;
            }
        }

        break;

    case LCP_CONFACK:
        if ((totLen != 6) || (pHead->data[3] != id)) {
            /* request default configuration */
            lcpConfigureRequest(++id);
            mPPPIF.lcpLinkState = PPP_FSM_REQSENT;
        } else {
            mPPPIF.lcpLinkState = PPP_FSM_ACKRCVD;
        }

        break;

    case LCP_TERMREQ:
        lcpTerminateAck(pHead->data[3]);
        mPPPIF.lcpLinkState = PPP_FSM_INITIAL;
        break;

    case LCP_TERMACK:
        /* reopen with request of default configuration */
        lcpConfigureRequest(id);
        mPPPIF.lcpLinkState = PPP_FSM_REQSENT;
        break;

    case LCP_ECHOREQ:
        lcpEchoReply(pHead->data[3]);
        break;

    case LCP_CONFNAK:
    case LCP_CONFREJ:
    case LCP_CODEREJ:
    case LCP_PROTREJ:
        /* deliberately ignore rejections: we are the minimum viable implementation already */
        id++;
        break;

    case LCP_ECHOREP:
    case LCP_DISCREQ:
        /* discard silently */
        break;

    case LCP_IDENTIF:
    case LCP_TIMEREM:
    default:
        /* reject unknown/not implemented LCP code */
        lcpCodeReject(id++, pHead, totLen);
        break;
    }
}

/**
 * send an RFC 1332 Configure-Request message to the peer
 *
 * @param[in] id Identifier to be used in this request
 */
static void
ipcpConfigureRequest(uint8_t id)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 6);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1332 Configure Request message */
    p->data[ix++] = (PPP_IPCP >> 8) & 0xff;     /* Proto(2) */
    p->data[ix++] = PPP_IPCP & 0xff;
    p->data[ix++] = LCP_CONFREQ;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2) */
    p->data[ix++] = 10;
    p->data[ix++] = 3;                          /* Type(1) */
    p->data[ix++] = 6;                          /* Length(1) */
    p->data[ix++] = 192;                        /* IP-Address(4) */
    p->data[ix++] = 168;
    p->data[ix++] = 100;
    p->data[ix++] = 254;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * send an RFC 1332 Configure-Ack message to the peer
 *
 * @param[in] id Identifier to be used in this request
 * @param[in] pHead sbuf chain holding PPP information field with acknowledged options
 * @param[in] totLen size of PPP Information field
 */
static void
ipcpConfigureAck(uint8_t id, struct sbuf *pHead, size_t totLen)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 6);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1332 Configure Ack message */
    p->data[ix++] = (PPP_IPCP >> 8) & 0xff;     /* Proto(2) */
    p->data[ix++] = PPP_IPCP & 0xff;
    p->data[ix++] = LCP_CONFACK;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2), actual length will be backpatched below */
    p->data[ix++] = 0;

    /* fill rest of sbuf with accepted options */
    for (size_t i = 6; (i < totLen) && (ix < sizeof(p->data));)
        p->data[ix++] = pHead->data[i++];

    /* fix length */
    const uint16_t len = ix - 2;
    p->data[4] = (len >> 8) & 0xff;
    p->data[5] = len & 0xff;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * send an RFC 1332 Configure-Reject message to the peer
 *
 * <DesignToCost>
 * We just reflect the first sbuf bytes of the rejected
 * configuration request back to the peer; This assumes
 * that all sane Configuration-Requests are well below 60
 * bytes
 * </DesignToCost>
 *
 * @param[in] id Identifier to be used in this request
 * @param[in] pHead sbuf chain holding PPP information field of rejected Configure-Request
 * @param[in] totLen size of PPP Information field
 */
static void
ipcpConfigureReject(uint8_t id, struct sbuf *pHead, size_t totLen)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 6);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1332 Configure Reject message */
    p->data[ix++] = (PPP_IPCP >> 8) & 0xff;     /* Proto(2) */
    p->data[ix++] = PPP_IPCP & 0xff;
    p->data[ix++] = LCP_CONFREJ;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2), actual length will be backpatched below */
    p->data[ix++] = 0;

    /* fill rest of sbuf with rejected options */
    for (size_t i = 6; (i < totLen) && (ix < sizeof(p->data));)
        p->data[ix++] = pHead->data[i++];

    /* fix length */
    const uint16_t len = ix - 2;
    p->data[4] = (len >> 8) & 0xff;
    p->data[5] = len & 0xff;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * send an RFC 1332 Terminate-Ack message to the peer
 *
 * @param[in] id Identifier to be used in this request
 */
static void
ipcpTerminateAck(uint8_t id)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 6);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1332 Terminate Ack message */
    p->data[ix++] = (PPP_IPCP >> 8) & 0xff;     /* Proto(2) */
    p->data[ix++] = PPP_IPCP & 0xff;
    p->data[ix++] = LCP_TERMACK;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2) */
    p->data[ix++] = 4;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * send an RFC 1332 Code-Reject message to the peer
 *
 * <DesignToCost>
 * We just reflect the first sbuf bytes of the rejected
 * IPCP packet back to the peer
 * </DesignToCost>
 *
 * @param[in] id Identifier to be used in this request
 * @param[in] pHead sbuf chain holding PPP information field of rejected request
 * @param[in] totLen size of PPP Information field
 */
static void
ipcpCodeReject(uint8_t id, struct sbuf *pHead, size_t totLen)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 6);

    struct sbuf *p  = sbufGet(true);
    size_t       ix = 0;

    /* build RFC 1332 Code Reject message */
    p->data[ix++] = (PPP_IPCP >> 8) & 0xff;     /* Proto(2) */
    p->data[ix++] = PPP_IPCP & 0xff;
    p->data[ix++] = LCP_CODEREJ;                /* Code(1) */
    p->data[ix++] = id;                         /* Identifier(1) */
    p->data[ix++] = 0;                          /* Length(2), actual length will be backpatched below */
    p->data[ix++] = 0;

    /* fill rest of sbuf with rejected Information */
    for (size_t i = 0; (i < totLen) && (ix < sizeof(p->data));)
        p->data[ix++] = pHead->data[i++];

    /* fix length */
    const uint16_t len = ix - 2;
    p->data[4] = (len >> 8) & 0xff;
    p->data[5] = len & 0xff;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = p,
        .len = ix
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(p);
        statsIncrPppIfTx();
    }
}

/**
 * process a PPP RFC 1332 IPCP frame
 *
 * @param[in] pHead sbuf chain holding PPP information field data
 * @param[in] totLen size of PPP information field
 */
static void
pppIfIPCP(struct sbuf *pHead, size_t totLen)
{
    static uint8_t id = 0;

    assert(pHead != NULL);
    assert(totLen > 0);

    switch (pHead->data[2]) {
    case LCP_CONFREQ:
        if ((totLen != 12) && (pHead->data[6] != 3))
            /* reject everything containing options other than IP-Address */
            ipcpConfigureReject(pHead->data[3], pHead, totLen);
        else {
            /* accept configuration */
            ipcpConfigureAck(pHead->data[3], pHead, totLen);

            if (mPPPIF.ncpLinkState == PPP_FSM_ACKRCVD) {
                mPPPIF.ncpLinkState = PPP_FSM_OPENED;
            } else {
                /* request default configuration */
                ipcpConfigureRequest(id);
                mPPPIF.ncpLinkState = PPP_FSM_REQSENT;
            }
        }

        break;

    case LCP_CONFACK:
        mPPPIF.ncpLinkState = PPP_FSM_ACKRCVD;
        break;

    case LCP_TERMREQ:
        ipcpTerminateAck(pHead->data[3]);
        mPPPIF.ncpLinkState = PPP_FSM_INITIAL;
        break;

    case LCP_TERMACK:
        /* reopen and request default configuration */
        ipcpConfigureRequest(id);
        mPPPIF.ncpLinkState = PPP_FSM_REQSENT;
        break;

    case LCP_CONFNAK:
    case LCP_CONFREJ:
    case LCP_CODEREJ:
        /* deliberately ignore rejections: we are the minimum viable implementation already */
        id++;
        break;

    default:
        /* reject unknown/not implemented IPCP code */
        ipcpCodeReject(id++, pHead, totLen);
        break;
    }
}

void
pppIfFwdIP(uint8_t *pIpData, size_t len)
{
    /* sanity */
    assert(pIpData != NULL);
    assert(len > 0);

    /* drop MAC frame, if sbuf queue nearly full */
    if ((sbufAvailable() * sbufDataSize) < len)
        return;

    /* initialize sbuf chain head */
    struct sbuf *pHead  = sbufGet(true);
    struct sbuf *pLast  = pHead;
    size_t       totLen = len + 2;

    /* put PPP protocol field into head of sbuf chain */
    pHead->data[0] = (PPP_IP >> 8) & 0xff;
    pHead->data[1] = PPP_IP & 0xff;

    /* fill rest of header sbuf with IP datagram data */
    size_t copyLen = len > (sizeof(pHead->data) - 2) ? (sizeof(pHead->data) - 2) : len;
    memcpy(pHead->data + 2, pIpData, copyLen);

    /* adjust source pointer and length */
    len     -= copyLen;
    pIpData += copyLen;

    /* copy rest of datagram into sbuf chain */
    while (len > 0) {
        pLast->pNext = sbufGet(true);
        pLast        = pLast->pNext;
        copyLen      = len > sizeof(pLast->data) ? sizeof(pLast->data) : len;

        memcpy(&pLast->data, pIpData, copyLen);

        /* adjust source pointer and remaining length */
        len     -= copyLen;
        pIpData += copyLen;
    }

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pHead,
        .len = totLen
    };

    if ((mPPPIF.hQueueTx == NULL) || (xQueueSend(mPPPIF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pHead);
        statsIncrPppIfTx();
    }
}

/**
 * PPP UART Tx task
 *
 * @param[in] arg task parameter passed via xTaskCreate()
 */
static void
pppIfTxTask(void *arg)
{
    /* unused */
    (void)arg;

    LOG_TRACE("# PPPIF-TX: started\r\n");

    /* start PPP LCP negotiation */
    lcpConfigureRequest(0);
    mPPPIF.lcpLinkState = PPP_FSM_REQSENT;

    /* FOREVER */
    for (;;) {
        /* receive Tx request */
        struct MsgSbufChain req;

        if (xQueueReceive(mPPPIF.hQueueTx, &req, portMAX_DELAY) == pdFALSE)
            /* time-out, nothing received */
            continue;

        /* sanity */
        assert(req.p != NULL);
        assert(req.len > 0);

        if (req.p == NULL)
            continue;

        if (req.len <= 0) {
            sbufPut(req.p);
            continue;
        }

        /* non-empty frame assured: serialize Tx request at UART Tx semaphore */
        xSemaphoreTake(mPPPIF.hSemTx, portMAX_DELAY);

        /* UART Tx buffer is now free and empty, start with header: FLAG|ADDRESS|CONTROL */
        size_t   txBufIx = 0;
        uint16_t fcs     = PPP_INITFCS16;

        mPPPIF.txBuffer[txBufIx++] = PPP_FLAG;
        mPPPIF.txBuffer[txBufIx++] = PPP_ALLSTATIONS;
        mPPPIF.txBuffer[txBufIx++] = PPP_ESC;
        mPPPIF.txBuffer[txBufIx++] = PPP_UI ^ PPP_TRANS;
        fcs                 = fcs16Incr(PPP_ALLSTATIONS, fcs);
        fcs                 = fcs16Incr(PPP_UI, fcs);

        /* transmit sbuf chain as PPP frame Information field */
        while ((req.p != NULL ) && (req.len > 0)) {
            /* calculate length of Tx segment */
            size_t txLen = 0;

            if (req.len > sizeof(req.p->data))
                txLen = sizeof(req.p->data);
            else
                txLen = req.len;

            /* marshall/transmit txLen worth of data */
            for (size_t i = 0; i < txLen; i++) {
                if (isEscape(req.p->data[i])) {
                    mPPPIF.txBuffer[txBufIx++] = PPP_ESC;
                    mPPPIF.txBuffer[txBufIx++] = req.p->data[i] ^ PPP_TRANS;
                } else {
                    mPPPIF.txBuffer[txBufIx++] = req.p->data[i];
                }

                fcs = fcs16Incr(req.p->data[i], fcs);
            }

            /* adjust/decrease total/remaining length of sbuf chain for amount of sent data */
            req.len -= txLen;

            /* get transmitted sbuf off the chain and return to ring */
            struct sbuf *p = req.p;
            req.p          = req.p->pNext;
            p->pNext       = NULL;
            sbufPut(p);

            if (req.len <= 0) {
                /* last Information byte written to buffer, append RFC 1662 trailer: FCS16|FLAG */
                fcs                  = ~fcs;
                const uint8_t fcsLsb = fcs & 0xff;
                const uint8_t fcsMsb = (fcs >> 8) & 0xff;

                if (isEscape(fcsLsb)) {
                    mPPPIF.txBuffer[txBufIx++] = PPP_ESC;
                    mPPPIF.txBuffer[txBufIx++] = fcsLsb ^ PPP_TRANS;
                } else {
                    mPPPIF.txBuffer[txBufIx++] = fcsLsb;
                }

                if (isEscape(fcsMsb)) {
                    mPPPIF.txBuffer[txBufIx++] = PPP_ESC;
                    mPPPIF.txBuffer[txBufIx++] = fcsMsb ^ PPP_TRANS;
                } else {
                    mPPPIF.txBuffer[txBufIx++] = fcsMsb;
                }

                mPPPIF.txBuffer[txBufIx++] = PPP_FLAG;

                /* transmit last Tx buffer */
                (void)XUartNs550_Send(&mPPPIF.drvUart, mPPPIF.txBuffer, txBufIx);
                txBufIx = 0;

                break;
            } else {
                /* transmit and serialize Tx request at UART Tx semaphore */
                (void)XUartNs550_Send(&mPPPIF.drvUart, mPPPIF.txBuffer, txBufIx);
                txBufIx = 0;

                xSemaphoreTake(mPPPIF.hSemTx, portMAX_DELAY);
            }
        }

        /* marshalling may add redundant sbuf to the chain; silently remove them now */
        sbufPut(req.p);
    }
}

/**
 * process a PPP flag character in the current frame parser state
 *
 * @param[in] pParserState PPP frame parser state
 */
static void
pppIfReassembleFlag(struct ParserState *pParserState)
{
    /* sanity */
    assert(pParserState != NULL);

    switch (pParserState->status) {
    case PDIDLE:
        /* sanity */
        assert(pParserState->fcs == PPP_INITFCS16);

        /* fallthrough */

    case PDADDRESS:
    case PDCONTROL:
    case PDPROTOCOL1:
        /* sanity */
        assert(pParserState->pHead  == NULL);
        assert(pParserState->pCurr  == NULL);
        assert(pParserState->totLen == 0);
        assert(pParserState->ix     == 0);

        /* goto PDADDRESS */
        pParserState->fcs    = PPP_INITFCS16;
        pParserState->status = PDADDRESS;
        break;

    case PDPROTOCOL2:
        /* sanity */
        assert(pParserState->pHead  != NULL);
        assert(pParserState->pCurr  != NULL);
        assert(pParserState->totLen <= 2);
        assert(pParserState->ix     <= 2);

        /* silently drop short frame */
        sbufPut(pParserState->pHead);

        pParserState->fcs    = PPP_INITFCS16;
        pParserState->totLen = 0;
        pParserState->ix     = 0;
        pParserState->pHead  = NULL;
        pParserState->pCurr  = NULL;

        /* goto PDADDRESS */
        pParserState->status = PDADDRESS;
        break;

    case PDDATA:
        /* sanity */
        assert(pParserState->pHead  != NULL);
        assert(pParserState->totLen >= 1);

        if (pParserState->fcs != PPP_GOODFCS16) {
            /* drop frame with FCS error */
            logMsg(WARN, ERR_APPLICATION, "PPPIF-RA: PPP frame with CRC error");

            sbufPut(pParserState->pHead);

            pParserState->fcs    = PPP_INITFCS16;
            pParserState->totLen = 0;
            pParserState->ix     = 0;
            pParserState->pHead  = NULL;
            pParserState->pCurr  = NULL;
        } else {
            /* strip FCS16 from DATA */
            pParserState->totLen -= 2;

            /* always assume 16 bit protocol field in network byte order */
            uint16_t proto = (pParserState->pHead->data[0] << 8) | pParserState->pHead->data[1];

            switch (proto) {
            case PPP_LCP:
                pppIfLCP(pParserState->pHead, pParserState->totLen);

                /* release processed frame */
                sbufPut(pParserState->pHead);
                break;

            case PPP_IPCP:
                pppIfIPCP(pParserState->pHead, pParserState->totLen);

                /* release processed frame */
                sbufPut(pParserState->pHead);
                break;

            case PPP_IP:
                plkReqFwdIp(pParserState->pHead, pParserState->totLen);
                break;

            default:
                lcpProtocolReject(proto, pParserState->pHead, pParserState->totLen);

                /* release processed frame */
                sbufPut(pParserState->pHead);
                break;
            }

            pParserState->fcs    = PPP_INITFCS16;
            pParserState->totLen = 0;
            pParserState->ix     = 0;
            pParserState->pHead  = NULL;
            pParserState->pCurr  = NULL;
        }

        /* always goto PDADDRESS after FLAG */
        pParserState->status = PDADDRESS;
        break;

    default:
        break;
    }
}

/**
 * append/push received character to PPP frame
 *
 * @param[in] c received byte
 * @param[in] pParserState PPP frame parser state
 */
static inline void
pppIfPushByte(uint8_t   c,
             struct ParserState *pParserState)
{
    /* sanity */
    assert(pParserState        != NULL);
    assert(pParserState->pCurr != NULL);
    assert(pParserState->ix     < sizeof(pParserState->pCurr->data));

    /* push message byte */
    pParserState->pCurr->data[pParserState->ix++] = c;
    pParserState->totLen++;

    /* update FCS16 */
    pParserState->fcs = fcs16Incr(c, pParserState->fcs);

    /* grow sbuf chain, if current buffer full */
    if (pParserState->ix >= sizeof(pParserState->pCurr->data)) {
        pParserState->pCurr->pNext = sbufGet(true);
        pParserState->pCurr        = pParserState->pCurr->pNext;
        pParserState->ix           = 0;
    }
}

/**
 * parse a PPP UART Rx character stream into RFC 1662 packet, i.e.,
 * the HDLC fame overhead (FLAG|ADDRESS|CONTROL|...|FCS|FLAG) is removed
 *
 * @param[in] c input character/octett
 * @param[in] pParserState PPP frame parser state
 */
static void
pppIfReassemble(uint8_t             c,
                struct ParserState *pParserState)
{
    /* sanity */
    assert(pParserState != NULL);

    switch (pParserState->status) {
    case PDIDLE:
        /* drop out-of frame characters */
        break;

    case PDADDRESS:
        if (c == PPP_ALLSTATIONS) {
            /* update FCS16 */
            pParserState->fcs = fcs16Incr(PPP_ALLSTATIONS, PPP_INITFCS16);

            /* goto PDCONTROL */
            pParserState->status = PDCONTROL;
        } else {
            /* drop invalid frame  */
            sbufPut(pParserState->pHead);

            pParserState->fcs    = PPP_INITFCS16;
            pParserState->totLen = 0;
            pParserState->ix     = 0;
            pParserState->pHead  = NULL;
            pParserState->pCurr  = NULL;

            /* goto PDIDLE */
            pParserState->status = PDIDLE;
        }

        break;

    case PDCONTROL:
        if (c == PPP_UI) {
            /* update FCS16 */
            pParserState->fcs = fcs16Incr(PPP_UI, pParserState->fcs);

            /* goto PDPROTOCOL1 */
            pParserState->status = PDPROTOCOL1;
        } else {
            /* drop invalid frame  */
            sbufPut(pParserState->pHead);

            pParserState->fcs    = PPP_INITFCS16;
            pParserState->totLen = 0;
            pParserState->ix     = 0;
            pParserState->pHead  = NULL;
            pParserState->pCurr  = NULL;

            /* goto PDIDLE */
            pParserState->status = PDIDLE;
        }

        break;

    case PDPROTOCOL1:
        /* start PPP frame information field reassembly */
        pParserState->totLen = 0;
        pParserState->ix     = 0;
        pParserState->pHead  = sbufGet(true);
        pParserState->pCurr  = pParserState->pHead;

        pppIfPushByte(c, pParserState);

        /* if LSB set, this is the last byte of the protocol field */
        pParserState->status = (c & 0x01) ? PDDATA : PDPROTOCOL2;

        break;

    case PDPROTOCOL2:
        /* 2-byte protocol fields are transmitted in big endian/network byte order: LSB must be set now */
        if (c & 0x01) {
            pppIfPushByte(c, pParserState);

            /* valid protocol field, goto PDDATA */
            pParserState->status = PDDATA;
        } else {
            /* drop invalid frame  */
            sbufPut(pParserState->pHead);

            pParserState->fcs    = PPP_INITFCS16;
            pParserState->totLen = 0;
            pParserState->ix     = 0;
            pParserState->pHead  = NULL;
            pParserState->pCurr  = NULL;

            /* goto PDIDLE */
            pParserState->status = PDIDLE;
        }
        break;

    case PDDATA:
        /* MTU/MRU check */
        if (pParserState->totLen > PPPIF_MTU) {
            /* drop oversized frame */
            logMsg(WARN, ERR_APPLICATION, "PPPIF-RA: oversized PPP frame");

            /* drop invalid frame  */
            sbufPut(pParserState->pHead);

            pParserState->fcs    = PPP_INITFCS16;
            pParserState->totLen = 0;
            pParserState->ix     = 0;
            pParserState->pHead  = NULL;
            pParserState->pCurr  = NULL;

            /* goto PDIDLE */
            pParserState->status = PDIDLE;
        } else {
            pppIfPushByte(c, pParserState);

            /* stay in PDDATA */
        }

        break;

    default:
        break;
    }
}

/**
 * PPP UART Reassemble task
 *
 * @param arg task parameter passed via xTaskCreate()
 */
static void
pppIfRaTask(void *arg)
{
    /* unused */
    (void)arg;

    LOG_TRACE("# PPPIF-RA: started\r\n");

    /* data link layer state: PPP_ESC scanned */
    bool inEscape = false;

    /* FOREVER */
    for (;;) {
        /* receive reassembly request */
        struct MsgSbufChain req;

        if (xQueueReceive(mPPPIF.hQueueRa, &req, portMAX_DELAY) == pdFALSE)
            /* time-out, nothing received */
            continue;

        /* sanity */
        assert(req.p != NULL);
        assert(req.len > 0);

        /* feed sbuf chain into reassembler */
        for (struct sbuf *p = req.p; (p != NULL) && (req.len > 0); p = p->pNext) {
            /* calculate length of reassemble segment */
            size_t raLen = 0;

            if (req.len > sizeof(p->data))
                raLen = sizeof(p->data);
            else
                raLen = req.len;

            /* parse character stream */
            for (unsigned int i = 0; i < raLen; i++) {
                if (inEscape) {
                    pppIfReassemble(p->data[i] ^ PPP_TRANS, &mPPPIF.parserState);
                    inEscape = false;
                } else if (p->data[i] == PPP_ESC)
                    inEscape = true;
                else if (p->data[i] == PPP_FLAG)
                    pppIfReassembleFlag(&mPPPIF.parserState);
                else
                    pppIfReassemble(p->data[i], &mPPPIF.parserState);
            }

            /* adjust received length for processed data */
            req.len -= raLen;
        }

        /* release sbuf chain */
        sbufPut(req.p);
    }
}

int
pppIfSetup(unsigned long speed)
{
    int ret = XST_SUCCESS;

    /* sanity */
    assert(speed > 0);

    if (speed <= 0)
        return XST_UART_BAUD_RANGE;

    LOG_TRACE("# PPPIF: setup PPP UART\r\n");

    /* reset frame parser state */
    mPPPIF.parserState.status = PDIDLE;
    mPPPIF.parserState.pHead  = NULL;
    mPPPIF.parserState.pCurr  = NULL;
    mPPPIF.parserState.totLen = 0;
    mPPPIF.parserState.ix     = 0;
    mPPPIF.parserState.fcs    = PPP_INITFCS16;

    /* reset link layer protocol states */
    mPPPIF.lcpLinkState = PPP_FSM_INITIAL;
    mPPPIF.ncpLinkState = PPP_FSM_INITIAL;

    /* initialize PPP UART driver instance data */
    memset(&mPPPIF.drvUart, 0, sizeof(mPPPIF.drvUart));

    if ((ret = XUartNs550_Initialize(&mPPPIF.drvUart, XPAR_PPP_UART_DEVICE_ID)) != XST_SUCCESS)
        return ret;

    /* driver self test */
    if ((ret = XUartNs550_SelfTest(&mPPPIF.drvUart)) != XST_SUCCESS)
        return ret;

    /* prepare RX data buffer */
    mPPPIF.drvUart.ReceiveBuffer.RequestedBytes = sizeof(mPPPIF.rxBuffer);
    mPPPIF.drvUart.ReceiveBuffer.RemainingBytes = sizeof(mPPPIF.rxBuffer);
    mPPPIF.drvUart.ReceiveBuffer.NextBytePtr    = mPPPIF.rxBuffer;

    /* configure speed, 8N1 format */
    XUartNs550Format format = {
            .BaudRate = speed,
            .DataBits = XUN_FORMAT_8_BITS,
            .Parity   = XUN_FORMAT_NO_PARITY,
            .StopBits = XUN_FORMAT_1_STOP_BIT
    };

    if ((ret = XUartNs550_SetDataFormat(&mPPPIF.drvUart, &format)) != XST_SUCCESS)
        return ret;

    XUartNs550_GetDataFormat(&mPPPIF.drvUart, &format);

    LOG_TRACE("# PPPIF: 8N1 format, speed %u bps\r\n", format.BaudRate);

    /* connect PPP UART interrupt to UART driver handler */
    LOG_TRACE("# PPPIF: register UART interrupt handler\r\n");

    /* Register UART driver interrupt callback for the PPP UART */
    XConnectToInterruptCntrl(XPAR_INTERRUPTCONTROLLER_PPP_UART_IP2INTC_IRPT_INTR,
                             (XInterruptHandler)XUartNs550_InterruptHandler,
                             (void *)&mPPPIF.drvUart,
                             XPAR_TICKTIMER_INTR_PARENT);
    XEnableIntrId(XPAR_INTERRUPTCONTROLLER_PPP_UART_IP2INTC_IRPT_INTR,
                  XPAR_TICKTIMER_INTR_PARENT);

    /* register application UART event handler */
    XUartNs550_SetHandler(&mPPPIF.drvUart, pppIfHandler, &mPPPIF.drvUart);

    /* enable data rx/tx interrupts, clear & enable FIFOs */
    XUartNs550_SetOptions(&mPPPIF.drvUart, XUN_OPTION_DATA_INTR |
                                           XUN_OPTION_RXLINE_INTR |
                                           XUN_OPTION_FIFOS_ENABLE |
                                           XUN_OPTION_RESET_TX_FIFO |
                                           XUN_OPTION_RESET_RX_FIFO);

    /* set FIFO thresholds (default 8 characters) */
    XUartNs550_SetFifoThreshold(&mPPPIF.drvUart, XUN_FIFO_TRIGGER_08);

    LOG_TRACE("# PPPIF: setup TX task, queue %u x |%lu|\r\n", PPPIF_TX_TASK_QUEUE_LENGTH, sizeof(struct MsgSbufChain));

    /* create PPP UART Tx semaphore and Give() it initially */
    mPPPIF.hSemTx = xSemaphoreCreateBinary();

    assert(mPPPIF.hSemTx != NULL);

    if (mPPPIF.hSemTx == NULL)
        return XST_FAILURE;

    (void)xSemaphoreGive(mPPPIF.hSemTx);           /* intentionally ignore return */

    /* create PPP UART Tx task queue */
    mPPPIF.hQueueTx = xQueueCreate(PPPIF_TX_TASK_QUEUE_LENGTH,
                                   sizeof(struct MsgSbufChain));

    assert(mPPPIF.hQueueTx != NULL);

    if (mPPPIF.hQueueTx == NULL)
        return XST_FAILURE;

    /* create PPP UART Tx task */
    xTaskCreate(pppIfTxTask,
                "PPPIF-TX",
                PPPIF_TX_TASK_STACK_SIZE,
                NULL,
                PPPIF_TX_TASK_PRIORITY,
                &mPPPIF.hTaskTx);

    assert(mPPPIF.hTaskTx != NULL);

    if (mPPPIF.hTaskTx == NULL)
        return XST_FAILURE;

    LOG_TRACE("# PPPIF: setup ReAssemble task, queue %u x |%lu|\r\n", PPPIF_RA_TASK_QUEUE_LENGTH, sizeof(struct MsgSbufChain));

    /* create PPP UART Reassemble task queue */
    mPPPIF.hQueueRa = xQueueCreate(PPPIF_RA_TASK_QUEUE_LENGTH,
                                   sizeof(struct MsgSbufChain));

    assert(mPPPIF.hQueueRa != NULL);

    if (mPPPIF.hQueueRa == NULL)
        return XST_FAILURE;

    /* create PPP UART Reassemble task */
    xTaskCreate(pppIfRaTask,
                "PPPIF-RA",
                PPPIF_RA_TASK_STACK_SIZE,
                NULL,
                PPPIF_RA_TASK_PRIORITY,
                &mPPPIF.hTaskRa);

    assert(mPPPIF.hTaskRa != NULL);

    if (mPPPIF.hTaskRa == NULL)
        return XST_FAILURE;

    return XST_SUCCESS;
}
