/**
 * @file powerlink.c
 * @brief CDI2 PowerLink layer; CN/Device variant
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>

#include <FreeRTOS.h>
#include <queue.h>
#include <task.h>
#include <timers.h>

#include <xparameters.h>
#include <xstatus.h>
#include <xiltimer.h>

#include <oplkcfg.h>
#include <oplk/oplk.h>
#include <target/openmac.h>
#include <systemtimer.h>
#include <omethlib_phycfg.h>

#include "cdi2.h"
#include "cdi2If.h"
#include "gw.h"
#include "logging.h"
#include "object_dictionary.h"
#include "powerlink.h"
#include "pppIf.h"
#include "sbuf.h"
#include "statistics.h"
#include "tts.h"
#include "wd.h"

/** length/depth of PowerLink layer request queue */
#define PLK_REQUEST_QUEUE_LENGTH 16

/** length/depth of PowerLink layer BN recognition queue */
#define PLK_BNRECOGNITION_QUEUE_LENGTH 20

/** length/depth of PowerLink layer BN result queue */
#define PLK_BNRESULT_QUEUE_LENGTH 20

/** PowerLink layer task stack size in StackType_t (uint32_t) words*/
#define PLK_TASK_STACK_SIZE (10240 / sizeof(StackType_t))

/** PowerLink layer task priority */
#define PLK_TASK_PRIORITY (configMAX_PRIORITIES - 2)

/**
 * PowerLink BNRESULT queue element/message
 * @see cdi2if.c struct MsgSbufChain
 */
struct PlkBnResultMessage {
    struct sbuf *p;                                 /**< address of head of sbuf chain */
    size_t       len;                               /**< data length over complete chain */
};

/** PLK module instance state */
struct PlkModule {
    QueueHandle_t        hQueueRequest;             /**< handle of PowerLink layer request queue */
    QueueHandle_t        hQueueBnRecognition;       /**< handle of PowerLink layer message queue */
    QueueHandle_t        hQueueBnResult;            /**< handle of PowerLink layer message queue */
    TaskHandle_t         hTaskPlk;                  /**< handle of PowerLink layer task */
    uint32_t             lastSeqNo;                 /**< last received BSMINFO sequence number */
    uint32_t             use2ndBfa;                 /**< trigger BFA#2 using BSMINFO slot 16 */
    uint8_t              nodeId;                    /**< configured CN nodeId */
    uint8_t              lastBsmState;              /**< last known BSM status */
    tNmtState            lastNmtState;              /**< last known NMT status */
    enum UpdateSwCommand lastSwuCmd;                /**< last polled UpdateSwCommand SDO */
    bool                 isPlkStarted;              /**< application has started the PowerLink stack */
} mPLK;

bool
plkIsStarted(void)
{
    return mPLK.isPlkStarted;
}

enum BsmStateEnum
plkGetLastBsmState(void)
{
    return mPLK.lastBsmState;
}

uint8_t
plkGetNmtStatus(void)
{
    return mPLK.lastNmtState & 0xff;
}

/**
 * user callback for PHY-specific configuration
 *
 * @param[in] hEth handle of ethernet driver, see omethCreate()
 * @return 0 (no error), -1 otherwise
 */
int
omethPhyCfgUser(OMETH_H hEth)
{
    return 0;
}

void
plkReqNil(void)
{
    /* sanity */
    if (mPLK.hQueueRequest == NULL)
        return;

    static struct PlkRequest req = { .id = PLK_REQ_nil };

    /* send request */
    BaseType_t ret = xQueueSend(mPLK.hQueueRequest, &req, 0);

    if (ret != pdTRUE)
        logMsg(ERROR, ERR_APPLICATION, "PLK: queue overflow sending nil");
}

void
plkReqStart(const unsigned int nodeId, const unsigned int use2ndBfa, const uint8_t mac[ETH_ALEN])
{
    /* sanity */
    if (mPLK.hQueueRequest == NULL)
        return;

    assert((nodeId > C_ADR_INVALID) && (nodeId <= kMaxDeviceCount));

    static struct PlkRequest req = { .id = PLK_REQ_start };

    req.start.nodeId    = nodeId;
    req.start.use2ndBfa = use2ndBfa;
    memcpy(req.start.mac, mac, ETH_ALEN);

    /* send request */
    BaseType_t ret = xQueueSend(mPLK.hQueueRequest, &req, 0);

    if (ret != pdTRUE)
        logMsg(ERROR, ERR_APPLICATION, "PLK: queue overflow sending start");
}

void
plkReqTickIsr(void)
{
    /* sanity */
    if (mPLK.hQueueRequest == NULL)
        return;

    static struct PlkRequest req = { .id = PLK_REQ_tick };

    /* send request */
    BaseType_t ret = xQueueSendFromISR(mPLK.hQueueRequest, &req, NULL);

    if (ret != pdTRUE)
        statsIncrPlkTick();
}

void
plkReqStop(void)
{
    /* sanity */
    if (mPLK.hQueueRequest == NULL)
        return;

    static struct PlkRequest req = { .id = PLK_REQ_stop };

    /* send request */
    BaseType_t ret = xQueueSend(mPLK.hQueueRequest, &req, 0);

    if (ret != pdTRUE)
        logMsg(ERROR, ERR_APPLICATION, "PLK: queue overflow sending stop");
}

void
plkReqFwdIp(struct sbuf *pData, size_t len)
{
    /* sanity */
    assert(pData != NULL);
    assert(len > 0);

    if (pData == NULL)
        return;

    if (mPLK.hQueueRequest == NULL) {
        sbufPut(pData);
        return;
    }

    /* do not queue implausible short non-PLK frames */
    if (len <= 0) {
        sbufPut(pData);
        return;
    }

    static struct PlkRequest req = { .id = PLK_REQ_fwdIp };

    req.fwdIp.p  = pData;
    req.fwdIp.len = len;

    /* send request */
    BaseType_t ret = xQueueSend(mPLK.hQueueRequest, &req, 0);

    if (ret != pdTRUE) {
        sbufPut(pData);
        logMsg(ERROR, ERR_APPLICATION, "PLK: queue overflow sending stop");
    }
}

void
plkSetErrorThresholds(uint32_t t)
{
    /* update DLL_CNLossSoC_REC (0x1C0B) with new threshold (sub-index 3); intentionally ignore return code */
    (void)obdWriteEntry(0x1C0B, 0x03, &t, sizeof(t));

    /* update DLL_CNLossPReq_REC (0x1C0D) with new threshold (sub-index 3); intentionally ignore return code */
    (void)obdWriteEntry(0x1C0D, 0x03, &t, sizeof(t));
}

void
plkSetSwUpdateStatus(enum UpdateSwStatus swuStatus)
{
    const uint8_t swuStat = swuStatus;

    /* update UpdateSwStatus_U8 (0x2003:2); intentionally ignore return code */
    (void)obdWriteEntry(0x2003, 0x02, &swuStat, sizeof(swuStat));
}

/**
 * NMT state change event
 *
 * This function is called in PLK task context in the course of
 * oplk_process() in the PowerLink stack tick
 *
 * @param[in] pNmtStateChange_p NMT state change data
 * @param[in] pUserArg_p unused
 *
 * @return kErrorOk or one of the kError* constants
 */
static tOplkError
processStateChangeEvent(const tEventNmtStateChange *pNmtStateChange_p,
                        void                       *pUserArg_p)
{
    /* unused */
    (void)pUserArg_p;

    /* update last known NMT status */
    mPLK.lastNmtState = pNmtStateChange_p->newNmtState;

    /* reset last BSM status if CN is not in an active PLK state */
    if (!NMT_IF_ACTIVE_CN(pNmtStateChange_p->newNmtState) || (pNmtStateChange_p->newNmtState <= kNmtCsStopped))
        mPLK.lastBsmState = BS_INVALID;

    if (pNmtStateChange_p->newNmtState == kNmtGsResetCommunication) {
        tOplkError ret = kErrorOk;

        /* reset multiple cycle assignment */
        for (int n = 0; n < kMaxDeviceCount; n++) {
            const uint8_t cycleNo = (n >> 2) + 1;

            if ((ret = obdWriteEntry(0x1F9b, n + 1, &cycleNo, sizeof(cycleNo))) != kErrorOk)
                return ret;
        }

        /* set CN assignment NMT_NodeAssignment_AU32 (0x1F81) */
        const uint32_t cnNodeAssignment = (NMT_NODEASSIGN_VALID |
                                           NMT_NODEASSIGN_START_CN |
                                           NMT_NODEASSIGN_NODE_EXISTS |
                                           NMT_NODEASSIGN_NODE_IS_CN |
                                           NMT_NODEASSIGN_MULTIPLEXED_CN);

        if ((ret = obdWriteEntry(0x1F81, mPLK.nodeId, &cnNodeAssignment, sizeof(cnNodeAssignment))) != kErrorOk)
            return ret;

        /* set MN assignment NMT_NodeAssignment_AU32 (0x1F81) */
        const uint32_t mnNodeAssignment = (NMT_NODEASSIGN_VALID |
                                           NMT_NODEASSIGN_NODE_EXISTS |
                                           NMT_NODEASSIGN_MN_PRES);

        if ((ret = obdWriteEntry(0x1F81, kDefaultMnId, &mnNodeAssignment, sizeof(mnNodeAssignment))) != kErrorOk)
            return ret;

        /* set PReq and PRes limits */
        const uint16_t presLimitCn = (uint16_t)sizeof(struct BnResult);

        if ((ret = obdWriteEntry(0x1F8D, mPLK.nodeId, &presLimitCn, sizeof(presLimitCn))) != kErrorOk)
            return ret;

        const uint16_t presLimitMn =sizeof(struct BsmInfo);

        if ((ret = obdWriteEntry(0x1F8D, kDefaultMnId, &presLimitMn, sizeof(presLimitMn))) != kErrorOk)
            return ret;

        /* setup PDO mapping */
        if ((ret = obdMapRxPdo(0x2010, sizeof(struct BsmInfo), 0)) != kErrorOk)
            return ret;

        if ((ret = obdMapTxPdo(0x2000, sizeof(struct BnResult), 0)) != kErrorOk)
            return ret;

        /* reset SwUpdateCommand and SwUpdateStatus */
        const uint8_t swuCmd = USC_IDLE;

        if ((ret = obdWriteEntry(0x2003, 1, &swuCmd, sizeof(swuCmd))) != kErrorOk)
            return ret;

        const uint8_t swuStat = USS_IDLE;

        if ((ret = obdWriteEntry(0x2003, 2, &swuStat, sizeof(swuStat))) != kErrorOk)
            return ret;
    }

    const QueueHandle_t hCdi2Queue = cdi2GetMessageQueue();

    if (hCdi2Queue != NULL) {
        static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_NmtStateChange };

        cdi2Msg.nmtStateChange.oldNmtState = pNmtStateChange_p->oldNmtState;
        cdi2Msg.nmtStateChange.newNmtState = pNmtStateChange_p->newNmtState;

        if (xQueueSend(hCdi2Queue, &cdi2Msg, 0) != pdTRUE)
            statsIncrNmtStatus();
    }

    return kErrorOk;
}

/**
 * PLK error event
 *
 * This function is called in PLK task context in the course of
 * oplk_process() in the PowerLink stack tick
 *
 * @param[in] pInternalError_p PLK error event data
 * @param[in] pUserArg_p unused
 *
 * @return kErrorOk
 */
static tOplkError
processErrorEvent(const tEventError *pInternalError_p,
                  void              *pUserArg_p)
{
    /* unused */
    (void)pUserArg_p;

    logFormat(ERROR, ERR_POWERLINK, "PLK[E]: S=0x%x E=0x%x, A=0x%x",
              pInternalError_p->eventSource, pInternalError_p->oplkError, pInternalError_p->errorArg.uintArg);

    return kErrorOk;
}

/**
 * PLK warning event
 *
 * This function is called in PLK task context in the course of
 * oplk_process() in the PowerLink stack tick
 *
 * @param[in] pInternalError_p PLK warning event data
 * @param[in] pUserArg_p unused
 *
 * @return kErrorOk
 */
static tOplkError
processWarningEvent(const tEventError *pInternalError_p,
                    void              *pUserArg_p)
{
    /* unused */
    (void)pUserArg_p;

    logFormat(WARN, ERR_POWERLINK, "PLK[W]: S=0x%x E=0x%x, A=0x%x",
              pInternalError_p->eventSource, pInternalError_p->oplkError, pInternalError_p->errorArg.uintArg);

    return kErrorOk;
}

/**
 * PLK history event
 *
 * This function is called in PLK task context in the course of
 * oplk_process() in the PowerLink stack tick
 *
 * @param[in] pHistoryEntry_p PLK history event data
 * @param[in] pUserArg_p unused
 *
 * @return kErrorOk
 */
static tOplkError
processHistoryEvent(const tErrHistoryEntry *pHistoryEntry_p,
                    void                   *pUserArg_p)
{
    /* unused */
    (void)pUserArg_p;

    switch (pHistoryEntry_p->errorCode) {
    case E_DLL_INVALID_FORMAT:
    case E_DLL_LOSS_PRES_TH:
        logFormat(ERROR, ERR_POWERLINK, "PLK[H]: T=0x%x C=0x%x CN=0x%x",
                  pHistoryEntry_p->entryType,
                  pHistoryEntry_p->errorCode,
                  (uint16_t)pHistoryEntry_p->aAddInfo[0]);
        break;

    case E_DLL_CYCLE_EXCEED:
    case E_DLL_CYCLE_EXCEED_TH:
        logFormat(ERROR, ERR_POWERLINK, "PLK[H]: T=0x%x C=0x%x E=0x%x",
                  pHistoryEntry_p->entryType,
                  pHistoryEntry_p->errorCode,
                  (uint16_t)pHistoryEntry_p->aAddInfo[0]);
        break;

    case E_DLL_LOSS_PREQ_TH:
        logMsg(ERROR, ERR_POWERLINK, "PLK[H]: Loss of PReq");
        break;

    case E_DLL_LOSS_SOA_TH:
        logMsg(ERROR, ERR_POWERLINK, "PLK[H]: Loss of SoA");
        break;

    case E_DLL_LOSS_SOC_TH:
        logMsg(ERROR, ERR_POWERLINK, "PLK[H]: Loss of SoC");
        break;

    default:
        logFormat(ERROR, ERR_POWERLINK, "PLK[H]: T=0x%x C=0x%x A=0x[%x:%x:%x:%x:%x:%x:%x:%x]",
                  pHistoryEntry_p->entryType,
                  pHistoryEntry_p->errorCode,
                  pHistoryEntry_p->aAddInfo[0],
                  pHistoryEntry_p->aAddInfo[1],
                  pHistoryEntry_p->aAddInfo[2],
                  pHistoryEntry_p->aAddInfo[3],
                  pHistoryEntry_p->aAddInfo[4],
                  pHistoryEntry_p->aAddInfo[5],
                  pHistoryEntry_p->aAddInfo[6],
                  pHistoryEntry_p->aAddInfo[7]);
    }

    return kErrorOk;
}

/**
 * MN NMT node events are ignored on CN
 *
 * @param[in] pNode_p unused
 * @param[in] pUserArg_p unused
 *
 * @return kErrorOk
 */
static tOplkError
processNodeEvent(const tOplkApiEventNode *pNode_p,
                 void                    *pUserArg_p)
{
    /* unused */
    (void)pNode_p;
    (void)pUserArg_p;

    return kErrorOk;
}

/**
 * OpenPowerLink Configuration Manager events are ignored on CN
 *
 * @param[in] pCfmProgress_p unused
 * @param[in] pUserArg_p unused
 *
 * @return kErrorOk
 */
static tOplkError
processCfmProgressEvent(const tCfmEventCnProgress *pCfmProgress_p,
                        void                      *pUserArg_p)
{
    /* unused */
    (void)pCfmProgress_p;
    (void)pUserArg_p;

    return kErrorOk;
}

/**
 * OpenPowerLink Configuration Manager events are ignored on CN
 *
 * @param[in] pCfmResult_p unused
 * @param[in] pUserArg_p unused
 *
 * @return kErrorOk
 */
static tOplkError
processCfmResultEvent(const tOplkApiEventCfmResult *pCfmResult_p,
                      void                         *pUserArg_p)
{
    /* unused */
    (void)pCfmResult_p;
    (void)pUserArg_p;

    return kErrorOk;
}

/**
 * OpenPowerLink Non-PLK frame received
 *
 * @param[in] pNonPlk_p Non-PLK/MAC frame event data
 * @param[in] pUserArg_p unused
 *
 * @return kErrorOk
 */
static tOplkError
processNonPlkEvent(const tOplkApiEventReceivedNonPlk *pNonPlk_p,
                   void                              *pUserArg_p)
{
    /* unused */
    (void)pUserArg_p;

    gwProcessMac(pNonPlk_p->pFrame, pNonPlk_p->frameSize);
    return kErrorOk;
}

/**
 * CDI2 CNs do not directly start/source SDO transactions. Thus,
 * any resources (i.e., allocated communication handles) are
 * properly released.
 *
 * @param[in] pSdoInfo_p SDO command data
 * @param[in] pUserArg_p unused
 *
 * @return kErrorOk
 */
static tOplkError
processSdoEvent(const tSdoComFinished *pSdoInfo_p,
                void                  *pUserArg_p)
{
    /* unused */
    (void)pUserArg_p;

    switch (pSdoInfo_p->sdoComConState) {
    case kSdoComTransferTxAborted:
        /* FALLTHROUGH */

    case kSdoComTransferRxAborted:
        /* FALLTHROUGH */

    case kSdoComTransferLowerLayerAbort:
        /* FALLTHROUGH */

    case kSdoComTransferRxSubAborted:
        LOG_TRACE("* SDO: CN%02u ix 0x%04x, subIx 0x%02x failed (%u)\r\n", pSdoInfo_p->nodeId, pSdoInfo_p->targetIndex, pSdoInfo_p->targetSubIndex, pSdoInfo_p->sdoComConState);
        /* FALLTHROUGH */

    case kSdoComTransferFinished: {
        /* sanity */
        if ((pSdoInfo_p->nodeId <= C_ADR_INVALID) || (pSdoInfo_p->nodeId > UINT8_MAX))
            break;

        /* like CDI2 simulator, release channel after successful transfer */
        oplk_freeSdoChannel(pSdoInfo_p->sdoComConHdl);
    }

    default:
        /* nil */;
    }

    return kErrorOk;
}

/**
 * OpenPowerLink default gateway changed event
 *
 * @param[in] pGw_p new event detailing the new default gateway
 * @param[in] pUserArg_p unused
 *
 * @return kErrorOk
 */
static tOplkError
processGwEvent(const tOplkApiEventDefaultGwChange *pGw_p,
               void                               *pUserArg_p)
{
    /* unused */
    (void)pUserArg_p;

    gwUpdateDefaultGateway(pGw_p->defaultGateway);
    return kErrorOk;
}

/**
 * OpenPowerLink API event callback
 *
 * @param[in] eventType_p OpenPowerLink event code/type
 * @param[in] pEventArg_p Pointer to the event argument
 * @param[in] pUserArg_p user defined argument pointer (@see tOplkApiInitParam.pEventUserArg)
 *
 * @return The function returns a tOplkError error code
 */
static tOplkError
processOplkEvent(tOplkApiEventType       eventType_p,
                 const tOplkApiEventArg *pEventArg_p,
                 void                   *pUserArg_p)
{
    switch (eventType_p) {
        case kOplkApiEventNmtStateChange:
            return processStateChangeEvent(&pEventArg_p->nmtStateChange, pUserArg_p);

        case kOplkApiEventCriticalError:
            return processErrorEvent(&pEventArg_p->internalError, pUserArg_p);

        case kOplkApiEventWarning:
            return processWarningEvent(&pEventArg_p->internalError, pUserArg_p);

        case kOplkApiEventHistoryEntry:
            return processHistoryEvent(&pEventArg_p->errorHistoryEntry, pUserArg_p);

        case kOplkApiEventNode:
            return processNodeEvent(&pEventArg_p->nodeEvent, pUserArg_p);

        case kOplkApiEventCfmProgress:
            return processCfmProgressEvent(&pEventArg_p->cfmProgress, pUserArg_p);

        case kOplkApiEventCfmResult:
            return processCfmResultEvent(&pEventArg_p->cfmResult, pUserArg_p);

        case kOplkApiEventReceivedNonPlk:
            return processNonPlkEvent(&pEventArg_p->receivedEth, pUserArg_p);

        case kOplkApiEventSdo:
            return processSdoEvent(&pEventArg_p->sdoInfo, pUserArg_p);

        case kOplkApiEventDefaultGwChange:
            return processGwEvent(&pEventArg_p->defaultGwChange, pUserArg_p);

        default:
            break;
    }

    return kErrorOk;
}

/**
 * Returns the current cycle count in a multiplexed PowerLink cycle.
 *
 * If the cycle count is 0, then multiplexing is disabled. The cycle count
 * indicates the position of the current cycle in a multiplexed cycle and
 * it ranges from 1 to kMultiplCycleCount. Different nodes are polled in
 * each cycle. The cycle count is used by the application to infer which
 * nodes have been polled and consequently which parts of the process
 * image contain new data.
 *
 * On a CN, the stack resets the cycle count to 1 every time a PReq is
 * received (i.e., the BSM polled the CN).
 *
 * @return current CDI2 cycle count
 */
static uint8_t
plkGetCycleCount(void)
{
    uint8_t cycleCount = oplk_cycleCount();

    if (cycleCount != 0) {
        /*
         * multiplexed cycle: data in PI is from previous cycle,
         * adjust accordingly.
         */
        if (--cycleCount == 0)
            /* wrap-around */
            cycleCount = kMultiplxdCycleCount;
    }

    return cycleCount;
}

void
plkBnRecognition(struct BnInfo *pBnRecognition)
{
    /* sanity */
    if (mPLK.hQueueBnRecognition == NULL)
        return;

    /* send request */
    BaseType_t ret = xQueueSend(mPLK.hQueueBnRecognition, pBnRecognition, 0);

    if (ret != pdTRUE)
        statsIncrBnRecognition();
}

void
plkBnResult(struct sbuf *pData, size_t len)
{
    /* sanity */
    assert(pData != NULL);
    assert(len > 0);

    if (pData == NULL)
        return;

    /* do not queue implausible short results (just ID + CRC16) */
    if (len <= 3) {
        sbufPut(pData);
        return;
    }

    /* defensive programming: truncate length to PDO size */
    if (len > (sizeof(struct BnResult) - offsetof(struct BnResult, bn_info)))
        len = sizeof(struct BnResult) - offsetof(struct BnResult, bn_info);

    struct PlkBnResultMessage bnResultData = {
        .p   = pData,
        .len = len
    };

    /* send request */
    if ((mPLK.hQueueBnResult == NULL) || (xQueueSend(mPLK.hQueueBnResult, &bnResultData, 0) != pdTRUE)) {
        sbufPut(pData);
        statsIncrBnResult();
    }
}

/**
 * copy BN result data from an CDI BNRESULT sbuf chain into linear PDO memory.
 *
 * @param[in] dest destination PDO memory buffer
 * @param[in] src source data on sbuf chain
 * @param[in] len number of bytes to copy
 */
static void
memcpyBnResult(void *dest, struct sbuf *src, size_t len)
{
    /* remove ID and CRC16 from total length */
    len = (len > 3) ? len - 3 : 0;

    /* 1st sbuf on chain has ID byte; ignore it */
    if ((src != NULL) && (len > 0)) {
        /* copy chunk */
        size_t copyLen = len > (sizeof(src->data) - 1) ? (sizeof(src->data) - 1): len;
        memcpy(dest, src->data + 1, copyLen);

        /* update remaining length, goto next sbuf on chain */
        len  -= copyLen;
        dest += copyLen;
        src   = src->pNext;

        /* poll UART RX FIFOs */
        cdi2IfPollUartIsr();
        pppIfPollUartIsr();
    }

    while ((src != NULL) && (len > 0)) {
        /* copy chunk */
        size_t copyLen = len > sizeof(src->data) ? sizeof(src->data) : len;
        memcpy(dest, src->data, copyLen);

        /* update remaining length, goto next sbuf on chain */
        len  -= copyLen;
        dest += copyLen;
        src   = src->pNext;

        /* poll UART RX FIFOs */
        cdi2IfPollUartIsr();
        pppIfPollUartIsr();
    }
}

void plkResetBnResultQueues(void)
{
    if (mPLK.hQueueBnRecognition != NULL)
        xQueueReset(mPLK.hQueueBnRecognition);

    if (mPLK.hQueueBnResult != NULL) {
        /* release all sbuf chains */
        for (struct PlkBnResultMessage bnResultData; xQueueReceive(mPLK.hQueueBnResult, &bnResultData, 0) == pdTRUE;)
            sbufPut(bnResultData.p);

        xQueueReset(mPLK.hQueueBnResult);
    }
}

/**
 * OpenPowerLink API SYNC event callback.
 *
 * This function is called in FreeRTOS interrupt context
 *
 * @return kErrorOk, some other kError* constant in case of error
 */
static tOplkError
processSyncEvent(void)
{
    const QueueHandle_t hCdi2Queue = cdi2GetMessageQueue();
    tOplkError          ret        = kErrorOk;

    /* poll UART RX FIFOs */
    cdi2IfPollUartIsr();
    pppIfPollUartIsr();

    /* kick watchdog */
    wdKick();

    /*
     * update system time with NETTIME from SoC.
     *
     * Note: Remember, tsSoC really is a 32 bit counter zero-padded to 64 bits
     * Note: 1st tsSoc is always 0 (because of triple buffering). Any underflows
     * because of this are cleared after the next cycle.
     */
    static tOplkApiSocTimeInfo socTime;
    ret = oplk_getSocTime(&socTime);

    if (ret == kErrorOk)
        timer_updateNettime(socTime.tsSoc, socTime.netTime.sec, socTime.netTime.nsec);

    /* rotate the BSM PRes frame of the current cycle into the process image and get frame pointer piOut */
    ret = oplk_exchangeProcessImageOut();

    if (ret != kErrorOk)
        return ret;

    void *piOut = oplk_getProcessImageOut();

    if (piOut == NULL)
        return kErrorApiNotInitialized;

    /* poll UART RX FIFOs */
    cdi2IfPollUartIsr();
    pppIfPollUartIsr();

    /* check BSMINFO only when we are NMT operational */
    if (mPLK.lastNmtState == kNmtCsOperational) {
        /* analyze process image BLOB into BsmInfo structure */
        struct BsmInfo *pBsmInfo = (struct BsmInfo *)piOut;

        if (pBsmInfo->header.command_id == CMDID_BSMINFO) {
            /* detect BSM status change */
            if (mPLK.lastBsmState != pBsmInfo->state) {
                static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_BsmStateChange };
                cdi2Msg.bsmStateChange.oldBsmState = mPLK.lastBsmState;
                cdi2Msg.bsmStateChange.newBsmState = pBsmInfo->state;

                if ((hCdi2Queue == NULL) || (xQueueSendFromISR(hCdi2Queue, &cdi2Msg, NULL) != pdTRUE))
                    statsIncrBsmStatus();
            }

            /* save new BSM state */
            mPLK.lastBsmState = pBsmInfo->state;

            /* check BSM sequence number only in [BS_INITIALIZATION, BS_REQUEST_TO_SHUTDOWN] */
            if ((mPLK.lastBsmState >= BS_INITIALISATION) && (mPLK.lastBsmState <= BS_REQUEST_TO_SHUT_DOWN)) {
                if ((mPLK.lastSeqNo + 1) != pBsmInfo->header.sequence_num)
                    statsIncrSeqNo();
            }

            /* save last BSMINFO sequence number */
            mPLK.lastSeqNo = pBsmInfo->header.sequence_num;

            /* update BSMINFO:TC counter measurements for this node */
            if ((mPLK.lastBsmState == BS_REQUEST_TO_SORT) || (mPLK.lastBsmState == BS_SORTING))
                ttsUpdateTcMeas(socTime.tsSoc, pBsmInfo->bn_triggers[mPLK.nodeId - 1].tc_count);

            if (mPLK.lastBsmState == BS_SORTING) {
                /* check BnTrigger announcement */
                if (pBsmInfo->bn_triggers[mPLK.nodeId - 1].id != BNINVALID) {
                    static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_BNTRIGGER };
                    cdi2Msg.bnTrigger.bntrigger = pBsmInfo->bn_triggers[mPLK.nodeId - 1];
                    cdi2Msg.bnTrigger.is2ndBfa  = 0;

                    if ((hCdi2Queue == NULL) || (xQueueSendFromISR(hCdi2Queue, &cdi2Msg, NULL) != pdTRUE))
                        statsIncrBnTrigger();
                }

                /* optionally check BnTrigger announcement for BFA#2 */
                if ((mPLK.use2ndBfa != 0) && (pBsmInfo->bn_triggers[15].id != BNINVALID)) {
                    static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_BNTRIGGER };
                    cdi2Msg.bnTrigger.bntrigger = pBsmInfo->bn_triggers[15];
                    cdi2Msg.bnTrigger.is2ndBfa  = 1;

                    if ((hCdi2Queue == NULL) || (xQueueSendFromISR(hCdi2Queue, &cdi2Msg, NULL) != pdTRUE))
                        statsIncrBnTrigger();
                }

                /* check BNINFO */
                if (pBsmInfo->bn_info.id != BNINVALID) {
                    static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_BNINFIO };
                    cdi2Msg.bninfo = pBsmInfo->bn_info;

                    if ((hCdi2Queue == NULL) || (xQueueSendFromISR(hCdi2Queue, &cdi2Msg, NULL) != pdTRUE))
                        statsIncrBnInfo();
                }
            }

            /* poll UART RX FIFOs */
            cdi2IfPollUartIsr();
            pppIfPollUartIsr();
        }
    }

    uint8_t cycleCount = plkGetCycleCount();

    if ((cycleCount != 0) && (cycleCount != 1))
        /* not polled, return */
        return kErrorOk;

    /*
     * Prepare the BNRESULT sent at next assigned slot
     */
    static uint32_t  seqNo     = 0;
    struct BnResult *pBnResult = (struct BnResult *)oplk_getProcessImageIn();

    if (pBnResult == NULL)
        return kErrorApiNotInitialized;

    memset(pBnResult, 0, sizeof(struct BnResult));

    pBnResult->header.command_id   = CMDID_BNRESULT;
    pBnResult->header.sequence_num = seqNo++;

    pBnResult->device_state      = cdi2GetDeviceStatus();
    pBnResult->error_state       = cdi2GetErrorStatus();
    pBnResult->maintenance_state = cdi2GetMaintenanceStatus();

    /* send queued BNRECOGNITIONs */
    static struct BnInfo bnRecognition;

    if (xQueueReceiveFromISR(mPLK.hQueueBnRecognition, &bnRecognition, NULL) == pdTRUE) {
        if (bnRecognition.id != BNINVALID)
            pBnResult->bn_recognition = bnRecognition;
    } else
        pBnResult->bn_recognition.id = BNINVALID;

    /* poll UART RX FIFOs */
    cdi2IfPollUartIsr();
    pppIfPollUartIsr();

    /* send queued BNRESULTs */
    static struct PlkBnResultMessage bnResultData;

    if (xQueueReceiveFromISR(mPLK.hQueueBnResult, &bnResultData, NULL) == pdTRUE) {
        memcpyBnResult((void *)pBnResult + offsetof(struct BnResult, bn_info), bnResultData.p, bnResultData.len);
        sbufPutIsr(bnResultData.p);
    } else
        pBnResult->bn_info.id = BNINVALID;

    /* rotate PRes frame into the process image BLOB */
    return oplk_exchangeProcessImageIn();
}

/**
 * start the OpenPowerLink stack
 */
static tOplkError
plkStartStack(const unsigned int nodeId, const uint8_t mac[ETH_ALEN])
{
    /* default CDI2 Device MAC address for CN01 */
    static const uint8_t defaultMacAddress[ETH_ALEN] = { 0x00, 0x60, 0x36, 0x9f, 0xc5, 0x21 };
    static const uint8_t macZero[ETH_ALEN]           = { 0, 0, 0, 0, 0, 0 };

    /* sanity */
    assert((nodeId > C_ADR_INVALID) && (nodeId <= kMaxDeviceCount));

    const static char device_name[] = "CDI2DEV";
    tOplkError        ret           = kErrorOk;
    tOplkApiInitParam initParam;

    memset(&initParam, 0, sizeof(initParam));
    initParam.sizeOfInitParam = sizeof(initParam);

    initParam.hwParam.pDevName = device_name;
    initParam.nodeId           = nodeId;

    /* set IP and MAC address of node */
    assert(sizeof(initParam.aMacAddress) == sizeof(defaultMacAddress));

    if (memcmp(mac, macZero, ETH_ALEN) == 0) {
        /* use CDI2 default CNxx MAC address */
        memcpy(initParam.aMacAddress, defaultMacAddress, sizeof(defaultMacAddress));
        initParam.aMacAddress[5] = 0x20 + initParam.nodeId;
    } else {
        /* use requested MAC address */
        memcpy(initParam.aMacAddress, mac, sizeof(defaultMacAddress));
    }

    initParam.ipAddress      = (kDefaultSubnetMask & kDefaultIpAddr) | initParam.nodeId;
    initParam.subnetMask     = kDefaultSubnetMask;
    initParam.defaultGateway = kDefaultGateway;
    gwUpdateLocalAddress(initParam.ipAddress, initParam.aMacAddress);

    initParam.fAsyncOnly           = FALSE;
    initParam.featureFlags         = UINT32_MAX;
    initParam.cycleLen             = kCycleLen;
    initParam.isochrTxMaxPayload   = C_DLL_ISOCHR_MAX_PAYL;
    initParam.isochrRxMaxPayload   = C_DLL_ISOCHR_MAX_PAYL;
    initParam.presMaxLatency       = 10000;
    initParam.preqActPayloadLimit  = sizeof(struct BsmInfo);            /* CN receive size = ||BsmInfo|| */
    initParam.presActPayloadLimit  = sizeof(struct BnResult);           /* CN send size = ||BnResult|| */
    initParam.asndMaxLatency       = 25000;
    initParam.multiplCylceCnt      = kMultiplxdCycleCount;
    initParam.asyncMtu             = kAsyncMTU;
    initParam.prescaler            = 0;
    initParam.lossOfFrameTolerance = 100000;
    initParam.asyncSlotTimeout     = 100000;
    initParam.waitSocPreq          = kWaitSocPreq;
    initParam.deviceType           = UINT32_MAX;
    initParam.vendorId             = UINT32_MAX;
    initParam.productCode          = UINT32_MAX;
    initParam.revisionNumber       = UINT32_MAX;
    initParam.serialNumber         = UINT32_MAX;
    initParam.syncNodeId           = C_ADR_SYNC_ON_SOA;
    initParam.fSyncOnPrcNode       = FALSE;

    snprintf((char *)initParam.sHostname, sizeof(initParam.sHostname),
             "%02x-%08x", initParam.nodeId, initParam.vendorId);

    /* plumb callback functions */
    initParam.pfnCbSync  = processSyncEvent;
    initParam.pfnCbEvent = processOplkEvent;

    /* initialize object dictionary */
    if ((ret = obdCreate(&initParam.obdInitParam)) != kErrorOk)
        return ret;

    /* initialize OpenPOWERLINK stack */
    if ((ret = oplk_initialize()) != kErrorOk)
        return ret;

    if ((ret = oplk_create(&initParam)) != kErrorOk)
        return ret;

    /* modify OpenPOWERLINK OBD with correct sizes for PRes/BnResult and PReq/BsmInfo */
    if ((ret = obdAddByteArray(0x2000, 1, initParam.presActPayloadLimit)) != kErrorOk)
        return ret;

    if ((ret = obdAddByteArray(0x2010, 1, initParam.preqActPayloadLimit)) != kErrorOk)
        return ret;

    /* setup process image */
    if ((ret = oplk_allocProcessImage(sizeof(struct BnResult), sizeof(struct BsmInfo))) != kErrorOk)
        return ret;

    /* link process image to OpenPOWERLINK OBD */
    UINT varEntries = 1;
    ret = oplk_linkProcessImageObject(0x2000,                           /* objIndex */
                                      0x01,                             /* firstSubindex_p */
                                      0,                                /* offsetPI */
                                      FALSE,                            /* fOutputPI_p */
                                      initParam.presActPayloadLimit,    /* entrySize_p */
                                      &varEntries);                     /* pVarEntries_p */

    if (ret != kErrorOk)
        return ret;

    varEntries = 1;
    ret = oplk_linkProcessImageObject(0x2010    ,                       /* objIndex */
                                      0x01,                             /* firstSubindex_p */
                                      0,                                /* offsetPI */
                                      TRUE,                             /* fOutputPI_p */
                                      initParam.preqActPayloadLimit,    /* entrySize_p */
                                      &varEntries);                     /* pVarEntries_p */

    if (ret != kErrorOk)
        return ret;

    /* prepare to receive Non-PLK MAC frames */
    ret = oplk_setNonPlkForward(TRUE);

    if (ret != kErrorOk)
        return ret;

    /* start NMT via NmtEventSwReset */
    if ((ret = oplk_execNmtCommand(kNmtEventSwReset)) != kErrorOk)
        return ret;

    return kErrorOk;
}

/**
 * PowerLink layer task
 *
 * @param arg task parameter passed via xTaskCreate()
 */
static void
plkTask(void *arg)
{
    /* unused */
    (void)arg;

    LOG_TRACE("# PLK: started\r\n");

    /* FOREVER */
    for (;;) {
        /* receive request */
        struct PlkRequest req;

        if (xQueueReceive(mPLK.hQueueRequest, &req, portMAX_DELAY) == pdFALSE)
            /* time-out, nothing received */
            continue;

        switch (req.id) {
        case PLK_REQ_nil:
            logMsg(TRACE, 0, "PLK: request nil");
            break;

        case PLK_REQ_start: {
            logFormat(TRACE, 0, "PLK: request start nodeId %u", req.start.nodeId);

            /* sanity */
            if ((req.start.nodeId <= C_ADR_INVALID) || (req.start.nodeId > kMaxDeviceCount))
                break;

            /* reset BNRESULT queues */
            plkResetBnResultQueues();

            mPLK.nodeId    = (uint8_t)req.start.nodeId;
            mPLK.use2ndBfa = (uint32_t)req.start.use2ndBfa;

            tOplkError ret = plkStartStack(req.start.nodeId, req.start.mac);

            if (ret != kErrorOk) {
                logFormat(FATAL, ERR_POWERLINK, "PLK: plkStartStack 0x%x", ret);
            } else {
                const QueueHandle_t hCdi2Queue = cdi2GetMessageQueue();

                mPLK.isPlkStarted = true;

                /* set relaxed SoC/PReq error thresholds */
                plkSetErrorThresholds(kRelaxedErrorThreshold);

                /* send start acknowledge to CDI2 layer */
                static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_PlkStarted };

                if ((hCdi2Queue == NULL) || (xQueueSend(hCdi2Queue, &cdi2Msg, 0) != pdTRUE))
                    statsIncrCdi2Queue();
            }

            break;
        }

        case PLK_REQ_tick:
            if (mPLK.isPlkStarted) {
                /* background task processing */
                tOplkError ret = oplk_process();

                if (ret != kErrorOk) {
                    logFormat(ERROR, ERR_POWERLINK, "PLK: oplk_process 0x%x", ret);
                    break;
                }

                /* check stack heartbeat */
                if (oplk_checkKernelStack() == FALSE) {
                    logMsg(ERROR, ERR_POWERLINK, "PLK: oplk_checkKernelStack");
                    break;
                }

                if (mPLK.lastBsmState == BS_REQUEST_TO_SHUT_DOWN) {
                    const QueueHandle_t hCdi2Queue = cdi2GetMessageQueue();

                    /* poll local SDO UpdateSwCommand */
                    static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_UpdateSwCommand };
                    cdi2Msg.swUpdateCommand.swuCmd    = USC_IDLE;

                    if ((ret = obdReadEntry(0x2003, 1, &cdi2Msg.swUpdateCommand.swuCmd, sizeof(cdi2Msg.swUpdateCommand.swuCmd))) != kErrorOk) {
                        logFormat(ERROR, ERR_POWERLINK, "PLK: obdReadEntry 0x%x", ret);
                        break;
                    }

                    if (cdi2Msg.swUpdateCommand.swuCmd != mPLK.lastSwuCmd) {
                        mPLK.lastSwuCmd = cdi2Msg.swUpdateCommand.swuCmd;

                        if ((hCdi2Queue == NULL) || (xQueueSend(cdi2GetMessageQueue(), &cdi2Msg, 0) != pdTRUE))
                            statsIncrCdi2Queue();
                    }
                }
            }

            break;

        case PLK_REQ_stop:
            logMsg(TRACE, 0, "PLK: request stop");

            /* shut down OpenPOWERLINK stack */
            (void)oplk_execNmtCommand(kNmtEventSwitchOff);

            /* cleanup OpenPOWERLINK stack environment; NOTE: this ends in tears */
#if 0
            (void)oplk_destroy();
            oplk_exit();
#endif

            /* OpenPowerLink stack not started; ignore stack tick */
            mPLK.isPlkStarted = false;

            /* reset last known BSM state */
            mPLK.lastBsmState = BS_INVALID;
            mPLK.lastNmtState = kNmtGsOff;

            logMsg(TRACE, 0, "PLK: OpenPOWERLINK stopped");
            break;

        case PLK_REQ_fwdIp:
            gwForwardIP(req.fwdIp.p, req.fwdIp.len);
            sbufPut(req.fwdIp.p);

            break;

        default:
            logFormat(WARN, ERR_APPLICATION, "PLK: unknown request: %d", req.id);
            break;
        }
    }
}

int
plkSetup(void)
{
    /* OpenPowerLink stack not started */
    mPLK.isPlkStarted = false;

    /* reset noded ID */
    mPLK.nodeId = C_ADR_INVALID;

    /* reset last known BSM and NMT state */
    mPLK.lastBsmState = BS_INVALID;
    mPLK.lastNmtState = kNmtGsOff;

    /* reset last polled UpdateSwCommand SDO */
    mPLK.lastSwuCmd = USC_IDLE;

    LOG_TRACE("# PLK: allocate BNRECOGNITION queue %d x |%u|\r\n", PLK_BNRECOGNITION_QUEUE_LENGTH, sizeof(struct BnInfo));

    mPLK.hQueueBnRecognition = xQueueCreate(PLK_BNRECOGNITION_QUEUE_LENGTH,
                                            sizeof(struct BnInfo));

    assert(mPLK.hQueueBnRecognition != NULL);

    if (mPLK.hQueueBnRecognition == NULL)
        return XST_FAILURE;

    LOG_TRACE("# PLK: allocate BNRESULT queue %d x |%u|\r\n", PLK_BNRESULT_QUEUE_LENGTH, sizeof(struct PlkBnResultMessage));

    mPLK.hQueueBnResult = xQueueCreate(PLK_BNRESULT_QUEUE_LENGTH,
                                       sizeof(struct PlkBnResultMessage));

    assert(mPLK.hQueueBnResult != NULL);

    if (mPLK.hQueueBnResult == NULL)
        return XST_FAILURE;

    LOG_TRACE("# PLK: setup layer task, queue %u x |%lu|\r\n", PLK_REQUEST_QUEUE_LENGTH, sizeof(struct PlkRequest));

    mPLK.hQueueRequest = xQueueCreate(PLK_REQUEST_QUEUE_LENGTH,
                                      sizeof(struct PlkRequest));

    assert(mPLK.hQueueRequest != NULL);

    if (mPLK.hQueueRequest == NULL)
        return XST_FAILURE;

    xTaskCreate(plkTask,
                "PLK",
                PLK_TASK_STACK_SIZE,
                NULL,
                PLK_TASK_PRIORITY,
                &mPLK.hTaskPlk);

    assert(mPLK.hTaskPlk != NULL);

    if (mPLK.hTaskPlk == NULL)
        return XST_FAILURE;

    return XST_SUCCESS;
}
