/**
 * @file object_dictionary.c
 * @brief CDI2 Device PowerLink layer; object dictionary
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#include <oplk/oplk.h>
#include <user/obdu.h>
#include <obdcreate/obdcreate.h>

#include "object_dictionary.h"

tOplkError
obdCreate(tObdInitParam *pParams)
{
    return obdcreate_initObd(pParams);
}

tOplkError
obdAddByteArray(unsigned int index,
                unsigned int subIx,
                size_t       size)
{
    /* sanity */
    assert(index <= UINT16_MAX);
    assert(subIx <= UINT8_MAX);
    assert(size  >  0);

    tVarParam domainParam = {
            .validFlag = kVarValidSize,
            .index     = index,
            .subindex  = subIx,
            .size      = size,
            .pData     = NULL
    };

    return obdu_defineVar(&domainParam);
}

tOplkError
obdWriteEntry(unsigned int  index,
              unsigned int  subIx,
              const void   *pData,
              size_t        size)
{
    /* sanity */
    assert(index <= UINT16_MAX);
    assert(subIx <= UINT8_MAX);
    assert(pData != NULL);
    assert(size  >  0);

    return obdu_writeEntry(index, subIx, pData, size);
}

tOplkError
obdReadEntry(unsigned int  index,
             unsigned int  subIx,
             void         *pData,
             size_t        size)
{
    /* sanity */
    assert(index <= UINT16_MAX);
    assert(subIx <= UINT8_MAX);
    assert(pData != NULL);
    assert(size  >  0);

    return obdu_readEntry(index, subIx, pData, &size);
}

/**
 * Generate the AU64 encoded PDO mapping parameter (EPSG DS 301 V1.5.0, §6.4.9.2)
 *
 * MSB                                                         LSB
 * -------.-------+-------.-------+-------+-------+-------.-------
 * Length         |Offset         |rsrvd  |SubIx  |Index
 * uint16_t       |uint16_t       |uint8_t|uint8_t|uint16_t
 * (bit count)    |(bit count)    |       |       |
 *
 * @param[in] index mapped PDO index
 * @param[in] subIx mapped PDO sub index
 * @param[in] offset mapped PDO offset from payload start [byte count]
 * @param[in] size mapped PDO size [bytes]
 * @return AU64 PDO mapping
 */
static inline uint64_t
EncodeMapping(unsigned int index,
              unsigned int subIx,
              size_t       offset,
              size_t       size)
{
    /* sanity */
    assert(index  <= UINT16_MAX);
    assert(subIx  <= UINT8_MAX);
    assert(offset <= (UINT16_MAX >> 3));
    assert(size   <= (UINT16_MAX >> 3));

    const uint32_t low  = (index & 0xFFFF) | ((subIx & 0xFF) << 16);
    const uint32_t high = ((offset << 3) & 0xFFFF) | ((size << 3) << 16);

    return (((uint64_t) high) << 32) | low;
}

tOplkError
obdMapRxPdo(unsigned int index,
            size_t       size,
            unsigned int slot)
{
    tOplkError ret             = kErrorOk;
    uint64_t   rxPdoMapping    = EncodeMapping(index + slot, 1, 0, size);
    uint8_t    rxPdoMappingNum = 1;

    /* set PDO_RxMappParam_XXh_AU64 RPDO mapping for RPDO channel 'slot' */
    ret = obdWriteEntry(0x1600 + slot, 1, &rxPdoMapping, sizeof(rxPdoMapping));

    if (ret == kErrorOk) {
        /* update number of RPDO mapping at sub index 0 */
        ret = obdWriteEntry(0x1600 + slot, 0, &rxPdoMappingNum, sizeof(rxPdoMappingNum));
    }

    return ret;
}

tOplkError
obdMapTxPdo(unsigned int index,
            size_t       size,
            unsigned int slot)
{
    tOplkError ret             = kErrorOk;
    uint64_t   txPdoMapping    = EncodeMapping(index + slot, 1, 0, size);
    uint8_t    txPdoMappingNum = 1;

    /* set PDO_TxMappParam_XXh_AU64 TPDO mapping for TPDO channel 'slot' */
    ret = obdWriteEntry(0x1A00 + slot, 1, &txPdoMapping, sizeof(txPdoMapping));

    if (ret == kErrorOk) {
        /* update number of TPDO mapping at sub index 0 */
        ret = obdWriteEntry(0x1A00 + slot, 0, &txPdoMappingNum, sizeof(txPdoMappingNum));
    }

    return ret;
}
