/**
 * @file main.c
 * @brief CDI2 Device application startup
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#include <stdint.h>

#include <FreeRTOS.h>
#include <task.h>
#include <sleep.h>

#include <mb_interface.h>
#include <xstatus.h>
#include <xil_exception.h>
#include <xiltimer.h>
#include <xinterrupt_wrap.h>

#include "cdi2.h"
#include "cdi2If.h"
#include "logging.h"
#include "powerlink.h"
#include "pppIf.h"
#include "sbuf.h"
#include "statistics.h"
#include "tts.h"
#include "wd.h"

/** Performance measurements: sampled system tick count at perfmNow() */
static uint64_t T0 = 0;

/**
 * Performance measurements: start tick counting
 */
void
perfmNow(void)
{
    XTime_GetTime(&T0);
}

/**
 * Performance measurements: get system ticks since last call to perfmNow()
 *
 * @return ticks since last startCycleCount()
 */
unsigned long
perfmDelta(void)
{
    uint64_t T1 = 0;
    XTime_GetTime(&T1);
    return ((unsigned long)T1 - (unsigned long)T0);
}

/** (soft) RESET processor */
void
RESET(void)
{
    logAsync("\r\n# RESET\r\n");

    __asm__ volatile (
        "mts rmsr, r0 \n"   /* disable interrupts, exceptions */
        "bra r0       \n"   /* jump to RESET vector */
    );
}

/** STOP processor */
void
STOP(void)
{
    logAsync("\r\n# STOP\r\n");

    __asm__ volatile (
        "mts rmsr, r0 \n"   /* disable interrupts, exceptions */
        "bri 0        \n"   /* FOREVER */
    );
}

/**
 * FreeRTOS heap overflow hook function.
 *
 * @see configUSE_MALLOC_FAILED_HOOK in BSP settings
 */
void
vApplicationMallocFailedHook()
{
    LOG_PRINT("* heap overflow\r\n");

    RESET();
}

/**
 * FreeRTOS task stack overflow hook function.
 *
 * @param[in] xTask the task that just exceeded its stack boundaries.
 * @param[in] pcTaskName a character string containing the name of the offending task.
 *
 * @see configCHECK_FOR_STACK_OVERFLOW in BSP settings
 */
void
vApplicationStackOverflowHook(TaskHandle_t  xTask,
                              char         *pcTaskName)
{
    LOG_PRINT("* stack overflow %s\r\n", pcTaskName);

    RESET();
}

/**
 * FreeRTOS (Microblaze port) callback installing the tick interrupt handler
 *
 * Vivado 2022.2: The implementation in portmicroblaze.c calculates
 * the period [ms] wrong (configTICK_RATE_HZ / 10). Correct is
 * 1000 / configTICK_RATE_HZ.
 *
 * @see portmicroblaze.c
 */
void
vApplicationSetupTimerInterrupt(void)
{
    extern void vPortTickISR(void *pvUnused, uint32_t TmrCtrNumber);

    /*
     * The Xilinx implementation of generating run time task stats uses the
     * same timer used for generating FreeRTOS ticks. In case user decides
     * to generate run time stats the timer time out interval is changed as
     * "configured tick rate * 10". The multiplying factor of 10 is hard
     * coded for Xilinx FreeRTOS ports.
     */
#if (configGENERATE_RUN_TIME_STATS == 1)
    /* XTimer_SetInterval() API expects delay in milli seconds
     * Convert the user provided tick rate to milli seconds.
     */
    XTimer_SetInterval(1000 / (10 * configTICK_RATE_HZ));
#else
    /* XTimer_SetInterval() API expects delay in milli seconds
     * Convert the user provided tick rate to milli seconds.
     */
    XTimer_SetInterval(1000 / configTICK_RATE_HZ);
#endif
    XTimer_SetHandler(vPortTickISR, 0, XINTERRUPT_DEFAULT_PRIORITY);
}

/**
 * FreeRTOS tick interrupt hook function
 *
 * The tick hook will only get called if configUSE_TICK_HOOK is set to 1 within
 * FreeRTOSConfig.h.
 * vApplicationTickHook() executes from within an ISR so must be very short,
 * not use much stack, and not call any API functions that don't end in
 * "FromISR" or "FROM_ISR".
 *
 * @see configUSE_TICK_HOOK in BSP settings
 */
void
vApplicationTickHook(void)
{
    /* kick watchdog */
    wdKick();

    if (plkIsStarted()) {
        /* poll TTS reset line */
        cdi2CheckReset();

        /* check TTS transport clock */
        cdi2CheckTc();

        /* OpenPowerLink stack tick message */
        plkReqTickIsr();
    }
}

/**
 * FreeRTOS Daemon/Timer task startup hook.
 *
 * Configuring and starting the sleep and tick AXI timers by the
 * porting layer of FreeRTOS clobbers the interrupt controller configuration.
 * Thus, layers that install interrupt handlers must be setup/configured
 * after the FreeRTOS timers are up and running.
 *
 * @see configUSE_DAEMON_TASK_STARTUP_HOOK in BSP settings
 */
void
vApplicationDaemonTaskStartupHook()
{
    int ret = XST_SUCCESS;

    /* stop scheduling; the original *Setup() functions are not thread-safe */
    vTaskSuspendAll();
    taskENTER_CRITICAL();

    /*
     * initialize the different subsystems:
     *  - sbuf ring
     *  - TTS interface
     *  - CDI2 UART
     *  - PPP UART
     *  - PowerLink stack
     *  - CDI2 application
     *  - Watchdog
     */
    ret = sbufSetupRing();

    if (ret != XST_SUCCESS) {
        LOG_PRINT("* error creating sbuf ring: %d\r\n", ret);
        RESET();
        return;
    }

    ret = ttsSetup();

    if (ret != XST_SUCCESS) {
        LOG_PRINT("* error TTS setup: %d\r\n", ret);
        RESET();
        return;
    }

    ret = cdi2IfSetup(1250000);

    if (ret != XST_SUCCESS) {
        LOG_PRINT("* error CDI2 UART setup: %d\r\n", ret);
        RESET();
        return;
    }

    ret = pppIfSetup(1250000);

    if (ret != XST_SUCCESS) {
        LOG_PRINT("* error PPP UART setup: %d\r\n", ret);
        RESET();
        return;
    }

    ret = plkSetup();

    if (ret != XST_SUCCESS) {
        LOG_PRINT("* error PLK setup: %d\r\n", ret);
        RESET();
        return;
    }

    ret = cdi2Setup();

    if (ret != XST_SUCCESS) {
        LOG_PRINT("* error CDI2 setup: %d\r\n", ret);
        RESET();
        return;
    }

    ret = wdSetup();

    if (ret != XST_SUCCESS) {
        LOG_PRINT("* error WD setup: %d\r\n", ret);
        RESET();
        return;
    }

    /* resume scheduling */
    taskEXIT_CRITICAL();
    xTaskResumeAll();
}

/**
 * CDI2 Application startup
 *
 * @param[in] argc unused
 * @param[in] argv unused
 *
 * @return application is an endless loop, should never return
 */
int
main(int argc, char *argv[])
{
    extern void sbrkInit(void);

    /* unused */
    (void)argc;
    (void)argv;

    /* reset heap management state */
    sbrkInit();

    /* suppress all log messages */
    setLevel(0);

    /* clear error/statistic counters */
    statsReset();

    LOG_PRINT("\r\n** CDI2 Device Interface Core " SW_VERSION "\r\n * " __DATE__ "T" __TIME__ "\r\n");

    /*
     * <braindamage>
     * XTime_GetTime() ultimately uses the counter of the AXI sleep timer.
     * However, this timer is properly initialized only after the 1st call to
     * axi_timer.c::XAxiTimer_ModifyInterval() which is called by a
     * non-header-exported function xiltimer.c::XilTimer_Sleep which is
     * the central function to implement the sleep(), msleep() and usleep()
     * library functions.
     * </braindamage>
     */
    sleep(0);

    /* Start FreeRTOS kernel */
    vTaskStartScheduler();

    LOG_PRINT("* FreeRTOS startup failure\r\n");
    STOP();

    /* NOTREACHED */
    return 0;
}
