/**
 * @file gw.c
 * @brief PPP/PLK gateway
 *
 * @author AIT
 * @copyright &copy;2023 Austrian Institute of Technology (AIT)
 */

#include <assert.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <string.h>

#include <oplk/oplk.h>

#include "gw.h"
#include "logging.h"
#include "pppIf.h"
#include "sbuf.h"

/* sanity */
#if __BYTE_ORDER__ != __ORDER_LITTLE_ENDIAN__
#error "this code assumes little endian machine"
#endif

/*
 * IEEE 802.3 Ethernet magic constants.  The frame sizes omit the preamble
 * and FCS/CRC (frame check sequence).
 */
#define ETH_TLEN        2           /**< Octets in ethernet type field */
#define ETH_HLEN        14          /**< Total octets in header.     */
#define ETH_ZLEN        60          /**< Min. octets in frame sans FCS */
#define ETH_DATA_LEN    1500        /**< Max. octets in payload  */
#define ETH_FRAME_LEN   1514        /**< Max. octets in frame sans FCS */
#define ETH_FCS_LEN     4           /**< Octets in the FCS       */

#define ETH_MIN_MTU     68          /**< Min IPv4 MTU per RFC791    */
#define ETH_MAX_MTU     0xFFFFU     /**< 65535, same as IP_MAX_MTU  */

#define ETH_P_IP        0x0800      /**< Internet Protocol packet   */
#define ETH_P_ARP       0x0806      /**< Address Resolution packet  */

/* ARP protocol HARDWARE identifiers. */
#define ARPHRD_NETROM        0      /**< from KA9Q: NET/ROM pseudo */
#define ARPHRD_ETHER         1      /**< Ethernet 10Mbps */
#define ARPHRD_EETHER        2      /**< Experimental Ethernet */
#define ARPHRD_AX25          3      /**< AX.25 Level 2 */
#define ARPHRD_PRONET        4      /**< PROnet token ring */
#define ARPHRD_CHAOS         5      /**< Chaosnet */
#define ARPHRD_IEEE802       6      /**< IEEE 802.2 Ethernet/TR/TB */
#define ARPHRD_ARCNET        7      /**< ARCnet */
#define ARPHRD_APPLETLK      8      /**< APPLEtalk */
#define ARPHRD_DLCI         15      /**< Frame Relay DLCI */
#define ARPHRD_ATM          19      /**< ATM */
#define ARPHRD_METRICOM     23      /**< Metricom STRIP (new IANA id) */
#define ARPHRD_IEEE1394     24      /**< IEEE 1394 IPv4 - RFC 2734 */
#define ARPHRD_EUI64        27      /**< EUI-64 */
#define ARPHRD_INFINIBAND   32      /**< InfiniBand */

/** MAC header layout */
struct ethhdr {
    uint8_t  h_dest[ETH_ALEN];      /**< destination eth addr   */
    uint8_t  h_source[ETH_ALEN];    /**< source ether addr  */
    uint16_t h_proto;               /**< packet type ID field   */
} __attribute__ ((packed));

/** the ARP message, see RFC 826 ("Packet format") */
struct arphdr {
    uint16_t hwtype;
    uint16_t proto;
    uint8_t  hwlen;
    uint8_t  protolen;
    uint16_t opcode;
    uint8_t  shwaddr[ETH_ALEN];
    uint32_t sipaddr;
    uint8_t  dhwaddr[ETH_ALEN];
    uint32_t dipaddr;
}  __attribute__ ((packed));

/** ARP protocol opcodes. */
enum etharp_opcode {
  ARP_REQUEST   = 1,                /**< ARP request */
  ARP_REPLY     = 2,                /**< ARP reply */
  ARP_RREQUEST  = 3,                /**< RARP request */
  ARP_RREPLY    = 4,                /**< RARP reply */
  ARP_InREQUEST = 8,                /**< InARP request */
  ARP_InREPLY   = 9,                /**< InARP reply */
  ARP_NAK       = 10                /**< (ATM)ARP NAK */
};

/** IP header layout */
struct iphdr {
    uint8_t  versionIhl;
    uint8_t  tos;
    uint16_t tot_len;
    uint16_t id;
    uint16_t frag_off;
    uint8_t  ttl;
    uint8_t  protocol;
    uint16_t check;
    uint32_t saddr;
    uint32_t daddr;
} __attribute__ ((packed));

/* special/well-known indices/entries in cache */
#define ARPCACHEIX_MN  0            /**< MN/240: 192.168.100.240 */
#define ARPCACHEIX_CN0 1            /**< CN0/1 : 192.168.100.1 */
#define ARPCACHEIX_GW  17           /**< GW    : 192.168.100.254 */

/* well-known CDI2 IP addresses */
#define NETMASK_PLK 0xffffff00u     /**< 255.255.255.0 */
#define NETADDR_PLK 0xc0a86400u     /**< 192.168.100.0 */
#define IPADDR_CN0  0xc0a86401u     /**< 192.168.100.1 */
#define IPADDR_CN1  0xc0a86402u     /**< 192.168.100.2 */
#define IPADDR_CN2  0xc0a86403u     /**< 192.168.100.3 */
#define IPADDR_CN3  0xc0a86404u     /**< 192.168.100.4 */
#define IPADDR_CN4  0xc0a86405u     /**< 192.168.100.5 */
#define IPADDR_CN5  0xc0a86406u     /**< 192.168.100.6 */
#define IPADDR_CN6  0xc0a86407u     /**< 192.168.100.7 */
#define IPADDR_CN7  0xc0a86408u     /**< 192.168.100.8 */
#define IPADDR_CN8  0xc0a86409u     /**< 192.168.100.9 */
#define IPADDR_CN9  0xc0a8640au     /**< 192.168.100.10 */
#define IPADDR_CNA  0xc0a8640bu     /**< 192.168.100.11*/
#define IPADDR_CNB  0xc0a8640cu     /**< 192.168.100.12 */
#define IPADDR_CNC  0xc0a8640du     /**< 192.168.100.13 */
#define IPADDR_CND  0xc0a8640eu     /**< 192.168.100.14 */
#define IPADDR_CNE  0xc0a8640fu     /**< 192.168.100.15 */
#define IPADDR_CNF  0xc0a86410u     /**< 192.168.100.16 */
#define IPADDR_MN   0xc0a864f0u     /**< 192.168.100.240 */
#define IPADDR_GW   0xc0a864feu     /**< 192.168.100.254 */

/** GW module instance state */
struct GwModule {
    uint32_t localIpAddress;            /**< local IP address on PowerLink segment; derived from node ID */
    uint32_t defaultGateway;            /**< default gateway address */
    uint8_t  localMacAddress[ETH_ALEN]; /**< local MAC address */
    uint8_t  arpCache[18][ETH_ALEN];    /**< our ARP cache: one entry for each possible CN, MN, and GW */
} mGW;

/**
 * put the MAC address of a well-known IP address into our ARP cache
 *
 * @param[in] ipAddr IP address to test
 * @param[in] macAddr MAC address to cache
 */
static inline void
gwUpdateArpCache(const uint32_t ipAddr, const uint8_t *macAddr)
{
    if ((ipAddr & NETMASK_PLK) == NETADDR_PLK) {
        /* this is on the PowerLink segment/network, test for well-known nodeID */
        const uint8_t nodeId = (ipAddr & ~NETMASK_PLK);

        if ((nodeId >= 1) && (nodeId <= 16))
            memcpy(mGW.arpCache[nodeId], macAddr, ETH_ALEN);
        else if (nodeId == 240)
            memcpy(mGW.arpCache[ARPCACHEIX_MN], macAddr, ETH_ALEN);
        else if ((nodeId == 254) || (ipAddr == mGW.defaultGateway))
            memcpy(mGW.arpCache[ARPCACHEIX_GW], macAddr, ETH_ALEN);
    }
}

/**
 * lookup the cached the MAC address of an IP address. Unknown IP addresses
 * will get the MAC address of the GW, if available.
 *
 * @param[in] ipAddr IP address to test
 * @param[out] macAddr cached MAC address.
 * @return true if destination MAC (might be the GW) found
 */
static inline bool
gwLookupArpCache(const uint32_t ipAddr, uint8_t *macAddr)
{
    const static uint8_t macZero[ETH_ALEN] = { 0, 0, 0, 0, 0, 0 };

    if ((ipAddr & NETMASK_PLK) == NETADDR_PLK) {
        /* this is on the PowerLink segment/network, test for well-known nodeID */
        const uint8_t nodeId = (ipAddr & ~NETMASK_PLK);

        if ((nodeId >= 1) && (nodeId <= 16))
            memcpy(macAddr, mGW.arpCache[nodeId], ETH_ALEN);
        else if (nodeId == 240)
            memcpy(macAddr, mGW.arpCache[ARPCACHEIX_MN], ETH_ALEN);
        else
            memcpy(macAddr, mGW.arpCache[ARPCACHEIX_GW], ETH_ALEN);
    } else
        memcpy(macAddr, mGW.arpCache[ARPCACHEIX_GW], ETH_ALEN);

    return memcmp(macAddr, macZero, ETH_ALEN) != 0;
}

/**
 * Query MAC address for an IP address.
 *
 * This function constructs and ARP query and broadcast
 * it in an ASYNC slot of the PowerLink cycle
 *
 * @param[in] IP address (host byte order) to query
 */
static void
gwIssueArpQuery(const uint32_t ipAddr)
{
    /* space for ARP query + MAC header */
    static uint8_t arpQuery[sizeof(struct ethhdr) + sizeof(struct arphdr)];

    struct ethhdr *pMac = (struct ethhdr *)arpQuery;
    memset(pMac->h_dest, 0xff, sizeof(pMac->h_dest));
    memcpy(pMac->h_source, mGW.localMacAddress, sizeof(pMac->h_source));
    pMac->h_proto = __builtin_bswap16(ETH_P_ARP);

    struct arphdr *pArp = (struct arphdr *)(arpQuery + sizeof(struct ethhdr));
    pArp->hwtype   = __builtin_bswap16(ARPHRD_ETHER);
    pArp->proto    = __builtin_bswap16(ETH_P_IP);
    pArp->hwlen    = ETH_ALEN;
    pArp->protolen = 4;
    pArp->opcode   = __builtin_bswap16(ARP_REQUEST);
    memcpy(pArp->shwaddr, mGW.localMacAddress, sizeof(pMac->h_source));
    pArp->sipaddr  = __builtin_bswap32(mGW.localIpAddress);
    memset(pArp->dhwaddr, 0, sizeof(pArp->dhwaddr));
    pArp->dipaddr  = __builtin_bswap32(ipAddr);

    /* intentionally ignore error */
    (void)oplk_sendEthFrame((tPlkFrame *)arpQuery, sizeof(arpQuery));
}

void
gwForwardIP(struct sbuf *pHead, size_t totLen)
{
    /** MAC frame buffer for IP datagrams received over PPP */
    static uint8_t macFrame[sizeof(struct ethhdr) + ETH_DATA_LEN];

    /* sanity */
    assert(pHead != NULL);
    assert(totLen > (sizeof(struct iphdr) + 2));

    /* ignore IP datagrams without complete header or oversized datagrams */
    if ((totLen < (sizeof(struct iphdr) + 2)) || (totLen > (ETH_DATA_LEN + 2)))
        return;

    /* totLen includes the PPP protocol field; take it off now */
    totLen -= 2;

    /* build MAC header */
    struct iphdr  *ipHdr  = (struct iphdr *)(pHead->data + 2);  /* + 2 strips the PPP protocol field */
    struct ethhdr *macHdr = (struct ethhdr *)macFrame;

    if (gwLookupArpCache(__builtin_bswap32(ipHdr->daddr), macHdr->h_dest) != true) {
        gwIssueArpQuery(__builtin_bswap32(ipHdr->daddr));

        /* <DesignToCost>just drop IP datagram and rely on upper protocol layers for retransmission</DesignToCost> */
        return;
    }

    /* TODO: will we ever use broadcast addresses? */
    memcpy(macHdr->h_source, mGW.localMacAddress, ETH_ALEN);
    macHdr->h_proto = __builtin_bswap16(ETH_P_IP);

    /* copy IP datagram data from sbuf header into MAC frame buffer */
    uint8_t *p       = macFrame + sizeof(struct ethhdr);
    size_t   copyLen = totLen > (sizeof(pHead->data) - 2) ? (sizeof(pHead->data) - 2) : totLen;
    memcpy(p, pHead->data + 2, copyLen);

    /* adjust target pointer and remaining length */
    p      += copyLen;
    totLen -= copyLen;

    /* copy rest of sbuf chain into MAC frame buffer */
    while ((totLen > 0) && (pHead->pNext != NULL)) {
        pHead   = pHead->pNext;
        copyLen = totLen > sizeof(pHead->data) ? sizeof(pHead->data) : totLen;

        memcpy(p, pHead->data, copyLen);

        p      += copyLen;
        totLen -= copyLen;
    }

    /* request transmission in next ASYNC phase */
    (void)oplk_sendEthFrame((tPlkFrame *)macFrame, p - macFrame);
}

/**
 * process a MAC frame holding an ARP PDU
 *
 * @param[in] pFrame non-PLK frame
 * @param[in] frameSize size of frame data
 */
static void
gwProcessArp(tPlkFrame *pFrame, size_t frameSize)
{
    /* ignore implausibly short frames */
    if (frameSize < sizeof(struct ethhdr) + sizeof(struct arphdr))
        return;

    struct arphdr *pArp = (struct arphdr *)((uint8_t *)pFrame + sizeof(struct ethhdr));

    if ((pArp->hwtype   != __builtin_bswap16(ARPHRD_ETHER)) ||
        (pArp->proto    != __builtin_bswap16(ETH_P_IP)) ||
        (pArp->hwlen    != 6) ||
        (pArp->protolen != 4)) {
        /* unrecognized ARP frame, ignore */
        return;
    }

    /* update ARP cache with relevant information from source */
    gwUpdateArpCache(__builtin_bswap32(pArp->sipaddr), pArp->shwaddr);

    if (pArp->opcode == __builtin_bswap16(ARP_REPLY))
        /* update ARP cache with relevant information from destination */
        gwUpdateArpCache(__builtin_bswap32(pArp->dipaddr), pArp->dhwaddr);

    /* answer query, if we're addressed */
    if ((pArp->opcode == __builtin_bswap16(ARP_REQUEST)) &&
        (pArp->dipaddr == __builtin_bswap32(mGW.localIpAddress))) {
        /* construct ARP query response in-situ */
        struct ethhdr *pEth = (struct ethhdr *)pFrame;
        memcpy(pEth->h_dest, pEth->h_source, ETH_ALEN);
        memcpy(pEth->h_source, mGW.localMacAddress, ETH_ALEN);

        pArp->opcode  = __builtin_bswap16(ARP_REPLY);
        pArp->dipaddr = pArp->sipaddr;
        pArp->sipaddr = __builtin_bswap32(mGW.localIpAddress);
        memcpy(pArp->dhwaddr, pArp->shwaddr, ETH_ALEN);
        memcpy(pArp->shwaddr, mGW.localMacAddress, ETH_ALEN);

        /* intentionally ignore error */
        (void)oplk_sendEthFrame(pFrame, frameSize);
    }
}

/**
 * process a MAC frame holding an IP datagram
 *
 * @param[in] pFrame non-PLK frame
 * @param[in] frameSize size of frame data
 */
static void
gwProcessIP(tPlkFrame *pFrame, size_t frameSize)
{
    assert(pFrame != NULL);
    assert(frameSize > sizeof(struct ethhdr));

    /* ignore implausibly short MAC(IP) frames */
    if (frameSize <= sizeof(struct ethhdr))
        return;

    const uint16_t ipLength = __builtin_bswap16(((struct iphdr *)((uint8_t *)pFrame + sizeof(struct ethhdr)))->tot_len);

    /* drop short/oversized frames */
    if ((ipLength <= sizeof(struct iphdr)) || (ipLength > ETH_DATA_LEN))
        return;

    /* strip MAC header off frame data and forward to PPP peer */
    pppIfFwdIP((uint8_t *)pFrame + sizeof(struct ethhdr), frameSize - sizeof(struct ethhdr));
}

void
gwProcessMac(tPlkFrame *pFrame, size_t frameSize)
{
    assert(pFrame != NULL);
    assert(frameSize > sizeof(struct ethhdr));

    /* must have at least MAC header */
    if (frameSize < sizeof(struct ethhdr))
        return;

    /* process ARP or IP frames, ignore all others */
    if (pFrame->etherType == __builtin_bswap16(ETH_P_ARP))
        gwProcessArp(pFrame, frameSize);
    else if (pFrame->etherType == __builtin_bswap16(ETH_P_IP))
        gwProcessIP(pFrame, frameSize);
    else
        /* ignore all other protocols */;
}

void
gwUpdateLocalAddress(const uint32_t ipAddr, const uint8_t macAddr[ETH_ALEN])
{
    mGW.localIpAddress = ipAddr;
    mGW.defaultGateway = IPADDR_GW;
    memcpy(mGW.localMacAddress, macAddr, ETH_ALEN);
    gwUpdateArpCache(mGW.localIpAddress, mGW.localMacAddress);
}

void
gwUpdateDefaultGateway(const uint32_t ipAddr)
{
    /* sanity: ignore GW on non-PLK segment */
    if ((ipAddr & NETMASK_PLK) != NETADDR_PLK)
        return;

    /* save new default GW IP address */
    mGW.defaultGateway = ipAddr;

    /* query MAC address of default gateway */
    gwIssueArpQuery(ipAddr);
}
