/**
 * @file cdi2If.c
 *
 * @brief implementation details of the CDI2 UART interface
 * @author AIT
 * @copyright &copy;2023 Austrian Institute of Technology (AIT)
 */

#include <assert.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>

#include <FreeRTOS.h>
#include <queue.h>
#include <semphr.h>

#include <xparameters.h>
#include <xinterrupt_wrap.h>
#include <xstatus.h>
#include <xuartns550.h>

#include "cdi2If.h"
#include "crc16.h"
#include "logging.h"
#include "sbuf.h"
#include "statistics.h"
#include "wd.h"

/** CDI frame characters/octets which must be escaped on the line */
enum CdiFrameCharacters {
    CDI_RSV0 = 0xf8,                        /**< reserved */
    CDI_RSV1 = 0xf9,                        /**< reserved */
    CDI_RSV2 = 0xfa,                        /**< reserved */
    CDI_RSV3 = 0xfb,                        /**< reserved */
    CDI_STX  = 0xfc,                        /**< start of frame */
    CDI_ETX  = 0xfd,                        /**< end of frame */
    CDI_ESC  = 0xfe                         /**< character/octet escape */
};

/** CDI2 MTU is BNRESULT: Id(1), BNID(4), Series/Denomination/Orientation(4), Segment(4), Supplementary Data (1400) */
#define CDI2IF_MTU 1413

/** CDI2 UART sbuf chain message */
struct MsgSbufChain {
    struct sbuf *p;                         /**< address of head of sbuf chain */
    size_t       len;                       /**< data length over complete chain */
};

/** unmarshalling status */
enum UmaStatus {
    UMA_IDLE = 0,                           /**< not in message */
    UMA_STX  = 1,                           /**< STX scanned, in-message */
    UMA_ESC  = 2                            /**< in-message and ESC scanned */
};

/** reassemble/unmarshalling status */
struct RaState {
    enum UmaStatus  umaStatus;              /**< unmarshalling status */
    struct sbuf    *pHead;                  /**< command message head of sbuf chain */
    struct sbuf    *pCurr;                  /**< current sbuf */
    size_t          totLen;                 /**< current total length over sbuf chain */
    size_t          ix;                     /**< index into current sbuf */
};

/** depth of CDI2 UART Tx task queue */
#define CDI2IF_TX_TASK_QUEUE_LENGTH     SBUF_RING_SIZE

/** CDI2 UART Tx task stack size in StackType_t (uint32_t) words */
#define CDI2IF_TX_TASK_STACK_SIZE       (2048 / sizeof(StackType_t))

/** CDI2 UART Tx task priority */
#define CDI2IF_TX_TASK_PRIORITY         (tskIDLE_PRIORITY + 2)

/** depth of CDI2 UART Reassemble task queue */
#define CDI2IF_RA_TASK_QUEUE_LENGTH     SBUF_RING_SIZE

/** CDI2 UART Reassemble task stack size in StackType_t (uint32_t) words */
#define CDI2IF_RA_TASK_STACK_SIZE       (4096 / sizeof(StackType_t))

/** CDI2 UART Reassemble task priority */
#define CDI2IF_RA_TASK_PRIORITY         (configMAX_PRIORITIES - 4)

/** CDI2IF module instance state */
struct Cdi2IfModule {
    SemaphoreHandle_t hSemTx;               /**< CDI2 UART Tx serialization semaphore: Taken by the Tx task, given by the UART ISR */
    QueueHandle_t     hQueueTx;             /**< CDI2 UART Tx queue */
    QueueHandle_t     hQueueRa;             /**< CDI2 UART Reassemble queue */
    TaskHandle_t      hTaskTx;              /**< CDI2 UART Tx task */
    TaskHandle_t      hTaskRa;              /**< CDI2 UART Reassemble task */
    struct RaState    raState;              /**< CDI frame reassembler state */
    XUartNs550        drvUart;              /**< CDI2 UART driver instance data */

    /** Rx data buffer */
    uint8_t rxBuffer[sizeof(((struct sbuf *)NULL)->data)];
} mCDI2IF;

/**
 * test whether a character/octet is a reserved CDI frame character
 *
 * @param[in] c test character
 * @return true, if c is reserved CDI frame character
 */
static inline bool
isCdiChar(uint8_t c)
{
    return (c >= CDI_RSV0) && (c <= CDI_ESC);
}

/**
 * return escaped variant of input character c
 *
 * @param[in] c input character
 * @return escaped character
 */
static inline uint8_t
escapeCdiChar(uint8_t c)
{
    return ~c;
}

/** marshall a character/octet into a serial buffer (assuming enough space is available) */
#define CDI_MARSHALL(pSbuf, ix, c)                  \
    do {                                            \
        if (__builtin_expect(isCdiChar(c), 0)) {    \
            pSbuf->data[ix++] = CDI_ESC;            \
            pSbuf->data[ix++] = escapeCdiChar(c);   \
        } else                                      \
            pSbuf->data[ix++] = c;                  \
    } while (0)

/** marshall a character/octet into a serial buffer and update running CRC16 */
#define CDI_MARSHALL_CRC(pSbuf, ix, c, crc) \
    do {                                    \
        crc = crc16Incr(c, crc);            \
        CDI_MARSHALL(pSbuf, ix, c);         \
    } while (0)

void
cdi2IfProtocolVersionAnswer(enum Cdi2CoreStatus  coreStatus,
                            uint8_t              nmtStatus,
                            enum BsmStateEnum    bsmStatus,
                            enum DeviceStateEnum devStatus,
                            uint8_t              mtcStatus,
                            uint8_t              errStatus)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 21);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, CCR_ProtocolVersionAnswer, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, coreStatus,                crc);
    CDI_MARSHALL_CRC(pSbuf, ix, 0x00,                      crc);
    CDI_MARSHALL_CRC(pSbuf, ix, 0x02,                      crc);
    CDI_MARSHALL_CRC(pSbuf, ix, nmtStatus,                 crc);
    CDI_MARSHALL_CRC(pSbuf, ix, bsmStatus,                 crc);
    CDI_MARSHALL_CRC(pSbuf, ix, devStatus,                 crc);
    CDI_MARSHALL_CRC(pSbuf, ix, mtcStatus,                 crc);
    CDI_MARSHALL_CRC(pSbuf, ix, errStatus,                 crc);
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

void
cdi2IfTtsReset()
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 7);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, CCR_TtsReset, crc);
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

void
cdi2IfNmtReset(enum Cdi2CoreResetCause cause)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 9);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, CCR_NmtReset, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, cause,        crc);
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

void
cdi2IfStartCdi2Answer()
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 7);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, CCR_StartCdi2Answer, crc);
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

void
cdi2IfBsmStatus(enum BsmStateEnum bsmStatus)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 9);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, CCR_BsmStatus, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, bsmStatus,     crc);
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

void
cdi2IfBanknoteId(uint32_t bnid,
                 uint32_t tcCount,
                 uint32_t tcTrigger,
                 uint32_t t_trig)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 39);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, CCR_BnId,                 crc);
    CDI_MARSHALL_CRC(pSbuf, ix, bnid              & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (bnid >> 8)       & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (bnid >> 16)      & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (bnid >> 24)      & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, tcCount           & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (tcCount >> 8)    & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (tcCount >> 16)   & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (tcCount >> 24)   & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, tcTrigger         & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (tcTrigger >> 8)  & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (tcTrigger >> 16) & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (tcTrigger >> 24) & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, t_trig            & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (t_trig >> 8)     & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (t_trig >> 16)    & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (t_trig >> 24)    & 0xff, crc);
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

void
cdi2IfBanknoteTrigger(bool     isBfa1,
                      uint32_t bnid)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 16);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, CCR_BnTrigger,       crc);
    CDI_MARSHALL_CRC(pSbuf, ix, isBfa1 ? 1 : 0,      crc);
    CDI_MARSHALL_CRC(pSbuf, ix, bnid         & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (bnid >> 8)  & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (bnid >> 16) & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (bnid >> 24) & 0xff, crc);
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

void
cdi2IfBanknoteInfo(uint32_t bnid,
                   uint8_t  series,
                   uint8_t  denomination,
                   uint8_t  orientation)
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 21);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, CCR_BnInfo,          crc);
    CDI_MARSHALL_CRC(pSbuf, ix, bnid         & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (bnid >> 8)  & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (bnid >> 16) & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, (bnid >> 24) & 0xff, crc);
    CDI_MARSHALL_CRC(pSbuf, ix, series,              crc);
    CDI_MARSHALL_CRC(pSbuf, ix, denomination,        crc);
    CDI_MARSHALL_CRC(pSbuf, ix, orientation,         crc);
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

void
cdi2IfPrepareUpdate()
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 7);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, CCR_PrepareUpdate, crc);
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

void
cdi2IfPerformUpdate()
{
    /* sanity; must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 7);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, CCR_PerformUpdate, crc);
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

void
cdi2IfLog(uint8_t  idByte,
          uint8_t  errStatus,
          char    *sMsg)
{
    /* sanity; at least idByte and errStatus and a '\0' must fit into single sbuf */
    assert(sizeof(((struct sbuf *)NULL)->data) >= 10);

    /* get sbuf (blocking) */
    struct sbuf *pSbuf = sbufGet(true);
    uint16_t     crc   = CRC16_INIT;
    int          ix    = 0;

    /* marshall message into serial buffer, assuming no overflow */
    pSbuf->pNext      = NULL;
    pSbuf->data[ix++] = CDI_STX;
    CDI_MARSHALL_CRC(pSbuf, ix, idByte, crc);

    if ((idByte >= CCR_Fatal) && (idByte <= CCR_Warning)) {
        CDI_MARSHALL_CRC(pSbuf, ix, errStatus, crc);
    }

    /* <DesignToCost>Limit error messages to one sbuf</DesignToCost> */
    while ((*sMsg != '\0') && ((ix + 2 + 1 + 5) <  sizeof(((struct sbuf *)NULL)->data))) {
        uint8_t c = *sMsg++;
        CDI_MARSHALL_CRC(pSbuf, ix, c, crc);
    }

    CDI_MARSHALL_CRC(pSbuf, ix, 0, crc);            /* force ASCIIZ */
    CDI_MARSHALL(pSbuf, ix, crc & 0xff);
    CDI_MARSHALL(pSbuf, ix, (crc >> 8) & 0xff);
    pSbuf->data[ix++] = CDI_ETX;

    /* send sbuf chain to Tx task */
    struct MsgSbufChain txMsg = {
        .p   = pSbuf,
        .len = ix
    };

    if ((mCDI2IF.hQueueTx == NULL) || (xQueueSend(mCDI2IF.hQueueTx, &txMsg, portMAX_DELAY) != pdTRUE)) {
        sbufPut(pSbuf);
        statsIncrCdi2IfTx();
    }
}

/**
 * allow external access to UART statistics
 *
 * @return address of AXI UART error/statistic counters
 */
XUartNs550Stats *
cdi2IfGetStats(void)
{
    return &mCDI2IF.drvUart.Stats;
}

/**
 * UART interrupt application event handler
 *
 * @param[in] arg callback reference from XUartNs550_SetHandler()
 * @param[in] ev callback event type
 * @param[in] evData callback event data
 */
static void
cdi2IfHandler(void *arg, uint32_t ev, unsigned int evData)
{
    /* unused */
    (void)arg;

    /* kick watchdog */
    wdKick();

    switch (ev) {
    case XUN_EVENT_RECV_DATA:
    case XUN_EVENT_RECV_TIMEOUT: {
        /* sanity */
        assert(evData <= sizeof(mCDI2IF.rxBuffer));
        assert(mCDI2IF.drvUart.ReceiveBuffer.NextBytePtr >= mCDI2IF.rxBuffer);
        assert(mCDI2IF.drvUart.ReceiveBuffer.NextBytePtr <= mCDI2IF.rxBuffer + sizeof(mCDI2IF.rxBuffer));

        if (evData > sizeof(mCDI2IF.rxBuffer))
            evData = sizeof(mCDI2IF.rxBuffer);

        /* keep collecting RX characters if sbuf not full and no CDI end-of-frame present */
        if ((evData < sizeof(mCDI2IF.rxBuffer)) && (memchr(mCDI2IF.rxBuffer, CDI_ETX, evData) == NULL))
            break;

        /* forward to reassemble task */
        struct MsgSbufChain raMsg = {
                .p   = sbufGetIsr(),
                .len = evData
        };

        if (raMsg.p != NULL) {
            memcpy(raMsg.p->data, mCDI2IF.rxBuffer, evData);

            if ((mCDI2IF.hQueueRa == NULL) || (xQueueSendFromISR(mCDI2IF.hQueueRa, &raMsg, NULL) != pdTRUE)) {
                sbufPutIsr(raMsg.p);
                statsIncrCdi2IfRa();
            }
        } else
            statsIncrCdi2IfRa();

        /* reset RX data buffer */
        mCDI2IF.drvUart.ReceiveBuffer.NextBytePtr    = mCDI2IF.rxBuffer;
        mCDI2IF.drvUart.ReceiveBuffer.RequestedBytes = sizeof(mCDI2IF.rxBuffer);
        mCDI2IF.drvUart.ReceiveBuffer.RemainingBytes = sizeof(mCDI2IF.rxBuffer);

        break;
    }

    case XUN_EVENT_SENT_DATA:
        /* data buffer transmitted */

        /* release CDI2 UART Tx semaphore, intentionally ignore return value. */
        (void)xSemaphoreGiveFromISR(mCDI2IF.hSemTx, NULL);
        break;

    case XUN_EVENT_RECV_ERROR: {
        /* sanity */
        assert(evData <= sizeof(mCDI2IF.rxBuffer));
        assert(mCDI2IF.drvUart.ReceiveBuffer.NextBytePtr >= mCDI2IF.rxBuffer);
        assert(mCDI2IF.drvUart.ReceiveBuffer.NextBytePtr <= mCDI2IF.rxBuffer + sizeof(mCDI2IF.rxBuffer));

        if (evData <= 0)
            break;
        else if (evData > sizeof(mCDI2IF.rxBuffer))
            evData = sizeof(mCDI2IF.rxBuffer);

        /* forward to reassemble task */
        struct MsgSbufChain raMsg = {
                .p   = sbufGetIsr(),
                .len = evData
        };

        if (raMsg.p != NULL) {
            memcpy(raMsg.p->data, mCDI2IF.rxBuffer, evData);

            if ((mCDI2IF.hQueueRa == NULL) || (xQueueSendFromISR(mCDI2IF.hQueueRa, &raMsg, NULL) != pdTRUE)) {
                sbufPutIsr(raMsg.p);
                statsIncrCdi2IfRa();
            }
        } else
            statsIncrCdi2IfRa();

        /* reset RX data buffer */
        mCDI2IF.drvUart.ReceiveBuffer.NextBytePtr    = mCDI2IF.rxBuffer;
        mCDI2IF.drvUart.ReceiveBuffer.RequestedBytes = sizeof(mCDI2IF.rxBuffer);
        mCDI2IF.drvUart.ReceiveBuffer.RemainingBytes = sizeof(mCDI2IF.rxBuffer);

        break;
    }

    case XUN_EVENT_MODEM:
        /* modem status lines changed */
        break;

    default:
        /* ignore unknown event codes */
        break;
    }
}

void
cdi2IfPollUartIsr()
{
    extern unsigned int XUartNs550_ReceiveBuffer(XUartNs550 *InstancePtr);

    /* save original position in mCDI2IF.rxBuffer */
    const uint8_t *p0 = mCDI2IF.drvUart.ReceiveBuffer.NextBytePtr;

    /* poll CDI2 UART */
    const unsigned int rxCnt = XUartNs550_ReceiveBuffer(&mCDI2IF.drvUart);

    if (rxCnt > 0) {
        /* forward to reassembly task, if RX data buffer full or CDI end-of-frame now present */
        if ((mCDI2IF.drvUart.ReceiveBuffer.RemainingBytes == 0) || (memchr(p0, CDI_ETX, rxCnt) != NULL)) {
            struct MsgSbufChain raMsg = {
                    .p   = sbufGetIsr(),
                    .len = sizeof(mCDI2IF.rxBuffer) - mCDI2IF.drvUart.ReceiveBuffer.RemainingBytes
            };

            if (raMsg.p != NULL) {
                memcpy(raMsg.p->data, mCDI2IF.rxBuffer, raMsg.len);

                if ((mCDI2IF.hQueueRa == NULL) || (xQueueSendFromISR(mCDI2IF.hQueueRa, &raMsg, NULL) != pdTRUE)) {
                    sbufPutIsr(raMsg.p);
                    statsIncrCdi2IfRa();
                }
            } else
                statsIncrCdi2IfRa();

            /* reset RX data buffer */
            mCDI2IF.drvUart.ReceiveBuffer.NextBytePtr    = mCDI2IF.rxBuffer;
            mCDI2IF.drvUart.ReceiveBuffer.RequestedBytes = sizeof(mCDI2IF.rxBuffer);
            mCDI2IF.drvUart.ReceiveBuffer.RemainingBytes = sizeof(mCDI2IF.rxBuffer);
        }
    }
}

/**
 * CDI2 UART Tx task
 *
 * @param[in] arg task parameter passed via xTaskCreate()
 */
static void
cdi2IfTxTask(void *arg)
{
    /* unused */
    (void)arg;

    LOG_TRACE("# CDI2IF-TX: started\r\n");

    /* FOREVER */
    for (;;) {
        /* receive Tx request */
        struct MsgSbufChain req;

        if (xQueueReceive(mCDI2IF.hQueueTx, &req, portMAX_DELAY) == pdFALSE)
            /* time-out, nothing received */
            continue;

        /* sanity */
        assert(req.p != NULL);
        assert(req.len > 0);

        /* transmit sbuf chain */
        while ((req.p != NULL ) && (req.len > 0)) {
            /* get current head of sbuf chain */
            struct sbuf *p = req.p;

            /* calculate length of Tx segment */
            size_t txLen = 0;

            if (req.len > sizeof(p->data))
                txLen = sizeof(p->data);
            else
                txLen = req.len;

            /* serialize Tx request at UART Tx semaphore */
            xSemaphoreTake(mCDI2IF.hSemTx, portMAX_DELAY);

            /* UART Tx is now free, send txLen characters/octets */
            (void)XUartNs550_Send(&mCDI2IF.drvUart, p->data, txLen);

            /* adjust requested length for sent data */
            req.len -= txLen;

            /* get transmitted sbuf off the chain and return to ring */
            req.p    = req.p->pNext;
            p->pNext = NULL;
            sbufPut(p);
        }

        /* marshalling may add redundant sbuf to the chain; silently remove them now */
        sbufPut(req.p);
    }
}

/**
 * unmarshal and reassemble a CDI2 UART Rx character stream into CDI2 commands/messages
 *
 * @param[in] c input character/octett
 * @param[in] pRaState reassembly/unmarshall state
 */
static void
cdi2IfReassemble(uint8_t         c,
                 struct RaState *pRaState)
{
    /* sanity */
    assert(pRaState != NULL);

    switch (pRaState->umaStatus) {
    case UMA_IDLE:
        if (c == CDI_STX) {
            /* sanity */
            assert(pRaState->pHead  == NULL);
            assert(pRaState->pCurr  == NULL);
            assert(pRaState->totLen == 0);
            assert(pRaState->ix     == 0);

            /* STX scanned, start message reassembly */
            pRaState->totLen = 0;
            pRaState->ix     = 0;
            pRaState->pHead  = sbufGet(true);
            pRaState->pCurr  = pRaState->pHead;

            /* goto UMA_STX */
            pRaState->umaStatus = UMA_STX;
        }

        break;

    case UMA_STX:
        if (c == CDI_STX) {
            /* STX scanned, drop the improperly terminated frame */
            if (pRaState->totLen > 0)
                logFormat(WARN, ERR_APPLICATION, "CDI2IF-RA: unterminated CDI frame ID 0x%02x %u", pRaState->pHead->data[0], pRaState->totLen);
            else
                logMsg(WARN, ERR_APPLICATION, "CDI2IF-RA: unterminated CDI frame");

            sbufPut(pRaState->pHead);

            /* restart message reassembly */
            pRaState->totLen = 0;
            pRaState->ix     = 0;
            pRaState->pHead  = sbufGet(true);
            pRaState->pCurr  = pRaState->pHead;

            /* stay in UMA_STX */
            break;
        } else if (c == CDI_ETX) {
            /* STX/ETX scanned, publish unmarshalled, reassembled message */
            if (pRaState->totLen <= 2) {
                /* drop empty messages */
                logMsg(WARN, ERR_APPLICATION, "CDI2IF-RA: short CDI frame");
                sbufPut(pRaState->pHead);

                pRaState->totLen = 0;
                pRaState->ix     = 0;
                pRaState->pHead  = NULL;
                pRaState->pCurr  = NULL;

                /* goto RAS_IDLE */
                pRaState->umaStatus = UMA_IDLE;
                break;
            } else {
                /* sanity */
                assert(pRaState->pHead != NULL);
                assert(pRaState->pCurr != NULL);

                if (crc16Check(pRaState->pHead, pRaState->totLen)) {
                    const QueueHandle_t hCdi2Queue = cdi2GetMessageQueue();

                    if (hCdi2Queue == NULL) {
                        /* CDI2 layer not up, silently drop frame */
                        sbufPut(pRaState->pHead);
                    } else {
                        struct Cdi2Message cdi2Msg = {
                            .id       = CDI2_MSG_CDIFRAME,
                            .cdiFrame = {
                                .p   = pRaState->pHead,
                                .len = pRaState->totLen
                            }
                        };

                        if (xQueueSend(hCdi2Queue, &cdi2Msg, portMAX_DELAY) != pdTRUE) {
                            sbufPut(pRaState->pHead);
                            logFormat(ERROR, ERR_APPLICATION, "CDI2IF-RA: CDI2 queue overflow, CDI frame ID 0x%02x", pRaState->pHead->data[0]);
                        }
                    }
                } else {
                    logFormat(WARN, ERR_APPLICATION, "CDI2IF-RA: CDI frame ID 0x%02x CRC error %u", pRaState->pHead->data[0], pRaState->totLen);
                    sbufPut(pRaState->pHead);
                }

                pRaState->totLen = 0;
                pRaState->ix     = 0;
                pRaState->pHead  = NULL;
                pRaState->pCurr  = NULL;

                /* goto UMA_IDLE */
                pRaState->umaStatus = UMA_IDLE;
                break;
            }
        } else if (c == CDI_ESC) {
            /* goto UMA_ESC */
            pRaState->umaStatus = UMA_ESC;
            break;
        } else {
            /* sanity */
            assert(pRaState->pHead  != NULL);
            assert(pRaState->pCurr  != NULL);
            assert(pRaState->ix < sizeof(pRaState->pCurr->data));

            if (pRaState->totLen >= CDI2IF_MTU) {
                /* drop oversized CDI frame */
                logFormat(WARN, ERR_APPLICATION, "CDI2IF-RA: oversized CDI frame ID 0x%02x", pRaState->pHead->data[0]);
                sbufPut(pRaState->pHead);

                pRaState->totLen = 0;
                pRaState->ix     = 0;
                pRaState->pHead  = NULL;
                pRaState->pCurr  = NULL;

                /* goto UMA_IDLE */
                pRaState->umaStatus = UMA_IDLE;
                break;
            } else {
                /* push message byte */
                pRaState->pCurr->data[pRaState->ix++] = c;
                pRaState->totLen++;

                /* grow sbuf chain, if current buffer full */
                if (pRaState->ix >= sizeof(pRaState->pCurr->data)) {
                    pRaState->pCurr->pNext = sbufGet(true);
                    pRaState->pCurr        = pRaState->pCurr->pNext;
                    pRaState->ix           = 0;
                }

                /* stay in UMA_STX */
                break;
            }
        }

    case UMA_ESC:
        /* sanity */
        assert(pRaState->pHead  != NULL);
        assert(pRaState->pCurr  != NULL);
        assert(pRaState->ix < sizeof(pRaState->pCurr->data));

        if (pRaState->totLen >= CDI2IF_MTU) {
            /* drop oversized CDI frame */
            logFormat(WARN, ERR_APPLICATION, "CDI2IF-RA: oversized CDI frame ID 0x%02x", pRaState->pHead->data[0]);
            sbufPut(pRaState->pHead);

            pRaState->totLen = 0;
            pRaState->ix     = 0;
            pRaState->pHead  = NULL;
            pRaState->pCurr  = NULL;

            /* goto UMA_IDLE */
            pRaState->umaStatus = UMA_IDLE;
            break;
        } else {
            /* push message byte */
            pRaState->pCurr->data[pRaState->ix++] = ~c;
            pRaState->totLen++;

            /* grow sbuf chain, if current buffer full */
            if (pRaState->ix >= sizeof(pRaState->pCurr->data)) {
                pRaState->pCurr->pNext = sbufGet(true);
                pRaState->pCurr        = pRaState->pCurr->pNext;
                pRaState->ix           = 0;
            }

            /* goto UMA_STX */
            pRaState->umaStatus = UMA_STX;
            break;
        }

    default:
        break;
    }
}

/**
 * CDI2 UART Reassemble task
 *
 * @param arg task parameter passed via xTaskCreate()
 */
static void
cdi2IfRaTask(void *arg)
{
    /* unused */
    (void)arg;

    LOG_TRACE("# CDI2IF-RA: started\r\n");

    /* FOREVER */
    for (;;) {
        /* receive reassembly request */
        struct MsgSbufChain req;

        if (xQueueReceive(mCDI2IF.hQueueRa, &req, portMAX_DELAY) == pdFALSE)
            /* time-out, nothing received */
            continue;

        /* sanity */
        assert(req.p != NULL);
        assert(req.len > 0);

        /* feed sbuf chain into reassembler */
        for (struct sbuf *p = req.p; (p != NULL) && (req.len > 0); p = p->pNext) {
            /* calculate length of reassemble segment */
            size_t raLen = 0;

            if (req.len > sizeof(p->data))
                raLen = sizeof(p->data);
            else
                raLen = req.len;

            /* unmarshall and reassemble */
            for (unsigned int i = 0; i < raLen; i++)
                cdi2IfReassemble(p->data[i], &mCDI2IF.raState);

            /* adjust received length for processed data */
            req.len -= raLen;
        }

        /* release sbuf chain */
        sbufPut(req.p);
    }
}

int
cdi2IfSetup(unsigned long speed)
{
    int ret = XST_SUCCESS;

    /* sanity */
    assert(speed > 0);

    if (speed <= 0)
        return XST_UART_BAUD_RANGE;

    LOG_TRACE("# CDI2IF: setup CDI2 UART\r\n");

    /* reset CDI frame reassmebler stte */
    mCDI2IF.raState.umaStatus = UMA_IDLE;
    mCDI2IF.raState.pHead     = NULL;
    mCDI2IF.raState.pCurr     = NULL;
    mCDI2IF.raState.totLen    = 0;
    mCDI2IF.raState.ix        = 0;

    /* initialize CDI2 UART driver instance data */
    memset(&mCDI2IF.drvUart, 0, sizeof(mCDI2IF.drvUart));

    if ((ret = XUartNs550_Initialize(&mCDI2IF.drvUart, XPAR_CDI2_UART_DEVICE_ID)) != XST_SUCCESS)
        return ret;

    /* driver self test */
    if ((ret = XUartNs550_SelfTest(&mCDI2IF.drvUart)) != XST_SUCCESS)
        return ret;

    /* prepare RX data buffer */
    mCDI2IF.drvUart.ReceiveBuffer.RequestedBytes = sizeof(mCDI2IF.rxBuffer);
    mCDI2IF.drvUart.ReceiveBuffer.RemainingBytes = sizeof(mCDI2IF.rxBuffer);
    mCDI2IF.drvUart.ReceiveBuffer.NextBytePtr    = mCDI2IF.rxBuffer;

    /* configure speed, 8N1 format */
    XUartNs550Format format = {
            .BaudRate = speed,
            .DataBits = XUN_FORMAT_8_BITS,
            .Parity   = XUN_FORMAT_NO_PARITY,
            .StopBits = XUN_FORMAT_1_STOP_BIT
    };

    if ((ret = XUartNs550_SetDataFormat(&mCDI2IF.drvUart, &format)) != XST_SUCCESS)
        return ret;

    XUartNs550_GetDataFormat(&mCDI2IF.drvUart, &format);

    LOG_TRACE("# CDI2IF: 8N1 format, speed %u bps\r\n", format.BaudRate);

    /* connect CDI2 UART interrupt to UART driver handler */
    LOG_TRACE("# CDI2IF: register UART interrupt handler\r\n");

    /* Register UART driver interrupt callback for the CDI2 UART */
    XConnectToInterruptCntrl(XPAR_INTERRUPTCONTROLLER_CDI2_UART_IP2INTC_IRPT_INTR,
                             (XInterruptHandler)XUartNs550_InterruptHandler,
                             (void *)&mCDI2IF.drvUart,
                             XPAR_TICKTIMER_INTR_PARENT);
    XEnableIntrId(XPAR_INTERRUPTCONTROLLER_CDI2_UART_IP2INTC_IRPT_INTR,
                  XPAR_TICKTIMER_INTR_PARENT);

    /* register application UART event handler */
    XUartNs550_SetHandler(&mCDI2IF.drvUart, cdi2IfHandler, &mCDI2IF.drvUart);

    /* enable data rx/tx interrupts, clear & enable FIFOs */
    XUartNs550_SetOptions(&mCDI2IF.drvUart, XUN_OPTION_DATA_INTR |
                                            XUN_OPTION_RXLINE_INTR |
                                            XUN_OPTION_FIFOS_ENABLE |
                                            XUN_OPTION_RESET_TX_FIFO |
                                            XUN_OPTION_RESET_RX_FIFO);

    /* set FIFO thresholds (default 8 characters) */
    XUartNs550_SetFifoThreshold(&mCDI2IF.drvUart, XUN_FIFO_TRIGGER_08);

    LOG_TRACE("# CDI2IF: setup TX task, queue %u x |%lu|\r\n", CDI2IF_TX_TASK_QUEUE_LENGTH, sizeof(struct MsgSbufChain));

    /* create CDI2 UART Tx semaphore and Give() it initially */
    mCDI2IF.hSemTx = xSemaphoreCreateBinary();

    assert(mCDI2IF.hSemTx != NULL);

    if (mCDI2IF.hSemTx == NULL)
        return XST_FAILURE;

    (void)xSemaphoreGive(mCDI2IF.hSemTx);           /* intentionally ignore return */

    /* create CDI2 UART Tx task queue */
    mCDI2IF.hQueueTx = xQueueCreate(CDI2IF_TX_TASK_QUEUE_LENGTH,
                                    sizeof(struct MsgSbufChain));

    assert(mCDI2IF.hQueueTx != NULL);

    if (mCDI2IF.hQueueTx == NULL)
        return XST_FAILURE;

    /* create CDI2 UART Tx task */
    xTaskCreate(cdi2IfTxTask,
                "CDI2IF-TX",
                CDI2IF_TX_TASK_STACK_SIZE,
                NULL,
                CDI2IF_TX_TASK_PRIORITY,
                &mCDI2IF.hTaskTx);

    assert(mCDI2IF.hTaskTx != NULL);

    if (mCDI2IF.hTaskTx == NULL)
        return XST_FAILURE;

    LOG_TRACE("# CDI2IF: setup ReAssemble task, queue %u x |%lu|\r\n", CDI2IF_RA_TASK_QUEUE_LENGTH, sizeof(struct MsgSbufChain));

    /* create CDI2 UART Reassemble task queue */
    mCDI2IF.hQueueRa = xQueueCreate(CDI2IF_RA_TASK_QUEUE_LENGTH,
                                    sizeof(struct MsgSbufChain));

    assert(mCDI2IF.hQueueRa != NULL);

    if (mCDI2IF.hQueueRa == NULL)
        return XST_FAILURE;

    /* create CDI2 UART Reassemble task */
    xTaskCreate(cdi2IfRaTask,
                "CDI2IF-RA",
                CDI2IF_RA_TASK_STACK_SIZE,
                NULL,
                CDI2IF_RA_TASK_PRIORITY,
                &mCDI2IF.hTaskRa);

    assert(mCDI2IF.hTaskRa != NULL);

    if (mCDI2IF.hTaskRa == NULL)
        return XST_FAILURE;

    return XST_SUCCESS;
}
