/**
 * @file cdi2.c
 * @brief CDI2 Device CDI2 layer; CN/Device variant
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#include <assert.h>

#include "xiltimer.h"

#include <FreeRTOS.h>
#include <queue.h>
#include <task.h>

#include <oplkcfg.h>
#include <oplk/oplk.h>
#include <common/target.h>

#include "cdi2.h"
#include "cdi2If.h"
#include "logging.h"
#include "powerlink.h"
#include "statistics.h"
#include "tts.h"
#include "wd.h"

/** length/depth of CDI2 layer message queue */
#define CDI2_MESSAGE_QUEUE_LENGTH 16

/** CDI2 layer task stack size in StackType_t (uint32_t) words*/
#define CDI2_TASK_STACK_SIZE (2048 / sizeof(StackType_t))

/** CDI2 layer task priority */
#define CDI2_TASK_PRIORITY (configMAX_PRIORITIES - 3)

/** CDI2 module instance state */
struct Cdi2Module {
    QueueHandle_t        hQueueMessage;         /**< handle of CDI2 layer message queue */
    TaskHandle_t         hTaskCdi2;             /**< handle of CDI2 layer task */
    enum Cdi2CoreStatus  cdi2CoreState;         /**< CDI2 IP Core state */
    enum DeviceStateEnum cdi2DeviceState;       /**< CDI2 Device Status */
    uint8_t              cdi2ErrorState;        /**< CDI2 error status */
    uint8_t              cdi2MaintenanceState;  /**< CDI2 maintenance status */
    bool                 isTtsDevice;           /**< USE TTS interface signals */
    enum TcCheck         tcStatus;              /**< last TC check result */
    unsigned int         ttsResetDuration;      /**< length of TTS reset pulse */
    struct BnTrigger     bnTriggerBfa1;         /**< last BSMINFO BN announcement BFA#1 */
    struct BnTrigger     bnTriggerBfa2;         /**< last BSMINFO BN announcement BFA#2 */
    uint64_t             tsTrigger;             /**< BSMINFO.tc_trigger (BFA#1) converted to system ticks */
    long                 meanTriggerOffset;     /**< average offset from tsTrigger to actual BPPRESENT time stamp */
    long                 meanTriggerLatency;    /**< average time between BPPRESENT time stamp and task processing of BPPRESENT message */
    long                 numTrigger;            /**< number of trigger events used to calculate the average */
} mCDI2;

QueueHandle_t
cdi2GetMessageQueue(void)
{
    return mCDI2.hQueueMessage;
}

enum Cdi2CoreStatus
cdi2GetCoreStatus(void)
{
    return mCDI2.cdi2CoreState;
}

enum DeviceStateEnum
cdi2GetDeviceStatus(void)
{
    return mCDI2.cdi2DeviceState;
}

void
cdi2SetDeviceStatus(const enum DeviceStateEnum ds)
{
    /* sanity; allow DS_INVALID */
    assert((ds >= DS_INVALID) && (ds <= DS_LAST_STATE));
    mCDI2.cdi2DeviceState = ds;
}

uint8_t
cdi2GetErrorStatus(void)
{
    return mCDI2.cdi2ErrorState;
}

void
cdi2SetErrorStatus(uint8_t es)
{
    mCDI2.cdi2ErrorState = es;
}

uint8_t
cdi2GetMaintenanceStatus(void)
{
    return mCDI2.cdi2MaintenanceState;
}

void
cdi2SetMaintenanceStatus(uint8_t ms)
{
    mCDI2.cdi2MaintenanceState = ms;
}

void
cdi2CheckTc(void)
{
    const QueueHandle_t hCdi2Queue = cdi2GetMessageQueue();

    if (hCdi2Queue == NULL)
        return;

    if (mCDI2.isTtsDevice && (mCDI2.tcStatus == TTSTC_OK)) {
        if ((mCDI2.cdi2DeviceState == DS_READY_TO_SORT) || (mCDI2.cdi2DeviceState == DS_SORTING)) {
            mCDI2.tcStatus = ttsCheckTc();

            if (mCDI2.tcStatus == TTSTC_NOCLOCK) {
                static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_TtsNoClock };

                if (xQueueSendFromISR(hCdi2Queue, &cdi2Msg, NULL) != pdTRUE)
                    statsIncrCdi2Queue();
            } else if (mCDI2.tcStatus == TTSTC_RANGE) {
                static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_TtsClockRange };

                if (xQueueSendFromISR(hCdi2Queue, &cdi2Msg, NULL) != pdTRUE)
                    statsIncrCdi2Queue();
            }
        }
    }
}

void
cdi2CheckReset(void)
{
    const QueueHandle_t hCdi2Queue = cdi2GetMessageQueue();

    if (hCdi2Queue == NULL)
        return;

    if (mCDI2.isTtsDevice) {
        const unsigned int ttsReset = ttsSampleReset();

        if (ttsReset == 0) {
            mCDI2.ttsResetDuration = 0;
        } else {
            if (mCDI2.ttsResetDuration < kTtsResetMinLength) {
                ++mCDI2.ttsResetDuration;
            } else if (mCDI2.ttsResetDuration == kTtsResetMinLength) {
                /* valid TTS:RESET pulse scanned, notify CDI2 layer */
                static struct Cdi2Message cdi2Msg = { .id = CDI2_MSG_TtsReset };

                if (xQueueSendFromISR(hCdi2Queue, &cdi2Msg, NULL) != pdTRUE)
                    statsIncrCdi2Queue();

                mCDI2.ttsResetDuration++;
            }
        }
    }
}

/** clear/reset TTS signal status */
static inline void
cdi2ResetTtsState(void)
{
    mCDI2.ttsResetDuration = 0;
    mCDI2.tcStatus         = TTSTC_OK;
}
/** clear/reset CDI2 BN trigger statistics and status */
static inline void
cdi2ResetBnTriggerState(void)
{
    mCDI2.bnTriggerBfa1.id         = BNINVALID;
    mCDI2.bnTriggerBfa1.tc_count   = 0;
    mCDI2.bnTriggerBfa1.tc_trigger = 0;
    mCDI2.bnTriggerBfa2.id         = BNINVALID;
    mCDI2.bnTriggerBfa2.tc_count   = 0;
    mCDI2.bnTriggerBfa2.tc_trigger = 0;
    mCDI2.tsTrigger                = 0;
    mCDI2.meanTriggerOffset        = 0;
    mCDI2.meanTriggerLatency       = 0;
    mCDI2.numTrigger               = 0;
}

/**
 * CDI2 LayerTask
 *
 * @param[in] pArg task argument set by xTaskCreate
 */
static void
cdi2Task(void *arg)
{
    /* unused */
    (void)arg;

    LOG_TRACE("# CDI2: started\r\n");

    /* reset TTS signal status */
    cdi2ResetTtsState();

    /* reset BN trigger status/statistics */
    cdi2ResetBnTriggerState();

    /* start initial protocol version handshake */
    mCDI2.isTtsDevice   = false;
    mCDI2.cdi2CoreState = CCS_IP_CORE_STARTUP;
    cdi2IfProtocolVersionAnswer(cdi2GetCoreStatus(),
                                plkGetNmtStatus(),
                                plkGetLastBsmState(),
                                cdi2GetDeviceStatus(),
                                cdi2GetMaintenanceStatus(),
                                cdi2GetErrorStatus());

    /* FOREVER */
    for (;;) {
        /* receive request */
        struct Cdi2Message msg;

        if (xQueueReceive(mCDI2.hQueueMessage, &msg, portMAX_DELAY) == pdFALSE)
            /* time-out, nothing received */
            continue;

        switch (msg.id) {
        case CDI2_MSG_nil:
            break;

        case CDI2_MSG_NmtStateChange:
            /* reset reported CDI2 device status, if CN leaves the active PLK states */
            if (!NMT_IF_ACTIVE_CN(msg.nmtStateChange.newNmtState)) {
                if (NMT_IF_ACTIVE_CN(msg.nmtStateChange.oldNmtState)) {
                    enum Cdi2CoreResetCause cause = CCRC_Application;

                    switch (msg.nmtStateChange.newNmtState) {
                    case kNmtGsInitialising:
                        cause = CCRC_NMT_SwReset;
                        break;
                    case kNmtGsResetApplication:
                        cause = CCRC_NMT_ResetNode;
                        break;
                    case kNmtGsResetCommunication:
                        cause = CCRC_NMT_ResetCommunication;
                        break;
                    case kNmtGsResetConfiguration:
                        cause = CCRC_NMT_ResetConfiguration;
                        break;
                    }

                    /* notify Application about NMT reset */
                    cdi2IfNmtReset(cause);
                }

                plkSetErrorThresholds(kRelaxedErrorThreshold);
                plkResetBnResultQueues();
                ttsSetDeviceReady(false);
                cdi2ResetTtsState();
                ttsResetTcMeas();
                cdi2SetDeviceStatus(DS_INVALID);
                cdi2ResetBnTriggerState();
            } else if (!NMT_IF_ACTIVE_CN(msg.nmtStateChange.oldNmtState)) {
                plkSetErrorThresholds(kRelaxedErrorThreshold);
                plkResetBnResultQueues();
                ttsSetDeviceReady(false);
                cdi2ResetTtsState();
                ttsResetTcMeas();
                cdi2SetDeviceStatus(DS_START_UP);
                cdi2ResetBnTriggerState();
            }

            break;

        case CDI2_MSG_BsmStateChange:
            /* notify application about new BSM status */
            cdi2IfBsmStatus(msg.bsmStateChange.newBsmState);

            /* only <=10ms for DS_READY_TO_SORT to DS_SORTING not possible with UART + PLK round trip */
            if ((msg.bsmStateChange.newBsmState == BS_SORTING) && (cdi2GetDeviceStatus() == DS_READY_TO_SORT))
                cdi2SetDeviceStatus(DS_SORTING);

            /* only <=10ms for DS_READY_TO_SHUT_DOWN to DS_SHUTDOWN not possible with UART + PLK round trip */
            if ((msg.bsmStateChange.newBsmState == BS_SHUTDOWN) && (cdi2GetDeviceStatus() == DS_READY_TO_SHUT_DOWN))
                cdi2SetDeviceStatus(DS_SHUTDOWN);

            break;

        case CDI2_MSG_UpdateSwCommand:
            /* notify application about change in SDO UpdateSwCommand */
            if (msg.swUpdateCommand.swuCmd == USC_PREPARE_UPDATE)
                cdi2IfPrepareUpdate();
            else if (msg.swUpdateCommand.swuCmd == USC_PERFORM_UPDATE)
                cdi2IfPerformUpdate();
            else
                plkSetSwUpdateStatus(USS_IDLE);

            break;

        case CDI2_MSG_TtsReset :
            if (cdi2GetDeviceStatus() != DS_START_UP) {
                plkSetErrorThresholds(kRelaxedErrorThreshold);
                plkResetBnResultQueues();
                ttsSetDeviceReady(false);
                cdi2ResetTtsState();
                ttsResetTcMeas();
                cdi2ResetBnTriggerState();

                /* notify Application about TTS reset */
                cdi2IfTtsReset();
            }

            break;

        case CDI2_MSG_TtsNoClock:
            logMsg(ERROR, ERR_TTS, "loss_of_TC");
            break;

        case CDI2_MSG_TtsClockRange:
            logMsg(WARN, ERR_TTS, "TC_out_of_range");
            break;

        case CDI2_MSG_PlkStarted:
            cdi2IfStartCdi2Answer();
            logMsg(INFO, 0, "" SW_VERSION ", " __DATE__ "T" __TIME__);
            break;

        case CDI2_MSG_BNINFIO:
            /* notify application about BNINFO */
            if (msg.bninfo.id != BNINVALID)
                cdi2IfBanknoteInfo(msg.bninfo.id, msg.bninfo.series, msg.bninfo.denomination, msg.bninfo.orientation);

            break;

        case CDI2_MSG_BNTRIGGER:
            if (msg.bnTrigger.is2ndBfa == 0) {
                mCDI2.bnTriggerBfa1 = msg.bnTrigger.bntrigger;

                if (mCDI2.isTtsDevice)
                    mCDI2.tsTrigger = ttsTc2SysTick(mCDI2.bnTriggerBfa1.tc_trigger);
                else
                    mCDI2.tsTrigger = ttsScheduleBp(mCDI2.bnTriggerBfa1.tc_trigger);

                /* notify application about BN trigger announcement */
                cdi2IfBanknoteId(mCDI2.bnTriggerBfa1.id, mCDI2.bnTriggerBfa1.tc_count, mCDI2.bnTriggerBfa1.tc_trigger, (uint32_t)mCDI2.tsTrigger);
            } else
                mCDI2.bnTriggerBfa2 = msg.bnTrigger.bntrigger;

            break;

        case CDI2_MSG_BPPRESENT: {
            /* get running system clock [ticks] */
            uint64_t ts = 0;
            XTime_GetTime(&ts);

            if (mCDI2.bnTriggerBfa1.id != BNINVALID) {
                /* notify application about BN trigger */
                cdi2IfBanknoteTrigger(0, mCDI2.bnTriggerBfa1.id);

                mCDI2.bnTriggerBfa1.id  = BNINVALID;

                /* update offset, latency averages */
                ++mCDI2.numTrigger;
                const long triggerOffset  = msg.bpPresent.ts - mCDI2.tsTrigger;
                mCDI2.meanTriggerOffset   = mCDI2.meanTriggerOffset + (triggerOffset - mCDI2.meanTriggerOffset) / mCDI2.numTrigger;

                const long triggerLatency = ts - msg.bpPresent.ts;
                mCDI2.meanTriggerLatency  = mCDI2.meanTriggerLatency + (triggerLatency - mCDI2.meanTriggerLatency) / mCDI2.numTrigger;

                if (mCDI2.bnTriggerBfa2.id != BNINVALID) {
                    /* schedule trigger for BFA#2 (always time based) */
                    (void)ttsScheduleBp(mCDI2.bnTriggerBfa2.tc_trigger);
                }
            } else if (mCDI2.bnTriggerBfa2.id != BNINVALID) {
                /* notify application about BN trigger in BFA#2 */
                cdi2IfBanknoteTrigger(1, mCDI2.bnTriggerBfa2.id);
                mCDI2.bnTriggerBfa2.id = BNINVALID;
            }

            break;
        }

        case CDI2_MSG_CDIFRAME:
            /* sanity */
            assert(msg.cdiFrame.p != NULL);
            assert(msg.cdiFrame.len >= 3);

            switch (msg.cdiFrame.p->data[0]) {
            case CCR_Nil:
                /* release CDI frame data */
                sbufPut(msg.cdiFrame.p);
                break;

            case CCR_DetectorReset:
                if (msg.cdiFrame.len == 4) {
                    extern void STOP(void);

                    /* stop PowerLink */
                    plkReqStop();

                    plkSetErrorThresholds(kRelaxedErrorThreshold);
                    plkResetBnResultQueues();
                    ttsSetDeviceReady(false);
                    cdi2ResetTtsState();
                    ttsResetTcMeas();
                    cdi2SetDeviceStatus(DS_START_UP);
                    mCDI2.isTtsDevice = false;
                    cdi2ResetBnTriggerState();

                    /* Microblaze reset via Watchdog */
                    wdReset();
                    STOP();
                } else
                    logMsg(ERROR, ERR_APPLICATION, "CDI2: cannot parse Software Detector Reset request");

                /* release CDI frame data */
                sbufPut(msg.cdiFrame.p);
                break;

            case CCR_ProtocolVersion:
                if (msg.cdiFrame.len == 5) {
                    /* only version 2.0 supported */
                    if ((msg.cdiFrame.p->data[1] == 0) && (msg.cdiFrame.p->data[2] == 2))
                        mCDI2.cdi2CoreState = CCS_OK;
                    else
                        mCDI2.cdi2CoreState = CCS_WRONG_PROTOCOL;

                    cdi2IfProtocolVersionAnswer(cdi2GetCoreStatus(),
                                                plkGetNmtStatus(),
                                                plkGetLastBsmState(),
                                                cdi2GetDeviceStatus(),
                                                cdi2GetMaintenanceStatus(),
                                                cdi2GetErrorStatus());
                } else
                    logMsg(ERROR, ERR_APPLICATION, "CDI2: cannot parse Protocol Version request");

                /* release CDI frame data */
                sbufPut(msg.cdiFrame.p);
                break;

            case CCR_StartCdi2:
                if (msg.cdiFrame.len == 13) {
                    /* start CDI2 Core only after successful Protocol handshake */
                    if (cdi2GetCoreStatus() == CCS_OK) {
                        const unsigned int nodeId = msg.cdiFrame.p->data[1];
                        const bool use2ndBfa      = (msg.cdiFrame.p->data[2] != 0);
                        mCDI2.isTtsDevice         = (msg.cdiFrame.p->data[3] != 0);
                        const unsigned int level  = msg.cdiFrame.p->data[4];
                        const uint8_t *mac        = &msg.cdiFrame.p->data[5];

                        if ((nodeId > C_ADR_INVALID) && (nodeId <= kMaxDeviceCount)) {
                            setLevel(level);
                            plkReqStart(nodeId, use2ndBfa, mac);

                            if (mCDI2.isTtsDevice)
                                /* enable interrupt in TTS device */
                                (void)ttsSetBpInterruptEnable(true);
                            else
                                /* disable interrupt in TTS device */
                                (void)ttsSetBpInterruptEnable(false);
                        } else
                            logFormat(ERROR, ERR_APPLICATION, "CDI2: invalid node ID %u", nodeId);
                    } else
                        logMsg(ERROR, ERR_APPLICATION, "CDI2: ignoring Start CDI2 request in wrong Core state");
                } else
                    logMsg(ERROR, ERR_APPLICATION, "CDI2: cannot parse Start CDI2 request");

                /* release CDI frame data */
                sbufPut(msg.cdiFrame.p);
                break;

            case CCR_DetStatus:
                if (msg.cdiFrame.len == 4) {
                    switch (msg.cdiFrame.p->data[1]) {
                    case DS_START_UP:
                        cdi2SetDeviceStatus(DS_START_UP);

                        LOG_TRACE("# CDI2: [%lu] DS_STARTUP\r\n", (unsigned long)(target_getCurrentTimestamp() / 1000000000));

                        plkSetErrorThresholds(kRelaxedErrorThreshold);
                        plkResetBnResultQueues();
                        ttsSetDeviceReady(false);
                        cdi2ResetTtsState();
                        ttsResetTcMeas();
                        cdi2ResetBnTriggerState();
                        break;

                    case DS_INITIALISATION:
                        cdi2SetDeviceStatus(DS_INITIALISATION);

                        LOG_TRACE("# CDI2: [%lu] DS_INITIALISATION\r\n", (unsigned long)(target_getCurrentTimestamp() / 1000000000));

                        plkSetErrorThresholds(kRelaxedErrorThreshold);
                        plkResetBnResultQueues();
                        ttsSetDeviceReady(true);
                        cdi2ResetTtsState();
                        ttsResetTcMeas();
                        cdi2ResetBnTriggerState();
                        break;

                    case DS_INITIALISED:
                        cdi2SetDeviceStatus(DS_INITIALISED);

                        LOG_TRACE("# CDI2: [%lu] DS_INITIALISED\r\n", (unsigned long)(target_getCurrentTimestamp() / 1000000000));

                        plkSetErrorThresholds(kRelaxedErrorThreshold);
                        plkResetBnResultQueues();
                        ttsSetDeviceReady(true);
                        cdi2ResetTtsState();
                        ttsResetTcMeas();
                        cdi2ResetBnTriggerState();
                        break;

                    case DS_FEED_OFF:
                        cdi2SetDeviceStatus(DS_FEED_OFF);

                        LOG_TRACE("# CDI2: [%lu] DS_FEED_OFF\r\n", (unsigned long)(target_getCurrentTimestamp() / 1000000000));

                        if (mCDI2.numTrigger > 0)
                            logFormat(DEBUG, 0, "CDI2: trig num %lu, off %ld [clk], lat %ld [clk]", mCDI2.numTrigger, mCDI2.meanTriggerOffset, mCDI2.meanTriggerLatency);

                        plkSetErrorThresholds(kStandardErrorThreshold);
                        plkResetBnResultQueues();
                        cdi2ResetTtsState();
                        ttsResetTcMeas();
                        cdi2ResetBnTriggerState();
                        ttsSetDeviceReady(true);

                        statsLog();
                        break;

                    case DS_READY_TO_SORT:
                        cdi2SetDeviceStatus(DS_READY_TO_SORT);

                        LOG_TRACE("# CDI2: [%lu] DS_READY_TO_SORT\r\n", (unsigned long)(target_getCurrentTimestamp() / 1000000000));

                        ttsSetDeviceReady(true);
                        plkResetBnResultQueues();
                        break;

                    case DS_SORTING:
                        cdi2SetDeviceStatus(DS_SORTING);

                        LOG_TRACE("# CDI2: [%lu] DS_SORTING\r\n", (unsigned long)(target_getCurrentTimestamp() / 1000000000));

                        ttsSetDeviceReady(true);
                        break;

                    case DS_READY_TO_SHUT_DOWN:
                        cdi2SetDeviceStatus(DS_READY_TO_SHUT_DOWN);

                        LOG_TRACE("# CDI2: [%lu] DS_READY_TO_SHUT_DOWN\r\n", (unsigned long)(target_getCurrentTimestamp() / 1000000000));

                        plkSetErrorThresholds(kRelaxedErrorThreshold);
                        plkResetBnResultQueues();
                        ttsSetDeviceReady(true);
                        cdi2ResetTtsState();
                        ttsResetTcMeas();
                        cdi2ResetBnTriggerState();

                        statsLog();
                        break;

                    case DS_SHUTDOWN:
                        cdi2SetDeviceStatus(DS_SHUTDOWN);

                        LOG_TRACE("# CDI2: [%lu] DS_SHUTDOWN\r\n", (unsigned long)(target_getCurrentTimestamp() / 1000000000));

                        plkSetErrorThresholds(kRelaxedErrorThreshold);
                        plkResetBnResultQueues();
                        ttsSetDeviceReady(false);
                        cdi2ResetTtsState();
                        ttsResetTcMeas();
                        cdi2ResetBnTriggerState();

                        statsLog();
                        break;

                    case DS_ERROR:
                        cdi2SetDeviceStatus(DS_ERROR);

                        LOG_TRACE("# CDI2: [%lu] DS_ERROR\r\n", (unsigned long)(target_getCurrentTimestamp() / 1000000000));

                        if (mCDI2.numTrigger > 0)
                            logFormat(DEBUG, 0, "CDI2: trig num %lu, off %ld [clk], lat %ld [clk]", mCDI2.numTrigger, mCDI2.meanTriggerOffset, mCDI2.meanTriggerLatency);

                        plkSetErrorThresholds(kRelaxedErrorThreshold);
                        plkResetBnResultQueues();
                        ttsSetDeviceReady(false);
                        cdi2ResetTtsState();
                        ttsResetTcMeas();
                        cdi2ResetBnTriggerState();

                        statsLog();
                        break;

                    case DS_INVALID:
                    default:
                        /* ignore unknown status */
                        break;
                    }
                } else
                    logMsg(ERROR, ERR_APPLICATION, "CDI2: cannot parse DET Status request");

                /* release CDI frame data */
                sbufPut(msg.cdiFrame.p);
                break;

            case CCR_SetMaintenanceState:
                if (msg.cdiFrame.len == 4) {
                    cdi2SetMaintenanceStatus(msg.cdiFrame.p->data[1]);
                } else
                    logMsg(ERROR, ERR_APPLICATION, "CDI2: cannot parse Set MTC status request");

                /* release CDI frame data */
                sbufPut(msg.cdiFrame.p);
                break;

            case CCR_SetErrorState:
                if (msg.cdiFrame.len == 4) {
                    cdi2SetErrorStatus(msg.cdiFrame.p->data[1]);
                } else
                    logMsg(ERROR, ERR_APPLICATION, "CDI2: cannot parse Set ERR status request");

                /* release CDI frame data */
                sbufPut(msg.cdiFrame.p);
                break;

            case CCR_BnRecognition:
                if (msg.cdiFrame.len == 10) {
                    struct BnInfo *pBnRecognition = (struct BnInfo *)(&msg.cdiFrame.p->data[1]);

                    if (pBnRecognition->id != BNINVALID)
                        plkBnRecognition(pBnRecognition);
                } else
                    logMsg(ERROR, ERR_APPLICATION, "CDI2: cannot parse Banknote Recognition request");

                /* release CDI frame data */
                sbufPut(msg.cdiFrame.p);
                break;

            case CCR_BnResult:
                if (msg.cdiFrame.len >= 13) {
                    /* put sbuf chain onto PowerLInk layer BNRESULT queue */
                    plkBnResult(msg.cdiFrame.p, msg.cdiFrame.len);
                } else {
                    logMsg(ERROR, ERR_APPLICATION, "CDI2: cannot parse Banknote Result request");

                    /* release CDI frame data */
                    sbufPut(msg.cdiFrame.p);
                }

                break;

            case CCR_AcceptUpdate:
                /* update local OBD UpdateSwStatus with acceptance */
                plkSetSwUpdateStatus(USS_UPDATE_REQUEST_ACCEPTED);

                break;

            case CCR_RejectUpdate:
                /* update local OBD UpdateSwStatus with acceptance */
                plkSetSwUpdateStatus(USS_UPDATE_REQUEST_REJECTED);

                break;

            default:
                logFormat(ERROR, ERR_APPLICATION, "CDI2: invalid CDI frame ID 0x%02x", msg.cdiFrame.p->data[0]);

                /* release CDI frame data */
                sbufPut(msg.cdiFrame.p);
                break;
            }

            break;

        default:
            logFormat(ERROR, ERR_APPLICATION, "CDI2: invalid message ID 0x%02x", msg.id);
            break;
        }
    }
}

int
cdi2Setup(void)
{
    /* reset layer status */
    mCDI2.cdi2CoreState        = CCS_NOT_READY;
    mCDI2.cdi2DeviceState      = DS_INVALID;
    mCDI2.isTtsDevice          = false;
    mCDI2.cdi2ErrorState       = 0;
    mCDI2.cdi2MaintenanceState = 0;

    /* reset TTS signal status */
    cdi2ResetTtsState();

    /* reset BN trigger status/statistics */
    cdi2ResetBnTriggerState();

    LOG_TRACE("# CDI2: setup CDI2 layer task, queue %u x |%lu|\r\n", CDI2_MESSAGE_QUEUE_LENGTH, sizeof(struct Cdi2Message));

    mCDI2.hQueueMessage = xQueueCreate(CDI2_MESSAGE_QUEUE_LENGTH,
                                       sizeof(struct Cdi2Message));

    assert(mCDI2.hQueueMessage != NULL);

    if (mCDI2.hQueueMessage == NULL)
        return XST_FAILURE;

    xTaskCreate(cdi2Task,
                "CDI2",
                CDI2_TASK_STACK_SIZE,
                NULL,
                CDI2_TASK_PRIORITY,
                &mCDI2.hTaskCdi2);

    assert(mCDI2.hTaskCdi2 != NULL);

    if (mCDI2.hTaskCdi2 == NULL)
        return XST_FAILURE;

    return XST_SUCCESS;
}
