/**
 * @file tts.h
 * @brief CDI2 TTS Device logic interface
 *
 * @author AIT
 * @copyright &copy;2023 Austrian Institute of Technology (AIT)
 */

#pragma once
#ifndef TTS_H
#define TTS_H

#ifdef __cplusplus
extern "C"
{
#endif /* __cplusplus */

#include <stdbool.h>
#include <stdint.h>

/**
 * enable/disable the BP interrupt from the TTS device interface
 *
 * @param[in] isEnable enable (true) or disable (false) BP interrupts
 * @return current state of TTS irq_enable register
 */
uint32_t ttsSetBpInterruptEnable(bool isEnable);

/**
 * clear pending TTS BP interrupt
 *
 * @return current state of TTS irq_pending register
 */
uint32_t ttsClearBpInterrupt(void);

/**
 * @return current state of TTS irq_pending register
 */
uint32_t ttsGetPendingInterrupts(void);

/**
 * raise a TTS BP interrupt
 *
 * @return current state of TTS irq_pending register
 */
uint32_t ttsTriggerBpInterrupt(void);

/**
 * schedule a time-triggered BP interrupt
 *
 * @param[in] tcBp transport clock count of the interrupt
 * @return system clock count value of interrupt
 */
uint64_t ttsScheduleBp(uint32_t tcBp);

/**
 * @return measured period [ns] of TTS clock at TTS interface
 */
uint32_t ttsGetTcPeriod(void);

/** TTS Deviations; CDI2 Specification 2.7c §6.8.4 */
enum TcCheck {
    TTSTC_OK      = 0,      /**< TTS:TC OK */
    TTSTC_NOCLOCK = 1,      /**< TTS:TC no clock */
    TTSTC_RANGE   = 2       /**< TTS:TC clock range */
};

/**
 * Check TTS transport clock for presence and range
 *
 * This function can be called from task or interrupt context
 *
 * @return one of TTSTC_OK, TTSTC_NOCLOCK, or TTSTC_RANGE
 */
enum TcCheck ttsCheckTc(void);

/**
 * update TC clock measurements
 *
 * This function is called from PowerLink SYNC handler every 1ms in interrupt
 * context. It updates the running TC clock period measurement and computes
 * the clock offset relative to the system sleep timer clock
 *
 * @warning The system timestamp (time counter 0 of the AXI sleep timer)
 * is just a 32 bit counter, zero-padded to 64 bit by Xtime_Gettime().
 *
 * @param[in] tsSoc SoC time stamp [ticks]
 * @param[in] tc current TC count from BSMINFO
 *
 * @see powerlink.c:processSyncEvent()
 */
void ttsUpdateTcMeas(uint64_t tsSoc, uint32_t tc);

/** reset TC clock measurements */
void ttsResetTcMeas(void);

/**
 * convert a BSMINFO:TC clock value to a system/sleep timer clock count [ticks]
 *
 * @param[in] tc TC clock count [TC ticks]
 * @return system clock [system clock ticks]
 */
uint64_t ttsTc2SysTick(uint32_t tc);

/**
 * convert a system clock count [ticks] to a BSMINFO:TC clock value
 *
 * @param[in] sysClk system/sleep timer count [ticks]
 * @return TC clock ticks
 */
uint32_t ttsSysTick2Tc(uint64_t sysClk);

/**
 * sample the current value of the TTS RESET line.
 *
 * This function can be called from task or interrupt context
 *
 * @return sampled value, 0 or 1 (positive logic), of TTS RESET
 */
unsigned int ttsSampleReset(void);

/**
 * set/reset the TTS READY signal
 *
 * @param[in] isDeviceReady diesired status of the TTS READY signal
 */
void ttsSetDeviceReady(bool isDeviceReady);

/**
 * @brief setup TTS device interface
 *
 * @return XST_SUCCESS or error code
 */
int ttsSetup(void);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* TTS_H */
