/**
 * @file statstics.h
 * @brief CDI2 Device statistics and counters
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#pragma once
#ifndef STATISTICS_H
#define STATISTICS_H

#ifdef __cplusplus
extern "C"
{
#endif /* __cplusplus */

/** assorted CDI2 device statistics/error counters */
struct Statistics {
    unsigned long cdi2IfRaDropped;          /**< CDI2IF RA queue dropped packets */
    unsigned long cdi2IfTxDropped;          /**< CDI2IF TX queue dropped packets */
    unsigned long pppIfRaDropped;           /**< PPPIF RA queue dropped packets */
    unsigned long pppIfTxDropped;           /**< PPPIF TX queue dropped packets */
    unsigned long plkTickDropped;           /**< dropped PLK stack tick */
    unsigned long plkBnRecognitionDropped;  /**< dropped BNRECOGNITION */
    unsigned long plkBnResultDropped;       /**< dropped BNRESULT */
    unsigned long plkSeqNoError;            /**< wrong BSMINFO sequence number */
    unsigned long cdi2NmtStatusDropped;     /**< dropped NMT Status messages */
    unsigned long cdi2BsmStatusDropped;     /**< dropped BSM Status messages */
    unsigned long cdi2BpDropped;            /**< dropped BP messages */
    unsigned long cdi2BnTriggerDropped;     /**< dropped BSMINFO:Trigger messages */
    unsigned long cdi2BnInfoDropped;        /**< dropped BSMINFO:BNINFO messages */
    unsigned long cdi2QueueDropped;         /**< dropped CDI2 requests because queue full */
};

/** declaration of error/statistics counters */
extern struct Statistics cdi2Statistics;

/** reset CDI2 error statistic counters */
void statsReset(void);

/** bump CDI2IF RA dropped counter */
inline void statsIncrCdi2IfRa(void) { ++cdi2Statistics.cdi2IfRaDropped; }

/** bump CDI2IF TX dropped counter */
inline void statsIncrCdi2IfTx(void) { ++cdi2Statistics.cdi2IfTxDropped; }

/** bump PPPIF RA dropped counter */
inline void statsIncrPppIfRa(void) { ++cdi2Statistics.pppIfRaDropped; }

/** bump PPPIF TX dropped counter */
inline void statsIncrPppIfTx(void) { ++cdi2Statistics.pppIfTxDropped; }

/** bump PLK stack tick dropped counter */
inline void statsIncrPlkTick(void) { ++cdi2Statistics.plkTickDropped; }

/** bump BNRECOGNITION dropped counter */
inline void statsIncrBnRecognition(void) { ++cdi2Statistics.plkBnRecognitionDropped; }

/** bump BNRESULT dropped counter */
inline void statsIncrBnResult(void) { ++cdi2Statistics.plkBnResultDropped; }

/** bump BSMINFO sequence number error */
inline void statsIncrSeqNo(void) { ++cdi2Statistics.plkSeqNoError; }

/** bump NMT Status dropped counter */
inline void statsIncrNmtStatus(void) { ++cdi2Statistics.cdi2NmtStatusDropped; }

/** bump BSM Status dropped counter */
inline void statsIncrBsmStatus(void) { ++cdi2Statistics.cdi2BsmStatusDropped; }

/** bump BP dropped counter */
inline void statsIncrBp(void) { ++cdi2Statistics.cdi2BpDropped; }

/** bump BSMINFO:Trigger dropped counter */
inline void statsIncrBnTrigger(void) { ++cdi2Statistics.cdi2BnTriggerDropped; }

/** bump BSMINFO:BNINFO dropped counter */
inline void statsIncrBnInfo(void) { ++cdi2Statistics.cdi2BnInfoDropped; }

/** bump CDI2 queue dropped counter */
inline void statsIncrCdi2Queue(void) { ++cdi2Statistics.cdi2QueueDropped; }

/** log collected statistics as series of INFO messages */
void statsLog(void);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* STATISTICS_H */
