/**
 * @file sbuf.h
 * @brief Serial Interface Buffer management
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#pragma once
#ifndef SBUF_H
#define SBUF_H

#ifdef __cplusplus
extern "C"
{
#endif  /* __cplusplus */

#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>

/** number of serial buffers in the system */
#define SBUF_RING_SIZE  128

/** total size of serial buffer, including sbuf chain pointer */
#define sbufTotalSize   64

/** data size of serial buffer */
#define sbufDataSize    (sbufTotalSize - sizeof(struct sbuf *))

/** Serial buffer */
struct sbuf {
    struct sbuf *pNext;                 /**< pointer to chained buffer or NULL */
    uint8_t      data[sbufDataSize];    /**< sbuf data field */
};

/**
 * get a serial buffer from the ring
 *
 * @param[in] isBlocking block calling task until an element is available
 * @return pointer to sbuf or NULL
 */
struct sbuf *sbufGet(bool isBlocking);

/**
 * return a serial buffer chain to the ring
 *
 * @param[in] pHead head of sbuf chain. For convenience, passing NULL as head is not an error
 */
void sbufPut(struct sbuf *pHead);

/**
 * get a serial buffer from the ring in interrupt context
 *
 * @return pointer to sbuf or NULL
 */
struct sbuf *sbufGetIsr(void);

/**
 * return a serial buffer chain to the ring in interrupt context
 *
 * @param[in] pHead head of sbuf chain. For convenience, passing NULL as head is not an error
 */
void sbufPutIsr(struct sbuf *pHead);

/** @return number of currently available sbufs */
size_t sbufAvailable(void);

/**
 * setup the sbuf ring
 *
 * @return XST_SUCCESS if successful, otherwise any XST_* error code
 */
int sbufSetupRing(void);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* SBUF_H */
