/**
 * @file powerlink.h
 * @brief CDI2 Device PowerLink layer; CN/Device variant
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#pragma once
#ifndef POWERLINK_H
#define POWERLINK_H

#ifdef __cplusplus
extern "C"
{
#endif  /* __cplusplus */

#include <stddef.h>
#include <stdint.h>

#include "sbuf.h"
#include "cdi2.h"
#include "gw.h"

/**
 * PowerLink layer request codes
 */
enum PLK_Requests {
    PLK_REQ_nil    = 0,                         /**< null/nil/void/zero/empty request, ignored */
    PLK_REQ_start  = 1,                         /**< start PowerLink */
    PLK_REQ_tick   = 2,                         /**< PowerLink stack tick (1ms) */
    PLK_REQ_stop   = 3,                         /**< stop PowerLink */
    PLK_REQ_fwdIp  = 4,                         /**< forward non-PLK frame */

    PLK_REQ_LAST   = PLK_REQ_fwdIp              /**< canonical last element for sizing and loop boundaries */
};

/** PowerLink layer request message */
struct PlkRequest {
    enum PLK_Requests id;                   /**< request code */

    /** request data */
    union {
        /** PLK_REQ_nil */
        struct {
            int empty[0];
        } nil;

        /** PLK_REQ_start */
        struct {
            unsigned int nodeId;            /**< CN node ID */
            unsigned int use2ndBfa;         /**< also trigger BFA#2 (BNINFO slot#16) */
            uint8_t      mac[ETH_ALEN];     /**< CN MAC address */
        } start;

        /** PLK_REQ_tick */
        struct {
            int emtpy[0];
        } tick;

        /** PLK_REQ_stop */
        struct {
            int emtpy[0];
        } stop;

        /** PLK_REQ_fwdIp */
        struct {
            struct sbuf *p;                 /**< address of head of sbuf chain */
            size_t       len;               /**< data length over complete chain */
        } fwdIp;
    };
};

/** send nil request to PowerLink layer */
void plkReqNil(void);

/**
 * send start request to PowerLink layer
 *
 * @param[in] nodeID CN node ID
 * @param[in] use2ndBfa trigger BFA#2 using BSMINFO slot 16
 * @param[in] mac CN MAC address.
 */
void plkReqStart(const unsigned int nodeId, const unsigned int use2ndBfa, const uint8_t mac[ETH_ALEN]);

/** send stack tick to PowerLink layer */
void plkReqTickIsr(void);

/** send stop request to PowerLink layer */
void plkReqStop(void);

/**
 * send a non-PLK frame in the next ASYNC slot
 *
 * This call transfers ownership of the sbuf chain to the
 * PLK task
 *
 * @param[in] pData head of sbuf chain holding MAC frame
 * @param[in] len total length of result data
 */
void plkReqFwdIp(struct sbuf *pData, size_t len);

/**
 * queue new BN result data for transmission as BNRESULT over
 * the PowerLink interface.
 *
 * This call transfers ownership of the sbuf chain to the
 * PLK task
 *
 * @param[in] pData head of sbuf chain holding result data
 * @param[in] len total length of result data
 */
void plkBnResult(struct sbuf *pData, size_t len);

/**
 * queue a new BN recognition data for transmission as BNRESULT over
 * the PowerLink interface
 *
 * param[in] pBnRecognition BN recognition data
 */
void plkBnRecognition(struct BnInfo *pBnRecognition);

/** reset/clear the BNRESULT data queues */
void plkResetBnResultQueues(void);

/** @return true, if OpenPowerLInk stack ahs been started by the application */
bool plkIsStarted(void);

/** @return last known BSM status */
enum BsmStateEnum plkGetLastBsmState(void);

/** @return current NMT status */
uint8_t plkGetNmtStatus(void);

/**
 * set/update error thresholds
 *
 * This functions updates the Loss-of-SoC (1C0Bh: DLL_CNLossSoC_REC) and
 * Loss-of-PReq (Object 1C0Dh: DLL_CNLossPReq_REC) thresholds
 *
 * @param[in] t new error threshold
 */
void plkSetErrorThresholds(uint32_t t);

/**
 * set SW update status
 *
 * This function sets UpdateSwStatus_U8 (2003h:2) in the local OBD
 */
void plkSetSwUpdateStatus(const enum UpdateSwStatus swuStatus);

/**
 * setup the PowerLink layer
 *
 * @return XST_SUCCESS or error code
 */
int plkSetup(void);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* POWERLINK_H */
