/**
 * @file object_dictionary.h
 * @brief CDI2 Device PowerLink layer; object dictionary
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#pragma once
#ifndef OBJECT_DICTIONARY_H
#define OBJECT_DICTIONARY_H

#ifdef __cplusplus
extern "C"
{
#endif  /* __cplusplus */

#include <stddef.h>
#include <stdint.h>

#include <oplk/oplk.h>

/**
 * @brief initialize the object dictionary
 *
 * This function initializes the static object dictionary and sets a few
 * pointers in the tObdInitParam structure given as a parameter.
 *
 * @param[inout] pParams pointer to OpenPowerLink initialization parameters for the OBD
 * @return kErrorOk if everything went well, some other kError* constant otherwise
 */
tOplkError obdCreate(tObdInitParam *pParams);

/**
 * @brief utility function adding a BLOB to the object dictionary
 *
 * This helper function adds a BLOB of arbitrary size to the object dictionary.
 * This is useful to allocate a large data buffer for a PDO, without having
 * to declare every field separately.
 *
 * @param[in] index OBD index
 * @param[in] subIx OBD sub-index where the data should be written
 * @param[in] size  size of the BLOB [bytes]
 * @return kErrorOk if everything went well, some other kError* constant otherwise
 */
tOplkError
obdAddByteArray(unsigned int index,
                unsigned int subIx,
                size_t       size);

/**
 * @brief writes to the object dictionary
 *
 * This method updates an entry in the object dictionary.
 *
 * @param[in] index  Object dictionary index
 * @param[in] subIx  sub-index where the data should be written
 * @param[in] pData  pointer to the data
 * @param[in] size   size of the data
 * @return kErrorOk if everything went well, some other kError* constant otherwise
 */
tOplkError
obdWriteEntry(unsigned int  index,
              unsigned int  subIx,
              const void   *pData,
              size_t        size);

/**
 * @brief read from the object dictionary
 *
 * This method returns an entry in the object dictionary.
 *
 * @param[in] index  Object dictionary index
 * @param[in] subIx  sub-index where the data should be written
 * @param[in] pData  pointer to the data
 * @param[in] size   size of the data
 * @return kErrorOk if everything went well, some other kError* constant otherwise
 */
tOplkError
obdReadEntry(unsigned int  index,
             unsigned int  subIx,
             void         *pData,
             size_t        size);

/**
 * @brief map a receive PDO
 *
 * This updates PDO_RxMappParam_XXh_AU6 (EPSG DS 301V1.5.0, §6.4.9.2) with
 * a single RPDO mapping. The RPDO is assumed to be a variable information
 * structure residing in the application, sub index 0 containing the number
 * of entries (always 1) and sub index 1 the actual data. If more than one
 * RPDO slot/channel is mapped, PDO_RxMappParam_XXh_AU6 and RPDO indexing
 * are assumed to be compatible:
 *
 * 0x1600 + @p slot will hold the mapping of PDO @p index + @p slot
 *
 * @param[in] index OBD index of PDO object
 * @param[in] size  size of the OBD object in bytes
 * @param[in] slot  receive slot/RPDO channel
 * @return kErrorOk if everything went well, some other kError* constant otherwise
 */
tOplkError
obdMapRxPdo(unsigned int index,
            size_t       size,
            unsigned int slot);

/**
 * @brief map a transmit PDO
 *
 * This updates PDO_TxMappParam_XXh_AU6 (EPSG DS 301V1.5.0, §6.4.9.4) with
 * a single TPDO mapping. The TPDO is assumed to be a variable information
 * structure residing in the application, sub index 0 containing the number
 * of entries (always 1) and sub index 1 the actual data. If more than one
 * TPDO slot/channel is mapped, PDO_TxMappParam_XXh_AU6 and TPDO indexing
 * are assumed to be compatible:
 *
 * 0x1A00 + @p slot will hold the mapping of PDO @p index + @p slot
 *
 * @param[in] index OBD index of PDO object
 * @param[in] size  size of the OBD object in bytes
 * @param[in] slot  transmit slot/TPDO channel
 * @return kErrorOk if everything went well, some other kError* constant otherwise
 */
tOplkError
obdMapTxPdo(unsigned int index,
            size_t       size,
            unsigned int slot);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* OBJECT_DICTIONARY_H */
