/**
 * @file logging.h
 * @brief CDI2 Device logging
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#pragma once
#ifndef LOGGING_H
#define LOGGING_H

#ifdef __cplusplus
extern "C"
{
#endif /* __cplusplus */

#include <stdbool.h>
#include <stdio.h>

#include <xil_printf.h>

#include "cdi2If.h"
#include "sbuf.h"

/* if compiled with assertions enabled, write debug output to JTAG UART */
#ifndef NDEBUG
#define LOG_TRACE(...)      xil_printf(__VA_ARGS__)
#else
#define LOG_TRACE(...)
#endif

/* messages written with LOG_PRINT always go to the JTAG UART */
#define LOG_PRINT(...)      xil_printf(__VA_ARGS__)

/**
 * write an ASCIZ string asynchronously to JTAG UART
 *
 * @param[in] msg message string to dump
 */
void logAsync(const char *msg);

/** make logLevel threshold header-visible to allow inlining of isEnabledFor() */
extern unsigned int logLevel;

/**
 * Sets the threshold for this logger to level.
 *
 * Logging messages which are less severe than level will be ignored;
 * logging messages which have severity level or higher will be emitted
 *
 * @param[in] level new system log level
 */
void setLevel(unsigned int level);

/**
 * Indicates if a message of severity @p level would be processed by this logger.
 *
 * @param[in] level message severity
 * @return true, if message would be processed by this logger
 */
inline bool isEnabledFor(unsigned int level) { return logLevel >= level; }

/**
 * log a fixed message with severity level.
 *
 * @param[in] level must be one of FATAL, ERROR, WARNING, INFO, DEBUG, or TRACE
 * @param[in] errStatus @see ErrorState
 * @param[in] msg ASCIZZ string
 */
#define logMsg(level, errStatus, msg)                   \
    do {                                                \
        if (isEnabledFor(LOG_##level))                  \
            cdi2If##level(errStatus, msg);              \
    } while (0)

/**
 * log a formatted message with severity level.
 *
 * @param[in] level must be one of FATAL, ERROR, WARN, INFO, DEBUG, or TRACE
 * @param[in] errStatus @see ErrorState
 * @param[in] ... variardic arguments passed to snprintf() to format a message
 */
#define logFormat(level, errStatus, ...)                \
    do {                                                \
        if (isEnabledFor(LOG_##level)) {                \
            char buf[sbufDataSize - 9];                 \
            snprintf(buf, sizeof(buf), __VA_ARGS__);    \
            cdi2If##level(errStatus, buf);              \
        }                                               \
    } while (0)

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* LOGGING_H */
