/**
 * @file gw.h
 * @brief PPP/PLK gateway
 *
 * @author AIT
 * @copyright &copy;2023 Austrian Institute of Technology (AIT)
 */

#pragma once
#ifndef GW_H
#define GW_H

#ifdef __cplusplus
extern "C"
{
#endif  /* __cplusplus */

#include <stddef.h>
#include <stdint.h>

#include <oplk/oplk.h>

#include "sbuf.h"

#define ETH_ALEN 6              /**< Octets in one ethernet addr */

/**
 * forward a PPP RFC 1332 IP frame over PowerLink
 *
 * This call does *not* transfer ownership of the sbuf chain
 *
 * @param[in] pHead sbuf chain holding PPP information field data
 * @param[in] totLen size of PPP information field
 */
void gwForwardIP(struct sbuf *pHead, size_t totLen);

/**
 * process a MAC frame received in the ASYNC phase of OpenPowerLink
 *
 * @param[in] pFrame non-PLK frame
 * @param[in] frameSize size of frame data
 */
void gwProcessMac(tPlkFrame *pFrame, size_t frameSize);

/**
 * update ARP cache after after finally deciding local(Mac|Ip)Address
 *
 * @param[in] ipAddr local IP address
 * @param[in] macAddr local MAC address
 */
void gwUpdateLocalAddress(const uint32_t ipAddr, const uint8_t macAddr[ETH_ALEN]);

/**
 * update default gateway
 *
 * @param[in] ipGw n ew default gateway IP address
 */
void gwUpdateDefaultGateway(const uint32_t ipAddr);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* GW_H */
