/**
 * @file cdi2If.h
 * @brief CDI2 UART interface
 *
 * @author AIT
 * @copyright &copy;2023 Austrian Institute of Technology (AIT)
 */

#pragma once
#ifndef CDI2_IF_H
#define CDI2_IF_H

#ifdef __cplusplus
extern "C"
{
#endif  /* __cplusplus */

#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>

#include "cdi2.h"

/**
 * send a Protocol Version Answer response
 *
 * @param[in] coreStatus CDI2 IP core status
 * @param[in] nmtStatus current PowerLInk NMT status
 * @param[in] bsmStatus last known BSM status
 * @param[in] devStatus current CDI2 device status
 * @param[in] mtcStatus CDI2 maintenance status bits
 * @param[in] errStatus CDI2 error status bits
 */
void cdi2IfProtocolVersionAnswer(enum Cdi2CoreStatus  coreStatus,
                                 uint8_t              nmtStatus,
                                 enum BsmStateEnum    bsmStatus,
                                 enum DeviceStateEnum devStatus,
                                 uint8_t              mtcStatus,
                                 uint8_t              errStatus);

/**
 * send a TTS Reset indication
 */
void cdi2IfTtsReset(void);

/**
 * send an NMT Reset indication
 *
 * @param[in] cause sepcific NMT reset cause or reason
 */
void cdi2IfNmtReset(enum Cdi2CoreResetCause cause);

/**
 * send a Start CDI2 Answer response
 */
void cdi2IfStartCdi2Answer(void);

/**
 * send a BSM status change indication
 *
 * @param[in] bsmStatus new BSM status
 */
void cdi2IfBsmStatus(enum BsmStateEnum bsmStatus);

/**
 * send a Banknote ID announcement
 *
 * @param[in] bnid banknode ID
 * @param[in] tcCount current device transport clock counter
 * @param[in] tcTrigger expected trigger instant [TC clock ticks]
 * @param[in] t_trig expected arrival time [ns]
 */
void cdi2IfBanknoteId(uint32_t bnid,
                      uint32_t tcCount,
                      uint32_t tcTrigger,
                      uint32_t t_trig);

/**
 * send a Banknote Trigger indication
 *
 * @param[in] isBfa1 true, if this trigger message is for BFA#1
 * @param[in] bnid banknote ID
 */
void cdi2IfBanknoteTrigger(bool     isBfa1,
                           uint32_t bnid);

/**
 * send a Banknote Info message
 *
 * @param[in] bnid banknote ID
 * @param[in] series series; CDI2 specification 2.7c, Table 33
 * @param[in] denomination denomination; CDI2 Specification 2.7c, Table 33
 * @param[in] orientation orientation; CDI2 Specification 2.7c, Table 34
 */
void cdi2IfBanknoteInfo(uint32_t bnid,
                        uint8_t  series,
                        uint8_t  denomination,
                        uint8_t  orientation);

/**
 * send a Prepare Update indication
 */
void cdi2IfPrepareUpdate(void);

/**
 * send a Perform Update indication
 */
void cdi2IfPerformUpdate(void);

/**
 * send one of the FATAL...TRACE messages
 *
 * @param[in] idByte message id, [0x29, 0x2e]
 * @param[in] errStatus error classification, @see MaintenanceState
 * @param[in] sMsg zero-terminated error message
 */
void cdi2IfLog(uint8_t  idByte,
               uint8_t  errStatus,
               char    *sMsg);

/** send a FATAL log message */
#define cdi2IfFATAL(errStatus, msg) cdi2IfLog(CCR_Fatal, errStatus, msg)

/** send an ERROR log message */
#define cdi2IfERROR(errStatus, msg) cdi2IfLog(CCR_Error, errStatus, msg)

/** send a WARNING log message */
#define cdi2IfWARN(errStatus, msg) cdi2IfLog(CCR_Warning, errStatus, msg)

/** send an INFO log message */
#define cdi2IfINFO(errStatus, msg) cdi2IfLog(CCR_Info, errStatus, msg)

/** send a DEBUG log message */
#define cdi2IfDEBUG(errStatus, msg) cdi2IfLog(CCR_Debug, errStatus, msg)

/** send a TRACE log message */
#define cdi2IfTRACE(errStatus, msg) cdi2IfLog(CCR_Trace, errStatus, msg)

/**
 * The 16 character deep RX FIFO overruns after 128us. Unfortunately,
 * FreeRTOS/Microblaze does not support nested interrupts, so we poll the
 * UARTs at various points in the (expensive/long) PowerLink ISR
 */
void cdi2IfPollUartIsr(void);

/**
 * @brief setup the CDI2 UART interface
 *
 * @param[in] speed desired UART speed in [baud]
 *
 * @return XST_SUCCESS or error code
 */
int cdi2IfSetup(unsigned long speed);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* CDI2_IF_H */
