/**
 * @file cdi2.h
 * @brief CDI2 Device CDI2 layer; CN/Device variant
 *
 * @author AIT
 * @copyright &copy;2023 AIT Austrian Institute of Technology
 */

#pragma once
#ifndef CDI2_H
#define CDI2_H

#ifdef __cplusplus
extern "C"
{
#endif /* __cplusplus */

#include <stdint.h>

#include <FreeRTOS.h>
#include <queue.h>

#include "sbuf.h"

/** SW Version */
#if defined(__has_include)
  #if __has_include("cdi2_sw_version.h")
    #include "cdi2_sw_version.h"
  #else
    #define SW_VERSION "v1.0"
  #endif /* __has_include */
#else
  #define SW_VERSION "v1.0"
#endif /* defined */

#define BNINVALID ((uint32_t)-1)                /**< invalid Banknote ID; CDI2 Specification 2.7C §6.7.4.2 */

#define kDefaultGatewayId 254                   /**< default gateway address */
#define kDefaultMnId      240                   /**< standard managing node id. C_ADR_MN_DEF_NODE_ID, EPSG DS 301 V1.3.0 App. 3.8 */
#define kAsyncMTU         1500                  /**< MTU for non-plk frames over POWERLINK */
#define kWaitSocPreq      5000                  /**< time interval in ns between end of SoC transmission and begin of the next frame transmission (PReq or PResMN). EPSG DS 301 V1.3.0, Object 1F8Ah */

#define kDefaultIpAddr     0xC0A86401u          /**< 192.168.100.1 */
#define kDefaultGateway    0xC0A864FEu          /**< 192.168.100.254. C_ADR_RT1_DEF_NODE_ID, EPSG DS 301 V1.3.0 App. 3.8 */
#define kDefaultSubnetMask 0xFFFFFF00u          /**< 255.255.255.0 */

#define kCycleLen             1000              /**< CDI2 cycle duration in micro-seconds */
#define kMultiplxdCycleCount  4                 /**< CDI2 Multiplexed Cycle Count in multiples of POWERLINK cycles. EPSG DS 301 V1.3.0, 1F98h.7 */
#define kMultiplxdCycleLength 4                 /**< number of polled device nodes in a CDI2 multiplexed cycle */
#define kMaxDeviceCount       16                /**< CDI2 maximum number of device nodes */

#define kDefaultPowerlinkSendDelay  2           /**< the number of cycles the OpenPOWERLINK implementation needs until a packet is actually sent */

#define kTcMaxPeriod    100000                  /**< period [20ns] of minimum transport clock (0.5m/s / 1mm) */
#define kTcMinPeriod    385                     /**< period [20ns] of maximum transport clock (13m/s / 0.1mm) */

#define kTtsResetMinLength 10                   /**< minimum length of valid TTS RESET pulse [ms]; CDI2 Specification 2.7c §5.2 */

#define kStandardErrorThreshold  1              /**< standard error threshold for missing Soc or PReq */
#define kRelaxedErrorThreshold  23              /**< relaxed error thresholds for missing Soc or PReq */

/** CDI2 IP Core Status */
enum Cdi2CoreStatus {
    CCS_OK               = 0,
    CCS_NOT_READY        = 1,
    CCS_WRONG_PROTOCOL   = 2,
    CCS_IP_CORE_STARTUP  = 3,
    CCS_HW_ERROR         = 6,
    CCS_SELF_TEST_FAILED = 7,
    CCS_LAST_STATE       = CCS_SELF_TEST_FAILED
};

/** CDI2 IP Core reset reason/cause/code */
enum Cdi2CoreResetCause {
    CCRC_Application            = 0x00,         /**< application reset */
    CCRC_TTS                    = 0x01,         /**< TTS reset */
    CCRC_NMT_SwReset            = 0x10,         /**< NMT SW reset */
    CCRC_NMT_ResetNode          = 0x11,         /**< NMT node reset */
    CCRC_NMT_ResetCommunication = 0x12,         /**< NMT communication reset */
    CCRC_NMT_ResetConfiguration = 0x13          /**< NMT configuration reset */
};

/** CDI2 IP Core log levels */
enum Cdi2CoreLogLevel {
    LOG_FATAL = 10,
    LOG_ERROR = 20,
    LOG_WARN  = 30,
    LOG_INFO  = 40,
    LOG_DEBUG = 50,
    LOG_TRACE = 60
};

/** CDI2 Maintenance State bits; CDI2 Specification 2.7c, §6.7.4.4 */
enum MaintenanceState {
    MAI_NO_MAINTENANCE_NEEDED = 0x00,           /**< no maintenance needed */
    MAI_AIR_PULSE             = 0x01,           /**< air pulse required */
    MAI_MANUAL_CLEANING       = 0x02,           /**< manual cleaning required */
    MAI_SELF_TEST             = 0x04,           /**< self-test required */
    MAI_CALIBRATION           = 0x08            /**< calibration required */
};

/** CDI2 Error State bits; CDI2 Specification 2.7c, §6.7.4.4 */
enum ErrorState  {
    ERR_NO_ERROR    = 0x00,                     /**< no error */
    ERR_POWERLINK   = 0x01,                     /**< PowerLink error */
    ERR_TTS         = 0x02,                     /**< TTS error */
    ERR_IDB         = 0x04,                     /**< IDB error */
    ERR_APPLICATION = 0x08                      /**< application error */
};

/** CDI2 System State; CDI2 Specification 2.7c, Table 23 */
enum SystemStateEnum {
    SYS_INCONSISTENT        = 0,
    SYS_START_UP            = 1,
    SYS_INITIALISATION      = 2,
    SYS_FEED_OFF            = 3,
    SYS_READY_TO_SORT       = 4,
    SYS_SORTING             = 5,
    SYS_READY_TO_SHUT_DOWN  = 6,
    SYS_SHUTDOWN            = 7,
    SYS_ERROR               = 8,
    SYS_LAST_STATE          = SYS_ERROR
};

/** CDI2 Banknote Sorting Machine State; CDI2 Specification 2.7c, Table 22 */
enum BsmStateEnum {
    BS_INVALID              = 0,
    BS_START_UP             = 1,
    BS_INITIALISATION       = 2,
    BS_FEED_OFF             = 3,
    BS_REQUEST_TO_SORT      = 4,
    BS_SORTING              = 5,
    BS_REQUEST_TO_SHUT_DOWN = 6,
    BS_SHUTDOWN             = 7,
    BS_ERROR                = 10,
    BS_LAST_STATE           = BS_ERROR
};

/** CDI2 Device State; CDI2 Specification 2.7c, Table 22 */
enum DeviceStateEnum {
    DS_INVALID                = 0,
    DS_START_UP               = 1,
    DS_INITIALISATION         = 2,
    DS_INITIALISED            = 3,
    DS_FEED_OFF               = 4,
    DS_READY_TO_SORT          = 5,
    DS_SORTING                = 6,
    DS_READY_TO_SHUT_DOWN     = 7,
    DS_SHUTDOWN               = 8,
    DS_ERROR                  = 10,
    DS_INTENSIVE_SELF_TEST    = 11,             /**< transitional simulator state */
    DS_SHORT_SELF_TEST        = 12,             /**< transitional simulator state */
    DS_PREPARING_TO_SHUT_DOWN = 13,             /**< transitional simulator state */
    DS_RESET                  = 14,             /**< transitional simulator state */
    DS_READY_TO_RESTART       = 15,             /**< transitional simulator state */
    DS_LAST_STATE             = DS_READY_TO_RESTART
};

/** CDI2 Command Identifications */
enum Cdi2CommandIdEnum {
    CMDID_BNRESULT = 0x41,
    CMDID_BSMINFO  = 0x81,
    CMDID_PREQ     = 0x82
};

/**
 * BNINFO Structure Layout
 */
struct BnInfo {
    uint32_t id;                        /**< Banknote ID */
    uint8_t  series;                    /**< Series; CDI2 specification 2.7c, Table 33 */
    uint8_t  denomination;              /**< Denomination; CDI2 Specification 2.7c, Table 33 */
    uint8_t  orientation;               /**< Orientation; CDI2 Specification 2.7c, Table 34 */
    uint8_t  reserved;
} __attribute__((packed));

/**
 * BNTRIGGER Structure Layout
 */
struct BnTrigger {
    uint32_t tc_count;                  /**< TC (Transport Clock) Counter */
    uint32_t id;                        /**< Banknote ID */
    uint32_t tc_trigger;                /**< Trigger Instant */
} __attribute__((packed));

/**
 * Common CDI2 PowerLink PDU Header
 */
struct CmdHeader {
    uint8_t  command_id;                /**< CDI2 Command ID */
    uint32_t sequence_num;              /**< Sequence number */
} __attribute__((packed));

/**
 * BNRESULT Layout; CDI2 Specification 2.7c 6.7.4.4
 */
struct BnResult {
    struct CmdHeader header;            /**< common CDI2 PDU header */
    uint8_t          device_state;      /**< device state */
    uint8_t          error_state;       /**< error state */
    uint8_t          maintenance_state; /**< maintenance state */
    struct BnInfo    bn_recognition;    /**< BNRECOGNITION */
    struct BnInfo    bn_info;           /**< BNINFO */
    uint8_t          segment;           /**< BNRESULT segment # */
    uint8_t          reserved[3];
    uint8_t          data[1400];        /**< supplementary data block */
} __attribute__((packed));

/**
 * BSMINFO Layout; CDI2 Specification 2.7c 6.7.4.2
 */
struct BsmInfo {
    struct CmdHeader header;            /**< common CDI2 PDU header */
    uint8_t          state;             /**< BSM state */
    uint16_t         reserved;
    struct BnTrigger bn_triggers[16];   /**< BNTRGGER info for 16 CN */
    struct BnInfo    bn_info;           /**< BNINFO */
} __attribute__((packed));

/**
 * SDO UpdateSwCommand literals; CDI2 Specification 2.7c 6.7.6
 */
enum UpdateSwCommand {
  USC_IDLE           = 0,
  USC_PREPARE_UPDATE = 1,
  USC_PERFORM_UPDATE = 2
};

/**
 * SDO UpdateSwStatus literals; CDI2 Specification 2.7c 6.7.6
 */
enum UpdateSwStatus {
  USS_IDLE                    = 0,
  USS_UPDATE_REQUEST_ACCEPTED = 1,
  USS_UPDATE_REQUEST_REJECTED = 2
};

/**
 * CDI2 layer message codes
 */
enum CDI2_Messages {
    CDI2_MSG_nil             = 0,                   /**< null/nil/void/zero/empty message, ignored */
    CDI2_MSG_NmtStateChange  = 1,                   /**< NMT status change */
    CDI2_MSG_BsmStateChange  = 2,                   /**< BSM status change */
    CDI2_MSG_UpdateSwCommand = 3,                   /**< SDO UpdateSwCommand change */
    CDI2_MSG_TtsReset        = 4,                   /**< TTS Reset detected */
    CDI2_MSG_TtsNoClock      = 5,                   /**< TTS:TC no clock error */
    CDI2_MSG_TtsClockRange   = 6,                   /**< TTS:TC out of range */
    CDI2_MSG_PlkStarted      = 7,                   /**< PLK start achnowledge */
    CDI2_MSG_BNINFIO         = 8,                   /**< BNINFO */
    CDI2_MSG_BNTRIGGER       = 9,                   /**< BNTRIGGER announcement */
    CDI2_MSG_BPPRESENT       = 10,                  /**< BN present trigger */
    CDI2_MSG_CDIFRAME        = 11,                  /**< CDI frame from CDI2IF */

    CDI2_MSG_LAST            = CDI2_MSG_CDIFRAME    /**< canonical last element for sizing and loop boundaries */
};

/** CDI2 layer message */
struct Cdi2Message {
    enum CDI2_Messages id;                          /**< message code */

    /** message data */
    union {
        /** CDI2_MSG_nil */
        struct {
            int empty[0];
        } nil;

        /** CDI2_MSG_NmtStateChange */
        struct {
            uint16_t oldNmtState;                   /**< old NMT state */
            uint16_t newNmtState;                   /**< new NMT state */
        } nmtStateChange;

        /** CDI2_MSG_BsmStateChange */
        struct {
            uint8_t oldBsmState;                    /**< old NMT state */
            uint8_t newBsmState;                    /**< new NMT state */
        } bsmStateChange;

        /** CDI2_MSG_UpdateSwCommand */
        struct {
            uint8_t swuCmd;                         /**< UpdateSwCommand */
        } swUpdateCommand;

        /** CDI2_MSG_TtsReset */
        struct {
            int empty[0];
        } ttsReset;

        /** CDI2_MSG_TtsNoClock */
        struct {
            int empty[0];
        } ttsNoClock;

        /** CDI2_MSG_TtsClockRange */
        struct {
            int empty[0];
        } ttsClockRange;

        /** CDI2_MSG_PlkStarted */
        struct {
            int empty[0];
        } plkStarted;

        /** CDI2_MSG_BNINFO */
        struct BnInfo bninfo;

        /** CDI2_MSG_BNTRIGGER */
        struct {
            struct BnTrigger bntrigger;
            uint32_t         is2ndBfa;
        } bnTrigger;

        /** CDI2_MSG_BPPRESENT */
        struct {
            uint64_t ts;                            /**< time stamp [system ticks] */
        } bpPresent;

        /** CDI2_MSG_CDIFRAME */
        struct {
            struct sbuf *p;                         /**< address of head of sbuf chain */
            size_t       len;                       /**< data length over complete chain */
        } cdiFrame;
    };
};

/** CDI Command/Response ID codes */
enum CdiCmdRspId {
    CCR_Nil                   = 0,      /**< void/invalid/empty command/response */
    CCR_DetectorReset         = 0x10,   /**< Software Detector Reset */
    CCR_ProtocolVersion       = 0x11,   /**< Protocol Version */
    CCR_ProtocolVersionAnswer = 0x21,   /**< Protocol Version Answer */
    CCR_TtsReset              = 0x23,   /**< TTS Reset */
    CCR_NmtReset              = 0x24,   /**< NmtReset */
    CCR_StartCdi2             = 0x15,   /**< Start CDI2 */
    CCR_StartCdi2Answer       = 0x25,   /**< Start CDI2 */
    CCR_DetStatus             = 0x16,   /**< Detector Status */
    CCR_BsmStatus             = 0x26,   /**< BSM Status */
    CCR_SetMaintenanceState   = 0x17,   /**< Set Maintenance State */
    CCR_SetErrorState         = 0x18,   /**< Set Error State */
    CCR_Fatal                 = 0x29,   /**< Log Message FATAL */
    CCR_Error                 = 0x2a,   /**< Log Message ERROR */
    CCR_Warning               = 0x2b,   /**< Log Message WARNING */
    CCR_Info                  = 0x2c,   /**< Log Message INFO */
    CCR_Debug                 = 0x2d,   /**< Log Message DEBUG */
    CCR_Trace                 = 0x2e,   /**< Log Message TRACE */
    CCR_BnTrigger             = 0x80,   /**< Banknote Trigger */
    CCR_BnId                  = 0x81,   /**< Banknote ID */
    CCR_BnInfo                = 0x82,   /**< Banknote Info */
    CCR_BnResult              = 0x41,   /**< Banknote Result */
    CCR_BnRecognition         = 0x42,   /**< Banknote Recognition */
    CCR_AcceptUpdate          = 0x47,   /**< Accept SW Update */
    CCR_RejectUpdate          = 0x48,   /**< Reject SW Update */
    CCR_PrepareUpdate         = 0x87,   /**< Prepare SW Update */
    CCR_PerformUpdate         = 0x88    /**< Perform SW Update */
};

/** @return CDI2 layer message queue handle */
QueueHandle_t cdi2GetMessageQueue(void);

/** @return current CDI2 Core status */
enum Cdi2CoreStatus cdi2GetCoreStatus(void);

/** @return current CDI2 device status */
enum DeviceStateEnum cdi2GetDeviceStatus(void);

/**
 * set new CDI2 device status
 *
 * @param[in] ds new device status
 */
void cdi2SetDeviceStatus(const enum DeviceStateEnum ds);

/** @return current CDI2 error status */
uint8_t cdi2GetErrorStatus(void);

/**
 * set new CDI2 error status
 *
 * @param[in] es new error status
 */
void cdi2SetErrorStatus(uint8_t es);

/** @return current CDI2 maintenance status */
uint8_t cdi2GetMaintenanceStatus(void);

/**
 * set new CDI2 maintenance status
 *
 * @param[in] ms new maintenance status
 */
void cdi2SetMaintenanceStatus(uint8_t ms);

/**
 * periodically check presence and range of TTS transport clock
 *
 * This polling function is called by the FreeRTOS application
 * tick hook every 1ms in interrupt context
 */
void cdi2CheckTc(void);

/**
 * sample the TTS RESET line for a valid (>= 10ms) long reset pulse
 *
 * This polling function is called by the FreeRTOS application
 * tick hook every 1ms in interrupt context
 */
void cdi2CheckReset(void);

/**
 * @brief setup the CDI2 layer
 *
 * @return XST_SUCCESS or error code
 */
int cdi2Setup(void);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* CDI2_H */
