#!/usr/bin/python3

##
# @file setSerial.py
# @brief tool to set non-canonical speeds for serial devices
#
# @version 1.0
# @author AIT
# @copyright &copy;2023 AIT Austrian Institute of Technology

import array
import argparse
from enum import Enum, IntEnum
import fcntl
import os.path
import termios

## Linux ioctl codes to access struct termios2
class TERMIOS2(IntEnum):
    TCGETS2 = 0x802C542A
    TCSETS2 = 0x402C542B

def setSerialSpeed(fUART, speed):
    """
    The actual CDI2 UART baud rate is 1.25Mbaud. Requesting the canonical speed of
    termios.B1152000 sets the DLL and DLM (i.e., divisor registers) of the AXI
    UART to 5, which translates to the desired rate. However, this does not work
    with the FTDI USB/UART transceivers which need accurate speed settings.
    Thus, we must use the struct termios2 and TCGETS/TCSET2 ioctl() to configure
    the UART speed.

    see termios(3), ioctl_tty(2)

    Parameters
    ----------
    fUART : io.BufferedIOBase
        file object of UART interface
    speed : int
        desired baud rate
    """

    # sizeof(struct termios2) is 44, align conservatively
    #
    # struct termios2 {
    #     tcflag_t c_iflag;     /*  [0] input mode flags */
    #     tcflag_t c_oflag;     /*  [1] output mode flags */
    #     tcflag_t c_cflag;     /*  [2] control mode flags */
    #     tcflag_t c_lflag;     /*  [3] local mode flags */
    #     cc_t c_line;          /*  [4] line discipline */
    #     cc_t c_cc[NCCS];      /*      control characters */
    #     speed_t c_ispeed;     /*  [9] input speed */
    #     speed_t c_ospeed;     /* [10] output speed */
    # };
    termios2 = array.array('I', [0] * 64)

    # read complete struct termios2
    fcntl.ioctl(fUART, TERMIOS2.TCGETS2, termios2)

    # update with custom speed
    termios2[2]  &= ~termios.CBAUD
    termios2[2]  |= termios.CBAUDEX
    termios2[9]   = args.speed
    termios2[10]  = args.speed

    # set updated termios2
    fcntl.ioctl(fUART, TERMIOS2.TCSETS2, termios2)

def configureUART(fUART, args):
    """
    configure local UART settings: raw mode, no modem and flow control, speed 1152000 baud

    The actual CDI2 UART baud rate is 1.25Mbaud. Requesting the canonical speed of
    termios.B1152000 sets the DLL and DLM (i.e., divisor registers) of the AXI
    UART to 5, which translates to the desired rate.

    see termios(3), cfmakeraw(3)

    Parameters
    ----------
    fUART : io.BufferedIOBase
        file object of UART interface
    """
    # get attributes
    [ iflag, oflag, cflag, lflag, ispeed, ospeed, cc ] = termios.tcgetattr(fUART)

    # setup raw mode; see termios(3), cfmakeraw(3)
    cc[termios.VTIME] = 1
    cc[termios.VMIN]  = 1
    iflag = iflag & ~(termios.IGNBRK |
                      termios.BRKINT |
                      termios.PARMRK |
                      termios.ISTRIP |
                      termios.INLCR |
                      termios.IGNCR |
                      termios.ICRNL |
                      termios.IXON)
    oflag = oflag & ~termios.OPOST
    cflag = cflag & ~(termios.CSIZE |
                      termios.PARENB)
    cflag = cflag | termios.CS8
    lflag = lflag & ~(termios.ECHO |
                      termios.ECHONL |
                      termios.ICANON |
                      termios.ISIG |
                      termios.IEXTEN)
    lflag = lflag | termios.CREAD | termios.CLOCAL

    #ispeed = termios.B1152000
    #ospeed = termios.B1152000

    termios.tcsetattr(fUART, termios.TCSANOW, [ iflag, oflag, cflag, lflag, ispeed, ospeed, cc ])
    setSerialSpeed(fUART, args.speed)

    # set to non-blocking mode
    os.set_blocking(fUART.fileno(), False)

# idiomatic conditional main script stanza
if __name__ == "__main__":
    # setup command line parser and options
    parser = argparse.ArgumentParser(description='Tool to configure serial device for non canonical speeds')
    parser.add_argument('-d', '--device',    action = 'store',     required = True,  help = 'serial interface to configure')
    parser.add_argument('-s', '--speed',     action = 'store',     required = True,  help = 'baud rate', type = int)

    # parse command line
    args = parser.parse_args()

    with open(args.device, 'rb+', buffering = 0) as fUART:
        configureUART(fUART,args)
