#!/usr/bin/python3

##
# @file cdiUartTest.py
# @brief UART test driver investigation device overruns
#
# @version 1.0
# @author AIT
# @copyright &copy;2023 AIT Austrian Institute of Technology

import argparse
import asyncio
import enum
import logging
import os.path
import platform
import random
import signal
import struct
import sys
import termios
import time

import crc16

## CDI frame special characters
class CdiFrameChar(enum.IntEnum):
    CDI_RSV0 = 0xf8
    CDI_RSV1 = 0xf9
    CDI_RSV2 = 0xfa
    CDI_RSV3 = 0xfb
    CDI_STX  = 0xfc
    CDI_ETX  = 0xfd
    CDI_ESC  = 0xfe

def formatCdiMessage(msg):
    """
    format a CDI message for transmission.

    This function takes a bytes object or byte array, msg, calculates the CDI checksum,
    escapes reserved bytes, frames the message with the required STX and ETX framing
    octets, and returns the thus prepared frame

    Parameters
    ----------
    msg : bytes, bytearray, or list of integers
        unescaped, unframed message to transmit

    Returns
    -------
    bytearray
        framed, checksummed, and escaped message ready to be transmitted
    """

    # start with start-of-frame marker and reset CRC16
    cdiMsg = bytearray([ CdiFrameChar.CDI_STX ])
    crc    = crc16.CRC16_INIT

    # append message bytes, escape reserved characters
    for b in msg:
        # update CRC16
        crc = crc16.crc16Incr(b, crc)

        # check for reserved character and escape
        if ((b >= CdiFrameChar.CDI_RSV0) and (b <= CdiFrameChar.CDI_ESC)):
            cdiMsg.append(CdiFrameChar.CDI_ESC)
            cdiMsg.append(~b & 0xff)
        else:
            cdiMsg.append(b)

    # pack CRC16 checksum as 2 bytes, little endian and append escaped to message
    for b in struct.pack('<H', crc):
        # check for reserved character and escape
        if ((b >= CdiFrameChar.CDI_RSV0) and (b <= CdiFrameChar.CDI_ESC)):
            cdiMsg.append(CdiFrameChar.CDI_ESC)
            cdiMsg.append(~b & 0xff)
        else:
            cdiMsg.append(b)

    # terminate with end-of frame marker and return frame
    cdiMsg.append(CdiFrameChar.CDI_ETX)
    return cdiMsg

def configureUART(fUART):
    """
    configure local UART settings: raw mode, no modem and flow control, speed 1152000 baud

    The actual CDI2 UART baud rate is 1.25Mbaud. Requesting the canonical speed of
    termios.B1152000 sets the DLL and DLM (i.e., divisor registers) of the AXI
    UART to 5, which translates to the desired rate.

    see termios(3), cfmakeraw(3)

    Parameters
    ----------
    fUART : io.BufferedIOBase
        file object of UART interface
    """
    # get attributes
    [ iflag, oflag, cflag, lflag, ispeed, ospeed, cc ] = termios.tcgetattr(fUART)

    # setup raw mode; see termios(3), cfmakeraw(3)
    cc[termios.VTIME] = 1
    cc[termios.VMIN]  = 1
    iflag = iflag & ~(termios.IGNBRK |
                      termios.BRKINT |
                      termios.PARMRK |
                      termios.ISTRIP |
                      termios.INLCR |
                      termios.IGNCR |
                      termios.ICRNL |
                      termios.IXON)
    oflag = oflag & ~termios.OPOST
    cflag = cflag & ~(termios.CSIZE |
                      termios.PARENB)
    cflag = cflag | termios.CS8
    lflag = lflag & ~(termios.ECHO |
                      termios.ECHONL |
                      termios.ICANON |
                      termios.ISIG |
                      termios.IEXTEN)
    lflag = lflag | termios.CREAD | termios.CLOCAL
    ispeed = termios.B1152000
    ospeed = termios.B1152000

    termios.tcsetattr(fUART, termios.TCSANOW, [ iflag, oflag, cflag, lflag, ispeed, ospeed, cc ])

    # set to non-blocking mode
    os.set_blocking(fUART.fileno(), False)

def traceUart(prefix, m):
    """
    trace message bytes transmitted to or received from the UART

    Parameters
    ----------
    prefix: string
        log prefix
    m: bytes or bytearray
        transmitted/received bytes
    """

    if (logging.getLogger().isEnabledFor(logging.DEBUG)):
        logging.debug(prefix + ' 0x[' + ':'.join('{:02x}'.format(b) for b in m) + ']')

def doSignal(sigName, fu):
    """
    handle INT/HUP/PIPE/TERM signals

    Parameters
    ----------
    sigName: string
        signal name
    fu: asyncio.Future
        future to satisfy
    """

    logging.debug('SIG: caught %s', sigName)

    if (not fu.done()):
        fu.set_result(1)

async def TOP(args):
    """
    top-level coroutine dumping CDI frames
    """

    lengthRange = [ 1 if x <= 0 else x for x in [int(l) - 3 for l in args.length.split(':')]]
    lenMin      = lengthRange[0]

    if (len(lengthRange) > 1):
        lenMax = int(lengthRange[1])
    else:
        lenMax = lenMin

    # get event loop
    loop = asyncio.get_running_loop()

    logging.info('TOP: connect and configure UART interface {0}'.format(args.device))
    with open(args.device, 'rb+', buffering = 0) as fUART:
        configureUART(fUART)

        for l in range(lenMin, lenMax + 1):
            cdiFrame = formatCdiMessage(bytes([(l + 3) % 256]) + random.randbytes(l))

            logging.info('TOP: sending {0} {1} byte message(s)'.format(args.number, l + 3))

            for i in range(0, args.number):
                fUART.write(cdiFrame)

    logging.info('TOP: done')

# idiomatic conditional main script stanza
if __name__ == "__main__":
    # setup command line parser and options
    parser = argparse.ArgumentParser(description='CDI2 Device test driver')
    parser.add_argument('-D', '--device', action = 'store', required = True,  help = 'UART interface to IP core')
    parser.add_argument('-r', '--role',   action = 'store', required = False, help = 'simulate Camera or generic Detector', default = 'Camera', choices = ['Camera', 'Device'])
    parser.add_argument('-n', '--number', action = 'store', required = False, help = 'number of messages/length',           default = 10,   type = int)
    parser.add_argument('-L', '--length', action = 'store', required = False, help = 'length or range of lengths min:max',  default = '10', type = str)
    parser.add_argument('-l', '--log',    action = 'store', required = False, help = 'log level',                           default = 'INFO')

    # parse command line
    args = parser.parse_args()

    # configure logging
    l = getattr(logging, args.log, None)

    if not isinstance(l, int):
        raise ValueError('invalid log level {0}'.format(args.log))

    logging.Formatter.converter = time.gmtime
    logging.basicConfig(format = '%(asctime)s %(levelname)s: %(message)s', level = l)

    try:
        # start TOP coroutine
        asyncio.run(TOP(args))
    except BaseException as err:
        logging.error(err)
