#!/bin/bash
#-------------------------------------------------------------------------------
#
# cdi2app.sh
#
# start pppd(8) and ftpd(8) deamons and the CDI2 Device applicaton
#
# (c)2023 AIT Austrian Institute of Technology
#-------------------------------------------------------------------------------
programname=$0

ROLE="Camera"
NODE_ID="" # 1=Camera, 2=Detector
UART1="/dev/ttyS0"
UART2="/dev/ttyS1"
LOG_LEVEL="INFO"
TTS=""
SIZE=""

MAIN="___"

logdate()
{
    echo "$(date '+%Y-%m-%d %T,%3N')"
}

printlog() {

  if [ "$2" == "NODATE" ]; then
    while IFS= read -r line; do printf '%s %s\n' "$1:" "$line"; done
  else
    while IFS= read -r line; do printf '%s %s\n' "$1: $(logdate)" "$line"; done
  fi
}

fix_usb_latency() {
  local DEV=$1
  if [ "$(echo $DEV | grep USB | wc -l)" == "1" ]; then
    CFG="/sys/bus/usb-serial/devices/$(echo $DEV | awk -F/ '{print $NF}')/latency_timer"
    echo "INFO: Setting latency_timer to 0 in file $CFG" | printlog "$MAIN"
    echo 0 > "$CFG"
  fi
}

print_exit_message() {
  local NAME=$1
  local RET=$2
  local DATE=$3
  if [ "$RET" != "0" ]; then
    echo "${DATE}ERROR: $NAME exited with code $RET"
  else
    echo "${DATE}INFO: $NAME exited with code $RET"
  fi
}

number_in_range() {
  ([ "$1" -ge $2 ] && [ "$1" -le $3 ]) 2> /dev/null && echo "$1" | bc
}

usage() {
    echo "usage: $programname [-r|--role arg] [-i|--nodeId arg] [-d|device arg] [-s|--resultSize arg] [-l|--log arg] [--help] [--noTTS]"
    echo "  -r, --role           Set the cdi2 device role [Camera|Detector]"
    echo "  -i, --nodeId         Set the cdi2 device node id [1,16], default is 1/2 depending on the role Camera/Detector."
    echo "  -d, --device         Set the uart devices to use, default is '/dev/ttyS0,/dev/ttyS1'"
    echo "  -s, --resultSize     Set the size of the BnResult [15,1412] being sent."
    echo "  -l, --log            Set the log level [DEBUG, INFO, WARNING, ERROR, CRITICAL], default is 'INFO'."
    echo "  --noTTS              Disable TTS"
    echo "  --help               Prints this description."
}

options=$(getopt -l "help,noTTS,role:,nodeId:,device:,resultSize:,log:" -o "r:i:d:s:l:" -a -- "$@")

if [ $? -ne 0 ]; then
  usage
  exit 1
fi

eval set -- "$options"
while true; do
  case "$1" in
    --help)
      usage
	  exit
      ;;
    --noTTS)
      TTS="--noTTS"
      ;;
	-r|--role)
	  shift
      ROLE=$1
      if [ "$ROLE" != "Detector" ] && [ "$ROLE" != "Camera" ]; then
        echo "ERROR: Invalid role '$ROLE'" | printlog "$MAIN"
        exit 1
      fi
      ;;
    -i|--nodeId)
	  shift
      NODE_ID="$(number_in_range $1 1 16)"
      if [ $? -ne 0 ]; then
        echo "ERROR: Invalid nodeId '$1'. It must be within [1,16]." | printlog "$MAIN"
        exit 1
      fi
      ;;
    -d|--device)
	  shift
      if [[ "$1" != *","* ]]; then
        echo -e "ERROR: Please specify two devices seperated with a colon, e.g. '/dev/ttyS0,/dev/ttyS1'" | printlog "$MAIN"
        exit 1
      fi
      UART1=$(echo $1 | cut -d, -f 1)
      UART2=$(echo $1 | cut -d, -f 2)
      ;;
    -s|--resultSize)
	  shift
      SIZE="-s$(number_in_range $1 15 1412)"
      if [ $? -ne 0 ]; then
        echo "ERROR: Invalid resultSize '$1'. It must be within [15,1412]." | printlog "$MAIN"
        exit 1
      fi
      ;;
    -l|--log)
	  shift
      LOG_LEVEL="$1"
      VALID=$(echo -e "DEBUG\nINFO\nWARNING\nERROR\nCRITICAL" | grep "^$1$" | wc -l)
      if [ "$VALID" != "1" ]; then
        echo "ERROR: Invalid log level '$1'. It must be one of [DEBUG, INFO, WARNING, ERROR, CRITICAL]." | printlog "$MAIN"
        exit 1
      fi
      ;;
    --)
      shift
      break;;
  esac
  shift
done

# Set default nodeId, if required
[ "$NODE_ID" == "" ] && [ "$ROLE" == "Camera" ] && NODE_ID=1
[ "$NODE_ID" == "" ] && [ "$ROLE" == "Detector" ] && NODE_ID=2

echo "INFO: Using ROLE=$ROLE, UART1=$UART1, UART2=$UART2, NODE_ID=$NODE_ID, TTS=$TTS, SIZE=$SIZE, LOG_LEVEL=$LOG_LEVEL" | printlog "$MAIN"

SPEED=""
if [ "$(cat /proc/cpuinfo | grep Xilinx | wc -l)" == "0" ]; then
  # set correct speed of uart interface (not required on zync)
  echo "INFO: Configuring UART2 interface $UART2 on external computer." | printlog "$MAIN"
  python setSerial.py -d $UART2 -s 1250000 2>&1 | printlog "$MAIN"
else
  # only on zync directly
  SPEED="1152000"
fi


fix_usb_latency $UART1
fix_usb_latency $UART2


{
  echo "INFO: Preparing ip-up.d script"
  cp ./50cdi2 /etc/ppp/ip-up.d/
  chmod a+x /etc/ppp/ip-up.d/50cdi2
  #start pppd(8) in background
  echo "INFO: Starting pppd in background"
  pppd $UART2 $SPEED 192.168.100.$NODE_ID: file options.uart2 nodetach debug record ppp.log
  print_exit_message "pppd" $?
} 2>&1 | printlog "PPP" &


{
  #start ftpd(8) in background
  echo "INFO: Starting tcpsvd (ftpd) in background"
  tcpsvd -vE 0.0.0.0 21 ftpd -w -A -vv ./ftp
  print_exit_message "tcpsvd/ftpd" $?
} 2>&1 | printlog "FTP" &


{
  #start httpd in background
  echo "INFO: Starting webserver (python http.server) in background"
  python -m http.server --directory ./ftp  80
  print_exit_message "httpd" $?
} 2>&1 | printlog "WEB" &


{
  #start TestApp in background
  echo "$(logdate) INFO: Starting cdi2DeviceApp.py in background"

  SRC="./ftp/device_info-cs.xml"
  [ "$ROLE" == "Detector" ] && SRC="./ftp/device_info-det.xml"
  cp "$SRC" ./ftp/device_info.xml
  echo "$(logdate) INFO: Setting role to '$ROLE' with '$SRC'"

  if [ "$TTS" == "--noTTS" ]; then
    sed -i 's/ttsSupport="true"/ttsSupport="false"/g' ./ftp/device_info.xml
  fi
  echo "$(logdate) INFO: Updated device_info.xml: $(cat ./ftp/device_info.xml | sed 's/ /\n/g' | grep ttsSupport)"

  sed -i "s/--NODEID--/$NODE_ID/g" ./ftp/device_info.xml
  echo "$(logdate) INFO: Updated device_info.xml: $(cat ./ftp/device_info.xml | grep nodeId)"

  chrt -r 10 python ./cdi2DeviceApp.py -D $UART1 -n $NODE_ID -r $ROLE $TTS $SIZE -l $LOG_LEVEL
  print_exit_message "cdi2DeviceApp.py" $? "$(logdate) "
} 2>&1 | printlog "APP" "NODATE" &


{
  #test ftpd after 4 seconds
  sleep 4
  echo "INFO: Checking ftpd"
  RES=$(netcat -w 2 localhost 21)
  if [ "$(echo $RES | grep "220" | wc -l)" == "1" ]; then
    echo "INFO: ftp server ok"
  else
    echo "ERROR ftp server not working"
  fi
} 2>&1 | printlog "FTP" &

{
  #print routing table in about 10s
  sleep 10
  echo "INFO: Checking routing table"
  ip route
  if [ "$(ip route | grep "192.168.100" | wc -l)" == "2" ]; then
    echo "INFO: routing table ok"
  else
    echo "WARN: routing likely wrong"
  fi
} 2>&1 | printlog "PPP" &
wait
