#!/usr/bin/python3

##
# @file cdi2DeviceApp.py
# @brief Test driver application for the CDI2 Device IP core
#
# @version 1.0
# @author AIT
# @copyright &copy;2023 AIT Austrian Institute of Technology

import array
import argparse
import asyncio
from enum import Enum, IntEnum
import fcntl
import logging
import os.path
import platform
import signal
import struct
import sys
import termios
import time

import crc16

## CDI frame special characters
class CdiFrameChar(IntEnum):
    CDI_RSV0 = 0xf8
    CDI_RSV1 = 0xf9
    CDI_RSV2 = 0xfa
    CDI_RSV3 = 0xfb
    CDI_STX  = 0xfc
    CDI_ETX  = 0xfd
    CDI_ESC  = 0xfe

# Reassembly task unmarshalling status
class UmaStatus(Enum):
    UMA_IDLE = 0            ##< not in a message, scanning for STX
    UMA_STX  = 1            ##< STX scanned, in-message
    UMA_ESC  = 2            ##< in-message and ESC scanned

## CDI Frame ID
class CdiFrameId(IntEnum):
    CCR_DetectorReset         = 0x10
    CCR_ProtocolVersion       = 0x11
    CCR_ProtocolVersionAnswer = 0x21
    CCR_TtsReset              = 0x23
    CCR_NmtReset              = 0x24
    CCR_StartCdi2             = 0x15
    CCR_StartCdi2Answer       = 0x25
    CCR_DetStatus             = 0x16
    CCR_BsmStatus             = 0x26
    CCR_SetMaintenanceState   = 0x17
    CCR_SetErrorState         = 0x18
    CCR_Fatal                 = 0x29
    CCR_Error                 = 0x2a
    CCR_Warning               = 0x2b
    CCR_Info                  = 0x2c
    CCR_Debug                 = 0x2d
    CCR_Trace                 = 0x2e
    CCR_BnTrigger             = 0x80
    CCR_BnId                  = 0x81
    CCR_BnInfo                = 0x82
    CCR_BnResult              = 0x41
    CCR_BnRecognition         = 0x42
    CCR_AcceptUpdate          = 0x47
    CCR_RejectUpdate          = 0x48
    CCR_PrepareUpdate         = 0x87
    CCR_PerformUpdate         = 0x88

## BSM status constants
class BsmState(IntEnum):
    BS_INVALID              = 0
    BS_START_UP             = 1
    BS_INITIALISATION       = 2
    BS_FEED_OFF             = 3
    BS_REQUEST_TO_SORT      = 4
    BS_SORTING              = 5
    BS_REQUEST_TO_SHUT_DOWN = 6
    BS_SHUTDOWN             = 7
    BS_ERROR                = 10

## DET status constants
class DeviceState(IntEnum):
    DS_INVALID            = 0
    DS_START_UP           = 1
    DS_INITIALISATION     = 2
    DS_INITIALISED        = 3
    DS_FEED_OFF           = 4
    DS_READY_TO_SORT      = 5
    DS_SORTING            = 6
    DS_READY_TO_SHUT_DOWN = 7
    DS_SHUTDOWN           = 8
    DS_ERROR              = 10

## Linux ioctl codes to access struct termios2
class TERMIOS2(IntEnum):
    TCGETS2 = 0x802C542A
    TCSETS2 = 0x402C542B

## standard invalid BNID
BNINVALID = 0xffffffff

## default BN serial number as used by simulator (delayed_camera_system.lua)
serialNo = 3530953213

def csResult(bnId, series, denomination, orientation, segment = 0):
    """
    prepare a camera result according to the fields in
    device_info-cs.xml

    Parameters
    ----------
    bnId : int
        banknote ID
    series : int
        bankote series
    denomination : int
        banknote denomination
    orientation : int
        banknote orientation
    segment : int
        BNRESULT segment number

    Returns
    -------
    bytearray
        CDI BNRESULT message, unescaped, unframed (1412 bytes + CDI frame
        ID byte), should be trimmed to desired length before passing it to
        formatCdiMessage()
    """

    # bump serial number
    global serialNo
    serialNo  = serialNo + 1

    # CS result fields
    judgement    = 0
    result       = 0
    quality      = 0
    serialNumber = bytes('PA' + '{0:010d}'.format(serialNo), 'ascii')
    bnlength     = 0
    bnwidth      = 0
    soil         = 0

    return struct.pack('<bIbbbxbxxxbbb16sbbbIII1378x',
                       CdiFrameId.CCR_BnResult,
                       bnId,
                       series, denomination, orientation,
                       segment,
                       judgement, result, quality,
                       serialNumber,
                       orientation, denomination, series,
                       bnlength, bnwidth, soil)

def detResult(bnId, series, denomination, orientation, segment = 0):
    """
    prepare a camera result according to the fields in
    device_info-det.xml

    Parameters
    ----------
    bnId : int
        banknote ID
    series : int
        bankote series
    denomination : int
        banknote denomination
    orientation : int
        banknote orientation
    segment : int
        BNRESULT segment number

    Returns
    -------
    bytearray
        CDI BNRESULT message, unescaped, unframed (1412 bytes + CDI frame
        ID byte), should be trimmed to desired length before passing it to
        formatCdiMessage()
    """

    # DET result fields
    judgement    = 0
    result       = 0
    quality      = 0
    intensity    = 0
    secret       = ord('\u0054')

    return struct.pack('<bIbbbxbxxxbbbxIH1402x',
                       CdiFrameId.CCR_BnResult,
                       bnId,
                       series, denomination, orientation,
                       segment,
                       judgement, result, quality,
                       intensity, secret)

def formatCdiMessage(msg):
    """
    format a CDI message for transmission.

    This function takes a bytes object or byte array, msg, calculates the CDI checksum,
    escapes reserved bytes, frames the message with the required STX and ETX framing
    octets, and returns the thus prepared frame

    Parameters
    ----------
    msg : bytes, bytearray, or list of integers
        unescaped, unframed message to transmit

    Returns
    -------
    bytearray
        framed, checksummed, and escaped message ready to be transmitted
    """

    # start with start-of-frame marker and reset CRC16
    cdiMsg = bytearray([ CdiFrameChar.CDI_STX ])
    crc    = crc16.CRC16_INIT

    # append message bytes, escape reserved characters
    for b in msg:
        # update CRC16
        crc = crc16.crc16Incr(b, crc)

        # check for reserved character and escape
        if ((b >= CdiFrameChar.CDI_RSV0) and (b <= CdiFrameChar.CDI_ESC)):
            cdiMsg.append(CdiFrameChar.CDI_ESC)
            cdiMsg.append(~b & 0xff)
        else:
            cdiMsg.append(b)

    # pack CRC16 checksum as 2 bytes, little endian and append escaped to message
    for b in struct.pack('<H', crc):
        # check for reserved character and escape
        if ((b >= CdiFrameChar.CDI_RSV0) and (b <= CdiFrameChar.CDI_ESC)):
            cdiMsg.append(CdiFrameChar.CDI_ESC)
            cdiMsg.append(~b & 0xff)
        else:
            cdiMsg.append(b)

    # terminate with end-of frame marker and return frame
    cdiMsg.append(CdiFrameChar.CDI_ETX)
    return cdiMsg

def setSerialSpeed(fUART, speed):
    """
    The actual CDI2 UART baud rate is 1.25Mbaud. Requesting the canonical speed of
    termios.B1152000 sets the DLL and DLM (i.e., divisor registers) of the AXI
    UART to 5, which translates to the desired rate. However, this does not work
    with the FTDI USB/UART transceivers which need accurate speed settings.
    Thus, we must use the struct termios2 and TCGETS/TCSET2 ioctl() to configure
    the UART speed.

    see termios(3), ioctl_tty(2)

    Parameters
    ----------
    fUART : io.BufferedIOBase
        file object of UART interface
    speed : int
        desired baud rate
    """

    # sizeof(struct termios2) is 44, align conservatively
    #
    # struct termios2 {
    #     tcflag_t c_iflag;     /*  [0] input mode flags */
    #     tcflag_t c_oflag;     /*  [1] output mode flags */
    #     tcflag_t c_cflag;     /*  [2] control mode flags */
    #     tcflag_t c_lflag;     /*  [3] local mode flags */
    #     cc_t c_line;          /*  [4] line discipline */
    #     cc_t c_cc[NCCS];      /*      control characters */
    #     speed_t c_ispeed;     /*  [9] input speed */
    #     speed_t c_ospeed;     /* [10] output speed */
    # };
    termios2 = array.array('I', [0] * 64)

    # read complete struct termios2
    fcntl.ioctl(fUART, TERMIOS2.TCGETS2, termios2)

    # update with custom speed
    termios2[2]  &= ~termios.CBAUD
    termios2[2]  |= termios.CBAUDEX
    termios2[9]   = speed
    termios2[10]  = speed

    # set updated termios2
    fcntl.ioctl(fUART, TERMIOS2.TCSETS2, termios2)

def configureUART(fUART):
    """
    configure local UART settings: raw mode, no modem and flow control, speed 1152000 baud

    The actual CDI2 UART baud rate is 1.25Mbaud. Requesting the canonical speed of
    termios.B1152000 sets the DLL and DLM (i.e., divisor registers) of the AXI
    UART to 5, which translates to the desired rate.

    see termios(3), cfmakeraw(3)

    Parameters
    ----------
    fUART : io.BufferedIOBase
        file object of UART interface
    """
    # get attributes
    [ iflag, oflag, cflag, lflag, ispeed, ospeed, cc ] = termios.tcgetattr(fUART)

    # setup raw mode; see termios(3), cfmakeraw(3)
    cc[termios.VTIME] = 1
    cc[termios.VMIN]  = 1
    iflag = iflag & ~(termios.IGNBRK |
                      termios.BRKINT |
                      termios.PARMRK |
                      termios.ISTRIP |
                      termios.INLCR |
                      termios.IGNCR |
                      termios.ICRNL |
                      termios.IXON)
    oflag = oflag & ~termios.OPOST
    cflag = cflag & ~(termios.CSIZE |
                      termios.PARENB)
    cflag = cflag | termios.CS8
    lflag = lflag & ~(termios.ECHO |
                      termios.ECHONL |
                      termios.ICANON |
                      termios.ISIG |
                      termios.IEXTEN)
    lflag = lflag | termios.CREAD | termios.CLOCAL
    ispeed = termios.B1152000
    ospeed = termios.B1152000

    termios.tcsetattr(fUART, termios.TCSANOW, [ iflag, oflag, cflag, lflag, ispeed, ospeed, cc ])
    setSerialSpeed(fUART, 1250000)

    # set to non-blocking mode
    os.set_blocking(fUART.fileno(), False)

def traceUart(prefix, m):
    """
    trace message bytes transmitted to or received from the UART

    Parameters
    ----------
    prefix: string
        log prefix
    m: bytes or bytearray
        transmitted/received bytes
    """

    if (logging.getLogger().isEnabledFor(logging.DEBUG)):
        logging.debug(prefix + ' 0x[' + ':'.join('{:02x}'.format(b) for b in m) + ']')

async def App(fUART, quApp, args):
    """
    CDI2 application task.

    Parameters
    ----------
    fUART : io.BufferedIOBase
        file object of UART interface
    quApp: asyncio.Queue
        application task queue
    args: argsparse.Namespace
        parsed command line arguments
    """

    logging.info('APP: CDI2 Device Application')

    # assume CDI2 Core status not ready until handshake complete
    stackStarted   = 0
    coreStatus     = 1
    nmtStatus      = 0
    bsmStatus      = BsmState.BS_INVALID
    devStatus      = DeviceState.DS_INVALID
    mtcStatus      = 0
    errStatus      = 0
    bnId           = BNINVALID
    bnSeries       = 0
    bnDenomination = 0
    bnOrientation  = 0
    bnIdInfo       = [ BNINVALID ] * 20
    bnIdTrig       = [ BNINVALID ] * 20
    needBsError    = False

    # start with CDI initial handshake
    logging.info('APP: Protocol Version 2.0')
    cdiReq = formatCdiMessage(struct.pack('bbb', CdiFrameId.CCR_ProtocolVersion, 0, 2))
    traceUart('APP: req', cdiReq)
    fUART.write(cdiReq)

    while True:
        try:
            msg = await asyncio.wait_for(quApp.get(), timeout = 1.0)
            traceUart('APP: msg', msg)

            # sanity
            if (len(msg) < 1):
                quApp.task_done()
                continue

            if (msg[0] == CdiFrameId.CCR_ProtocolVersionAnswer):
                (id, coreStatus, minor, major, nmtStatus, bsmStatus, devStatus, mtcStatus, errStatus) = struct.unpack('bbbbBbbbb', msg)
                logging.info('DEV: IP Core status %d, protocol %d.%d, BSM %d, DET %d, NMT 0x%02x', coreStatus, major, minor, bsmStatus, devStatus, nmtStatus)

                if ((major != 0x02) or (minor != 0x00)):
                    logging.warning('APP: cannot handle CDI2 IP Core protocol %d.%d', major, minor)
                    coreStatus = 2
                else:
                    if ((coreStatus == 0) and (nmtStatus == 0)):
                        # NMT status is NMT_GS_OFF, start stack
                        nodeId = args.nodeId
                        useTTS = not args.noTTS
                        bfa2   = args.bfa2
                        mac    = bytes([ 0x00, 0x60, 0x36, 0x9f, 0xc5, 0x20 + nodeId])
                        logging.info('APP: Start CDI2, nodeID %d, use2ndBFA %d, useTTS %d, logLevel %d', nodeId, bfa2, useTTS, 60)
                        cdiReq = formatCdiMessage(struct.pack('bbbbb6B', CdiFrameId.CCR_StartCdi2, args.nodeId, bfa2, useTTS, 60, *mac))
                        traceUart('APP: req', cdiReq)
                        fUART.write(cdiReq)
            elif (msg[0] == CdiFrameId.CCR_TtsReset):
                logging.info('DEV: TTS reset detected')

                # do not keep state over hard reset
                needBsError = False

                logging.info('APP: Software Detector Reset, cause 0x01')
                cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_DetectorReset, 1))
                traceUart('APP: req', cdiReq)
                fUART.write(cdiReq)
            elif (msg[0] == CdiFrameId.CCR_NmtReset):
                (id, reason) = struct.unpack('bb', msg)
                logging.info('DEV: NMT reset reported, reason 0x%02x', reason)

                # support BSMS_BSM-022 Powerlink Errors: stay in DS_ERROR until BS_* update
                if (devStatus == DeviceState.DS_ERROR) and (needBsError == True):
                    # set DS_ERROR
                    devStatus = DeviceState.DS_ERROR
                    logging.info('APP: DET Status %d', devStatus)
                    cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_DetStatus, devStatus))
                    traceUart('APP: req', cdiReq)
                    fUART.write(cdiReq)
                else:
                    # set DS_START_UP
                    devStatus = DeviceState.DS_START_UP
                    logging.info('APP: DET Status %d', devStatus)
                    cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_DetStatus, devStatus))
                    traceUart('APP: req', cdiReq)
                    fUART.write(cdiReq)

                    # reset error state
                    errStatus = 0
                    cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_SetErrorState, errStatus))
                    traceUart('APP: req', cdiReq)
                    fUART.write(cdiReq)

                    # do not keep state over hard reset
                    needBsError = False
                    bnIdInfo    = [ BNINVALID ] * len(bnIdInfo)
                    bnIdTrig    = [ BNINVALID ] * len(bnIdTrig)

                # send protocol handshake to update layer stati
                logging.info('APP: Protocol Version 2.0')
                cdiReq = formatCdiMessage(struct.pack('bbb', CdiFrameId.CCR_ProtocolVersion, 0, 2))
                traceUart('APP: req', cdiReq)
                fUART.write(cdiReq)
            elif (msg[0] == CdiFrameId.CCR_StartCdi2Answer):
                logging.info('DEV: Start CDI2 Answer')

                # do not keep state over hard reset
                needBsError = False
                bnIdInfo    = [ BNINVALID ] * len(bnIdInfo)
                bnIdTrig    = [ BNINVALID ] * len(bnIdTrig)

                # set DS_START_UP
                devStatus = DeviceState.DS_START_UP
                logging.info('APP: DET Status %d', devStatus)
                cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_DetStatus, devStatus))
                traceUart('APP: req', cdiReq)
                fUART.write(cdiReq)

                # send protocol handshake to update layer stati
                logging.info('APP: Protocol Version 2.0')
                cdiReq = formatCdiMessage(struct.pack('bbb', CdiFrameId.CCR_ProtocolVersion, 0, 2))
                traceUart('APP: req', cdiReq)
                fUART.write(cdiReq)
            elif (msg[0] == CdiFrameId.CCR_BsmStatus):
                (id, bsmStatus) = struct.unpack('bb', msg)
                logging.info('DEV: BSM Status %d', bsmStatus)

                # DtC: do as told by BSM, even when waiting for BS_ERROR
                needBsError = False

                if (bsmStatus == BsmState.BS_START_UP):
                    devStatus = DeviceState.DS_START_UP
                elif (bsmStatus == BsmState.BS_INITIALISATION):
                    if (devStatus != DeviceState.DS_ERROR):
                        # support BSMS_DEV-005 State Transitions
                        devStatus = DeviceState.DS_INITIALISATION
                        logging.info('APP: DET Status %d', devStatus)
                        cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_DetStatus, devStatus))
                        traceUart('APP: req', cdiReq)
                        fUART.write(cdiReq)
                        # sleep 5ms after DS_INITIALISATION to skip one PL cycle before DS_INITIALISED is send
                        time.sleep(0.005)

                        devStatus = DeviceState.DS_INITIALISED
                elif (bsmStatus == BsmState.BS_FEED_OFF):
                    devStatus = DeviceState.DS_FEED_OFF
                    bnIdInfo  = [ BNINVALID ] * len(bnIdInfo)
                    bnIdTrig  = [ BNINVALID ] * len(bnIdTrig)
                elif (bsmStatus == BsmState.BS_REQUEST_TO_SORT):
                    if (devStatus != DeviceState.DS_ERROR):
                        devStatus = DeviceState.DS_READY_TO_SORT
                        bnIdInfo  = [ BNINVALID ] * len(bnIdInfo)
                        bnIdTrig  = [ BNINVALID ] * len(bnIdTrig)
                elif (bsmStatus == BsmState.BS_SORTING):
                    if (devStatus != DeviceState.DS_ERROR):
                        devStatus = DeviceState.DS_SORTING
                        bnIdInfo  = [ BNINVALID ] * len(bnIdInfo)
                        bnIdTrig  = [ BNINVALID ] * len(bnIdTrig)
                elif (bsmStatus == BsmState.BS_REQUEST_TO_SHUT_DOWN):
                    if (devStatus == DeviceState.DS_INVALID):
                        devStatus = DeviceState.DS_START_UP
                    elif (devStatus != DeviceState.DS_START_UP):
                        devStatus = DeviceState.DS_READY_TO_SHUT_DOWN
                elif (bsmStatus == BsmState.BS_SHUTDOWN):
                    if (devStatus == DeviceState.DS_INVALID):
                        devStatus = DeviceState.DS_START_UP
                    elif (devStatus != DeviceState.DS_START_UP):
                        devStatus = DeviceState.DS_SHUTDOWN
                elif (bsmStatus == BsmState.BS_ERROR):
                    devStatus = DeviceState.DS_ERROR

                logging.info('APP: DET Status %d', devStatus)
                cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_DetStatus, devStatus))
                traceUart('APP: req', cdiReq)
                fUART.write(cdiReq)
            elif (msg[0] == CdiFrameId.CCR_BnId):
                (id, bnId, tc, tcTrig, tsTrig) = struct.unpack('<bIIII', msg)
                logging.debug('DEV: Banknote ID %u, ts %u [ns]', bnId, tsTrig)
            elif (msg[0] == CdiFrameId.CCR_BnTrigger):
                (id, bfa, bnId) = struct.unpack('<bbI', msg)
                logging.debug('DEV: Banknote Trigger %u', bnId)

                if (bnId != BNINVALID):
                    if (args.role == 'Camera'):
                        logging.debug('APP: Banknote Recognition %u', bnId)

                        bnSeries       = 1
                        bnDenomination = 2
                        bnOrientation  = 3

                        cdiReq = formatCdiMessage(struct.pack('<bIbbb', CdiFrameId.CCR_BnRecognition, bnId, bnSeries, bnDenomination, bnOrientation))
                        traceUart('APP: req', cdiReq)
                        fUART.write(cdiReq)
                    elif (args.role == 'Detector'):
                        bnIdTrig[bnId % 20] = bnId

                        if (bnIdInfo[bnId % 20] == bnId):
                            logging.debug('APP: Banknote Result %u', bnId)
                            cdiReq = formatCdiMessage(detResult(bnId, bnSeries, bnDenomination, bnOrientation, 0)[:args.resultSize + 1])
                            traceUart('APP: req', cdiReq)
                            fUART.write(cdiReq)
                    else:
                        raise RuntimeError('do not know how to act as ' + args.role)
            elif (msg[0] == CdiFrameId.CCR_BnInfo):
                (id, bnId, bnSeries, bnDenomination, bnOrientation) = struct.unpack('<bIbbb', msg)
                logging.debug('DEV: Banknote Information %u', bnId)

                if (bnId != BNINVALID):
                    if (args.role == 'Camera'):
                        cdiReq = formatCdiMessage(csResult(bnId, bnSeries, bnDenomination, bnOrientation, 0)[:args.resultSize + 1])
                        logging.debug('APP: Banknote Result %u', bnId)
                        traceUart('APP: req', cdiReq)
                        fUART.write(cdiReq)
                    elif (args.role == 'Detector'):
                        bnIdInfo[bnId % 20] = bnId

                        if (bnIdTrig[bnId % 20] == bnId):
                            logging.debug('APP: Banknote Result %u', bnId)
                            cdiReq = formatCdiMessage(detResult(bnId, bnSeries, bnDenomination, bnOrientation, 0)[:args.resultSize + 1])
                            traceUart('APP: req', cdiReq)
                            fUART.write(cdiReq)
                    else:
                        raise RuntimeError('do not know how to act as ' + args.role)
            elif (msg[0] == CdiFrameId.CCR_Fatal):
                logging.fatal('DEV[F]: %s', msg[2:-1].decode('ascii'))

                # set error state
                errStatus = msg[1]
                cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_SetErrorState, errStatus))
                traceUart('APP: req', cdiReq)
                fUART.write(cdiReq)

                # set DS_ERROR
                devStatus = DeviceState.DS_ERROR
                logging.info('APP: DET Status %d', devStatus)
                cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_DetStatus, devStatus))
                traceUart('APP: req', cdiReq)
                fUART.write(cdiReq)
            elif (msg[0] == CdiFrameId.CCR_Error):
                logging.error('DEV[E]: %s', msg[2:-1].decode('ascii'))

                if (devStatus not in [ DeviceState.DS_START_UP, DeviceState.DS_ERROR ]):
                    # set error state
                    errStatus = msg[1]
                    cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_SetErrorState, errStatus))
                    traceUart('APP: req', cdiReq)
                    fUART.write(cdiReq)

                    # set DS_ERROR
                    devStatus = DeviceState.DS_ERROR
                    logging.info('APP: DET Status %d', devStatus)
                    cdiReq = formatCdiMessage(struct.pack('bb', CdiFrameId.CCR_DetStatus, devStatus))
                    traceUart('APP: req', cdiReq)
                    fUART.write(cdiReq)

                    # must see BS_ERROR
                    needBsError = True
            elif (msg[0] == CdiFrameId.CCR_Warning):
                logging.warning('DEV[W]: %s', msg[2:-1].decode('ascii'))
            elif (msg[0] == CdiFrameId.CCR_Info):
                logging.info('DEV[I]: %s', msg[1:-1].decode('ascii'))
            elif (msg[0] == CdiFrameId.CCR_Debug):
                logging.info('DEV[D]: %s', msg[1:-1].decode('ascii'))
            elif (msg[0] == CdiFrameId.CCR_Trace):
                logging.info('DEV[T]: %s', msg[1:-1].decode('ascii'))
            elif (msg[0] == CdiFrameId.CCR_PrepareUpdate):
                logging.info('BSM: Prepare SW Update')
                logging.info('APP: Accept SW Update')
                cdiReq = formatCdiMessage(struct.pack('b', CdiFrameId.CCR_AcceptUpdate))
                traceUart('APP: req', cdiReq)
                fUART.write(cdiReq)
            elif (msg[0] == CdiFrameId.CCR_PerformUpdate):
                logging.info('BSM: Perform SW Update')

            # flush UART
            fUART.flush()

            quApp.task_done()
        except struct.error as err:
            logging.error('APP: error parsing msg: %s', err)
            quApp.task_done()
        except asyncio.TimeoutError:
            if (coreStatus != 0):
                logging.debug('APP: CDI2 Core status %d, keep polling', coreStatus)
                logging.info('APP: Protocol Version 2.0')
                cdiReq = formatCdiMessage(struct.pack('bbb', CdiFrameId.CCR_ProtocolVersion, 0, 2))
                traceUart('APP: req', cdiReq)
                fUART.write(cdiReq)
        except BaseException as ex:
            logging.error('APP: crashed with %s', repr(ex))
            sys.exit(-1)

def doRx(fUART, quRa):
    """
    read bytes from the CDI2 UART and pass the chunk to the reassembly task

    Parameters
    ----------
    fUART : io.BufferedIOBase
        file object of UART interface
    quRa: asyncio.Queue
        reassembly task queue
    """

    chunk = fUART.read()

    if (len(chunk) > 0):
        traceUart('RX: chunk', chunk)

        try:
            quRa.put_nowait(chunk)
            chunk = None
        except asyncio.QueueFull as err:
            logging.warning('RX: dropping %u bytes, reassembly queue full', len(chunk))

async def Reassembly(fUART, quRa, quApp):
    """
    CDI2 message reassembly task.

    Parameters
    ----------
    fUART : io.BufferedIOBase
        file object of UART interface
    quRa: asyncio.Queue
        reassembly task queue
    quApp: asyncio.Queue
        application task queue
    """

    logging.info('RA: CDI2 Message Reassembly task')

    # get event loop
    loop = asyncio.get_running_loop()

    # add RX callback
    loop.add_reader(fUART, doRx, fUART, quRa)

    # start unmarshaller scanning for STX
    umaStatus  = UmaStatus.UMA_IDLE
    cdiMessage = None

    while True:
        # get received chunk of bytes
        chunk = await quRa.get()

        if (len(chunk) <= 0):
            quRa.task_done()
            continue

        for c in chunk:
            if (umaStatus == UmaStatus.UMA_IDLE):
                if (c == CdiFrameChar.CDI_STX):
                    # STX scanned, goto in-message
                    umaStatus  = UmaStatus.UMA_STX
                    cdiMessage = bytearray(0)
                else:
                    # ignore any out-of-frame characters
                    pass
            elif (umaStatus == UmaStatus.UMA_STX):
                if (c == CdiFrameChar.CDI_STX):
                    # STX scanned, drop ill-terminated CDI frame
                    logging.warning('RA: dropping ill terminated CDI frame')
                    umaStatus  = UmaStatus.UMA_STX
                    cdiMessage = bytearray(0)
                elif (c == CdiFrameChar.CDI_ETX):
                    ## ETX scanned, close CDI frame and push to application layer

                    traceUart('RA: CDI frame', cdiMessage)

                    if (len(cdiMessage) > 2):
                        crcCheck = crc16.cdiCheck(cdiMessage)

                        if (crcCheck == 0):
                            try:
                                quApp.put_nowait(cdiMessage[0:-2])
                            except asyncio.QueueFull as err:
                                logging.warning('RA: dropping CDI frame ID 0x%02x with %u bytes, application queue full', cdiMessage[0], len(cdiMessage))
                        else:
                            logging.warning('RA: dropping CDI frame ID 0x%02x with CRC checksum error', cdiMessage[0])
                    else:
                        # minimum message length (w/o framing) is 3 (ID + CRC16), drop implausibly short frames
                        traceUart('RA: dropping short frame', cdiMessage)

                    # go back to scanning for STX
                    umaStatus  = UmaStatus.UMA_IDLE
                    cdiMessage = None
                elif (c == CdiFrameChar.CDI_ESC):
                    # ESC scanned, goto escape
                    umaStatus = UmaStatus.UMA_ESC
                else:
                    # push message byte
                    cdiMessage.append(c)
            elif (umaStatus == UmaStatus.UMA_ESC):
                # push escaped message byte
                cdiMessage.append(~c & 0xff)

                # go back to in-message
                umaStatus = UmaStatus.UMA_STX
            else:
                # start scanning for STX again
                logging.warning('RA: leaving implausible reassembly status %u', umaStatus)
                umaStatus  = UmaStatus.UMA_IDLE
                cdiMessage = None

        quRa.task_done()

def doSignal(sigName, fu):
    """
    handle INT/HUP/PIPE/TERM signals

    Parameters
    ----------
    sigName: string
        signal name
    fu: asyncio.Future
        future to satisfy
    """

    logging.debug('SIG: caught %s', sigName)

    if (not fu.done()):
        fu.set_result(1)

async def TOP(args):
    """
    top-level coroutine which starts the reassembly and appication simulator tasks
    in the event loop

    Parameters
    ----------
    args: argsparse.Namespace
        parsed command line arguments
    """

    # get event loop
    loop = asyncio.get_running_loop()

    logging.debug('TOP: create application and reassembly queues')
    quRa  = asyncio.Queue()
    quApp = asyncio.Queue()

    if "USB" in args.device:
         cfg='/sys/bus/usb-serial/devices/'+args.device.split('/')[-1]+'/latency_timer'
         logging.info('TOP: Setting latency_timer to 0 in file "{}"'.format(cfg))
         file = open(cfg, 'wb')
         file.write('0'.encode())
         file.close()

    logging.info('TOP: connect and configure UART interface {0}'.format(args.device))
    with open(args.device, 'rb+', buffering = 0) as fUART:
        configureUART(fUART)

        logging.debug('TOP: start application and reassembly tasks')
        taRa  = asyncio.create_task(Reassembly(fUART, quRa, quApp), name = 'RA')
        taApp = asyncio.create_task(App(fUART, quApp, args),        name = 'APP')

        # create 'termination' future
        fuTerm = loop.create_future()

        # register SIGINT, HUP, PIPE, and TERM signal handlers
        for signame in {'SIGINT', 'SIGHUP', 'SIGPIPE', 'SIGTERM'}:
            loop.add_signal_handler(getattr(signal, signame), doSignal, signame, fuTerm)

        # wait until signal handler indicates termination
        await fuTerm
        logging.info('TOP: stopping CDI2 application')
        taRa.cancel()
        taApp.cancel()

# idiomatic conditional main script stanza
if __name__ == "__main__":
    # setup command line parser and options
    parser = argparse.ArgumentParser(description='CDI2 Device test driver')
    parser.add_argument('-D', '--device',     action = 'store',      required = True,  help = 'UART interface to IP core')
    parser.add_argument('-n', '--nodeId',     action = 'store',      required = False, help = 'node ID', default = 1, choices = range(1, 17), type = int)
    parser.add_argument('-r', '--role',       action = 'store',      required = False, help = 'simulate Camera or generic Detector', default = 'Camera', choices = ['Camera', 'Detector'])
    parser.add_argument('-s', '--resultSize', action = 'store',      required = False, help = 'total BNRESULT size',                 default = 46, type = int)
    parser.add_argument('-t', '--noTTS',      action = 'store_true', required = False, help = 'no TTS interface',                    default = False)
    parser.add_argument('-b', '--bfa2',       action = 'store_true', required = False, help = 'use BFA#2',                           default = False)
    parser.add_argument('-l', '--log',        action = 'store',      required = False, help = 'log level',                           default = 'INFO')

    # parse command line
    args = parser.parse_args()

    # configure logging
    l = getattr(logging, args.log, None)

    if not isinstance(l, int):
        raise ValueError('invalid log level {0}'.format(args.log))

    logging.Formatter.converter = time.gmtime
    logging.basicConfig(format = '%(asctime)s %(levelname)s: %(message)s', level = l)

    if (args.resultSize < 15):
        raise ValueError('BNRESULT size must be >= 15 (header + judgment, result, and quality supplemental data)')

    if (args.resultSize > 1412):
        raise ValueError('BNRESULT size must be <= 1412 (max. segment size)')

    try:
        # start TOP coroutine
        asyncio.run(TOP(args))
    except BaseException as err:
        logging.error(err)
