
--------------------------------------------------------------------------------
-- date Oct 30, 2017
--------------------------------------------------------------------------------
-- author Ylber Hasani
--------------------------------------------------------------------------------
-- copyright Copyright (c) 2017, Austrian Institute of Technology (AIT)
--------------------------------------------------------------------------------



library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;


entity tts_deviceifc is
    port (

        clk_i                : in  std_logic;
        rst_n_i              : in  std_logic;

        tts_tc_i             : in  std_logic;
        tts_bp_i             : in  std_logic;
        tts_reset_i          : in  std_logic;
        tts_ready_o          : out std_logic;

        tts_bp_redge_o       : out std_logic;
        ttsready_device_i    : in  std_logic;

        ttstc_period_o       : out std_logic_vector(31 downto 0);
        new_ttstc_period_o   : out std_logic;

        ttsbp_period_o       : out std_logic_vector(31 downto 0);
        new_ttsbp_period_o   : out std_logic;

        ttsbp_duration_o     : out std_logic_vector(31 downto 0);
        new_ttsbp_duration_o : out std_logic;

        clear_ttsbp_numbers_i: in std_logic;
        ttsbp_numbers_o      : out std_logic_vector(31 downto 0);
        new_ttsbp_numbers_o  : out std_logic;

        ttsifc_status_o      : out std_logic_vector(31 downto 0)
        );
end entity tts_deviceifc;

architecture rtl of tts_deviceifc is

signal s_tc_r1          : std_logic;
signal s_tc_r2          : std_logic;
signal s_tc_r3          : std_logic;
signal s_tccnt_period   : unsigned(31 downto 0);

signal s_bp_r1          : std_logic;
signal s_bp_r2          : std_logic;
signal s_bp_r3          : std_logic;
signal s_bp_r4          : std_logic;

signal s_bpcnt_period   : unsigned(31 downto 0);
signal s_bpcnt_duration : unsigned(31 downto 0);
signal s_bpcnt_numbers  : unsigned(31 downto 0);

signal s_tts_reset_r1   : std_logic;
signal s_tts_reset_r2   : std_logic;

constant c_max_32Bit_counter : std_logic_vector(31 downto 0) := X"FFFFFFFF";

begin

  tts_ready_o <= ttsready_device_i;

  -- tc period
  p_ttstc_period: process (clk_i,rst_n_i) is
  begin
    if (rst_n_i = '0') then
        s_tc_r1 <= '0';
        s_tc_r2 <= '0';
        s_tc_r3 <= '0';
        s_tccnt_period <= (others => '0');
        ttstc_period_o <= (others => '0');
        new_ttstc_period_o <= '0';
    elsif (rising_edge(clk_i)) then

        -- sync tc_i
        s_tc_r1 <= tts_tc_i;
        s_tc_r2 <= s_tc_r1;
        s_tc_r3 <= s_tc_r2;

        -- tc rising edge
        if (s_tc_r2 = '1' and s_tc_r3 = '0') then
            s_tccnt_period <= conv_unsigned(0, s_tccnt_period'length);
        elsif (s_tccnt_period < unsigned(c_max_32Bit_counter)) then
            s_tccnt_period <= s_tccnt_period + conv_unsigned(1, 1);
        end if;

        new_ttstc_period_o <= '0';
        if (s_tc_r2 = '1' and s_tc_r3 = '0') then
            if (s_tccnt_period /= unsigned(c_max_32Bit_counter)) then
                ttstc_period_o <= std_logic_vector(s_tccnt_period);
                new_ttstc_period_o <= '1';
            end if;
        end if;

    end if;
  end process p_ttstc_period;

  -- bp period
  p_ttsbp_period: process (clk_i,rst_n_i) is
  begin
    if (rst_n_i = '0') then
        s_bp_r1 <= '0';
        s_bp_r2 <= '0';
        s_bp_r3 <= '0';
        s_bp_r4 <= '0';
        tts_bp_redge_o <= '0';
        s_bpcnt_period <= (others => '0');
        ttsbp_period_o <= (others => '0');
        new_ttsbp_period_o <= '0';
    elsif (rising_edge(clk_i)) then

        -- sync bp_i
        s_bp_r1 <= tts_bp_i;
        s_bp_r2 <= s_bp_r1;
        s_bp_r3 <= s_bp_r2;
        s_bp_r4 <= s_bp_r3;

        -- bp rising edge
        tts_bp_redge_o <= '0';
        if (s_bp_r2 = '1' and s_bp_r3 = '0') then
            tts_bp_redge_o <= '1';
        end if;

        -- at bp rising edge reset counter
        if (s_bp_r2 = '1' and s_bp_r3 = '0') then
            s_bpcnt_period <= conv_unsigned(0, s_bpcnt_period'length);
        elsif (s_bpcnt_period < unsigned(c_max_32Bit_counter)) then
            s_bpcnt_period <= s_bpcnt_period + conv_unsigned(1, 1);
        end if;

        -- at bp rising edge
        new_ttsbp_period_o <= '0';
        if (s_bp_r2 = '1' and s_bp_r3 = '0') then
            if (s_bpcnt_period /= unsigned(c_max_32Bit_counter)) then
                ttsbp_period_o <= std_logic_vector(s_bpcnt_period);
                new_ttsbp_period_o <= '1';
            end if;
        end if;

    end if;
  end process p_ttsbp_period;

  -- bp duration
  p_ttsbp_duration: process (clk_i,rst_n_i) is
  begin
    if (rst_n_i = '0') then
        s_bpcnt_duration <= (others => '0');
        new_ttsbp_duration_o <= '0';
    elsif (rising_edge(clk_i)) then

        -- at bp rising edge reset counter
        if (s_bp_r2 = '1' and s_bp_r3 = '0') then
            s_bpcnt_duration <= conv_unsigned(1, s_bpcnt_duration'length);
        elsif ((s_bp_r2 = '1') and (s_bpcnt_duration < unsigned(c_max_32Bit_counter))) then
            s_bpcnt_duration <= s_bpcnt_duration + conv_unsigned(1, 1);
        end if;

        -- at bp falling edge
        new_ttsbp_duration_o <= '0';
        if (s_bp_r2 = '0' and s_bp_r3 = '1') then
            if (s_bpcnt_duration /= unsigned(c_max_32Bit_counter)) then
                ttsbp_duration_o <= std_logic_vector(s_bpcnt_duration);
                new_ttsbp_duration_o <= '1';
            end if;
        end if;

    end if;
  end process p_ttsbp_duration;

  -- bp numbers
  p_ttsbp_numbers: process (clk_i,rst_n_i) is
  begin
    if (rst_n_i = '0') then
        s_bpcnt_numbers <= (others => '0');
        new_ttsbp_numbers_o <= '0';
        ttsbp_numbers_o <= (others => '0');
    elsif (rising_edge(clk_i)) then


        -- at bp rising edge count
        if (clear_ttsbp_numbers_i = '1') then
            s_bpcnt_numbers <= (others => '0');
        elsif (s_bp_r2 = '1' and s_bp_r3 = '0') then
            s_bpcnt_numbers <= s_bpcnt_numbers + conv_unsigned(1, 1);
        end if;

        new_ttsbp_numbers_o <= '0';
        if (s_bp_r3 = '1' and s_bp_r4 = '0') then
            ttsbp_numbers_o <= std_logic_vector(s_bpcnt_numbers);
            new_ttsbp_numbers_o <= '1';
        end if;

    end if;
  end process p_ttsbp_numbers;

  -- tts status
  p_tts_status: process (clk_i,rst_n_i) is
  begin
    if (rst_n_i = '0') then
        s_tts_reset_r1 <= '0';
        s_tts_reset_r2 <= '0';
        ttsifc_status_o <= (others => '0');
    elsif (rising_edge(clk_i)) then

        s_tts_reset_r1 <= tts_reset_i;
        s_tts_reset_r2 <= s_tts_reset_r1;
        ttsifc_status_o <= X"0000000" & '0' &  s_tts_reset_r2 & s_bp_r2 &  s_tc_r2;
    end if;
  end process p_tts_status;

end rtl;
